<?php
/**
 * Admin Panel
 * System administration interface
 */

require_once __DIR__ . '/../config/database.php';

class AdminAPI {
    private $db;
    private $admin_id;

    public function __construct($admin_id = null) {
        $this->db = new Database();
        $this->admin_id = $admin_id;
    }

    public function getDashboardStats() {
        try {
            $conn = $this->db->getConnection();
            
            // User statistics
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_users,
                    COUNT(CASE WHEN is_verified = 1 THEN 1 END) as verified_users,
                    COUNT(CASE WHEN kyc_status = 'approved' THEN 1 END) as kyc_approved,
                    COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as new_users_24h
                FROM users
            ");
            $stmt->execute();
            $user_stats = $stmt->fetch();

            // Transaction statistics
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_transactions,
                    SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END) as total_buy_volume,
                    SUM(CASE WHEN type = 'sell' THEN amount ELSE 0 END) as total_sell_volume,
                    SUM(fee) as total_fees_collected,
                    COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as transactions_24h
                FROM transactions
                WHERE status = 'confirmed'
            ");
            $stmt->execute();
            $transaction_stats = $stmt->fetch();

            // Wallet statistics
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_wallets,
                    SUM(balance) as total_crypto_balance,
                    COUNT(CASE WHEN balance > 0 THEN 1 END) as active_wallets
                FROM user_wallets
                WHERE is_active = 1
            ");
            $stmt->execute();
            $wallet_stats = $stmt->fetch();

            // Trading pair statistics
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_pairs,
                    SUM(volume_24h) as total_volume_24h,
                    AVG(change_24h) as avg_change_24h
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                WHERE tp.is_active = 1
            ");
            $stmt->execute();
            $market_stats = $stmt->fetch();

            return [
                'success' => true,
                'user_stats' => $user_stats,
                'transaction_stats' => $transaction_stats,
                'wallet_stats' => $wallet_stats,
                'market_stats' => $market_stats
            ];
        } catch (Exception $e) {
            error_log("Get dashboard stats error: " . $e->getMessage());
            return ['error' => 'Failed to get dashboard statistics'];
        }
    }

    public function getUsers($limit = 50, $offset = 0, $filters = []) {
        try {
            $conn = $this->db->getConnection();
            
            $where_conditions = [];
            $params = [];

            if (!empty($filters['search'])) {
                $where_conditions[] = "(u.email LIKE ? OR u.first_name LIKE ? OR u.last_name LIKE ?)";
                $search_term = '%' . $filters['search'] . '%';
                $params = array_merge($params, [$search_term, $search_term, $search_term]);
            }

            if (!empty($filters['country'])) {
                $where_conditions[] = "u.country = ?";
                $params[] = $filters['country'];
            }

            if (!empty($filters['kyc_status'])) {
                $where_conditions[] = "u.kyc_status = ?";
                $params[] = $filters['kyc_status'];
            }

            if (!empty($filters['is_active'])) {
                $where_conditions[] = "u.is_active = ?";
                $params[] = $filters['is_active'];
            }

            $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
            $params = array_merge($params, [$limit, $offset]);

            $stmt = $conn->prepare("
                SELECT u.*, p.date_of_birth, p.address, p.city, p.state,
                       (SELECT SUM(balance) FROM user_wallets WHERE user_id = u.id AND is_active = 1) as crypto_balance,
                       (SELECT SUM(balance) FROM user_fiat_balances WHERE user_id = u.id) as fiat_balance
                FROM users u
                LEFT JOIN user_profiles p ON u.id = p.user_id
                $where_clause
                ORDER BY u.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute($params);
            $users = $stmt->fetchAll();

            // Get total count for pagination
            $count_params = array_slice($params, 0, -2);
            $stmt = $conn->prepare("
                SELECT COUNT(*) as total
                FROM users u
                LEFT JOIN user_profiles p ON u.id = p.user_id
                $where_clause
            ");
            $stmt->execute($count_params);
            $total = $stmt->fetch()['total'];

            return [
                'success' => true,
                'users' => $users,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
        } catch (Exception $e) {
            error_log("Get users error: " . $e->getMessage());
            return ['error' => 'Failed to get users'];
        }
    }

    public function updateUserStatus($user_id, $status, $value) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();

            $allowed_statuses = ['is_active', 'is_verified', 'kyc_status'];
            if (!in_array($status, $allowed_statuses)) {
                return ['error' => 'Invalid status field'];
            }

            $stmt = $conn->prepare("UPDATE users SET $status = ? WHERE id = ?");
            $stmt->execute([$value, $user_id]);

            if ($stmt->rowCount() === 0) {
                return ['error' => 'User not found'];
            }

            // Log the action
            $this->logAdminAction('update_user_status', 'users', $user_id, null, [$status => $value]);

            $conn->commit();
            return ['success' => true, 'message' => 'User status updated successfully'];

        } catch (Exception $e) {
            $conn->rollback();
            error_log("Update user status error: " . $e->getMessage());
            return ['error' => 'Failed to update user status'];
        }
    }

    public function getTransactions($limit = 50, $offset = 0, $filters = []) {
        try {
            $conn = $this->db->getConnection();
            
            $where_conditions = [];
            $params = [];

            if (!empty($filters['type'])) {
                $where_conditions[] = "t.type = ?";
                $params[] = $filters['type'];
            }

            if (!empty($filters['status'])) {
                $where_conditions[] = "t.status = ?";
                $params[] = $filters['status'];
            }

            if (!empty($filters['user_id'])) {
                $where_conditions[] = "t.user_id = ?";
                $params[] = $filters['user_id'];
            }

            if (!empty($filters['date_from'])) {
                $where_conditions[] = "t.created_at >= ?";
                $params[] = $filters['date_from'];
            }

            if (!empty($filters['date_to'])) {
                $where_conditions[] = "t.created_at <= ?";
                $params[] = $filters['date_to'];
            }

            $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
            $params = array_merge($params, [$limit, $offset]);

            $stmt = $conn->prepare("
                SELECT t.*, u.email, u.first_name, u.last_name, c.symbol as crypto_symbol, n.name as network_name
                FROM transactions t
                JOIN users u ON t.user_id = u.id
                LEFT JOIN cryptocurrencies c ON t.crypto_id = c.id
                LEFT JOIN networks n ON t.network_id = n.id
                $where_clause
                ORDER BY t.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute($params);
            $transactions = $stmt->fetchAll();

            // Get total count
            $count_params = array_slice($params, 0, -2);
            $stmt = $conn->prepare("
                SELECT COUNT(*) as total
                FROM transactions t
                JOIN users u ON t.user_id = u.id
                LEFT JOIN cryptocurrencies c ON t.crypto_id = c.id
                LEFT JOIN networks n ON t.network_id = n.id
                $where_clause
            ");
            $stmt->execute($count_params);
            $total = $stmt->fetch()['total'];

            return [
                'success' => true,
                'transactions' => $transactions,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
        } catch (Exception $e) {
            error_log("Get transactions error: " . $e->getMessage());
            return ['error' => 'Failed to get transactions'];
        }
    }

    public function getSystemSettings() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("SELECT * FROM system_settings ORDER BY setting_key");
            $stmt->execute();
            $settings = $stmt->fetchAll();

            $formatted_settings = [];
            foreach ($settings as $setting) {
                $formatted_settings[$setting['setting_key']] = $setting['setting_value'];
            }

            return [
                'success' => true,
                'settings' => $formatted_settings
            ];
        } catch (Exception $e) {
            error_log("Get system settings error: " . $e->getMessage());
            return ['error' => 'Failed to get system settings'];
        }
    }

    public function updateSystemSettings($settings) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();

            foreach ($settings as $key => $value) {
                $stmt = $conn->prepare("
                    INSERT INTO system_settings (setting_key, setting_value)
                    VALUES (?, ?)
                    ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
                ");
                $stmt->execute([$key, $value]);
            }

            // Log the action
            $this->logAdminAction('update_system_settings', 'system_settings', null, null, $settings);

            $conn->commit();
            return ['success' => true, 'message' => 'System settings updated successfully'];

        } catch (Exception $e) {
            $conn->rollback();
            error_log("Update system settings error: " . $e->getMessage());
            return ['error' => 'Failed to update system settings'];
        }
    }

    public function getAuditLogs($limit = 50, $offset = 0, $filters = []) {
        try {
            $conn = $this->db->getConnection();
            
            $where_conditions = [];
            $params = [];

            if (!empty($filters['user_id'])) {
                $where_conditions[] = "al.user_id = ?";
                $params[] = $filters['user_id'];
            }

            if (!empty($filters['action'])) {
                $where_conditions[] = "al.action = ?";
                $params[] = $filters['action'];
            }

            if (!empty($filters['date_from'])) {
                $where_conditions[] = "al.created_at >= ?";
                $params[] = $filters['date_from'];
            }

            if (!empty($filters['date_to'])) {
                $where_conditions[] = "al.created_at <= ?";
                $params[] = $filters['date_to'];
            }

            $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
            $params = array_merge($params, [$limit, $offset]);

            $stmt = $conn->prepare("
                SELECT al.*, u.email, u.first_name, u.last_name
                FROM audit_logs al
                LEFT JOIN users u ON al.user_id = u.id
                $where_clause
                ORDER BY al.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute($params);
            $logs = $stmt->fetchAll();

            // Get total count
            $count_params = array_slice($params, 0, -2);
            $stmt = $conn->prepare("
                SELECT COUNT(*) as total
                FROM audit_logs al
                LEFT JOIN users u ON al.user_id = u.id
                $where_clause
            ");
            $stmt->execute($count_params);
            $total = $stmt->fetch()['total'];

            return [
                'success' => true,
                'logs' => $logs,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
        } catch (Exception $e) {
            error_log("Get audit logs error: " . $e->getMessage());
            return ['error' => 'Failed to get audit logs'];
        }
    }

    private function logAdminAction($action, $table_name, $record_id, $old_values = null, $new_values = null) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $this->admin_id,
                $action,
                $table_name,
                $record_id,
                $old_values ? json_encode($old_values) : null,
                $new_values ? json_encode($new_values) : null,
                $_SERVER['REMOTE_ADDR'] ?? '',
                $_SERVER['HTTP_USER_AGENT'] ?? ''
            ]);
        } catch (Exception $e) {
            error_log("Admin audit log error: " . $e->getMessage());
        }
    }
}
?>
