<?php
/**
 * Admin API
 * Handles all admin dashboard functionality
 */

require_once __DIR__ . '/../config/database.php';

class AdminAPI {
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
    }
    
    /**
     * Get dashboard statistics
     */
    public function getDashboardStats() {
        try {
            // Get user statistics
            $userStats = $this->getUserStats();
            
            // Get transaction statistics
            $transactionStats = $this->getTransactionStats();
            
            // Get wallet statistics
            $walletStats = $this->getWalletStats();
            
            return [
                'success' => true,
                'user_stats' => $userStats,
                'transaction_stats' => $transactionStats,
                'wallet_stats' => $walletStats
            ];
            
        } catch (Exception $e) {
            error_log("Admin dashboard error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load dashboard data'
            ];
        }
    }
    
    /**
     * Get user statistics
     */
    private function getUserStats() {
        $stmt = $this->conn->prepare("
            SELECT 
                COUNT(*) as total_users,
                COUNT(CASE WHEN is_active = 1 THEN 1 END) as active_users,
                COUNT(CASE WHEN is_verified = 1 THEN 1 END) as verified_users
            FROM users
        ");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get transaction statistics
     */
    private function getTransactionStats() {
        $stmt = $this->conn->prepare("
            SELECT 
                COUNT(*) as total_transactions,
                SUM(CASE WHEN type = 'buy' THEN amount ELSE 0 END) as total_buy_volume,
                SUM(CASE WHEN type = 'sell' THEN amount ELSE 0 END) as total_sell_volume,
                SUM(CASE WHEN type = 'send' THEN amount ELSE 0 END) as total_send_volume,
                SUM(CASE WHEN type = 'convert' THEN amount ELSE 0 END) as total_convert_volume,
                SUM(COALESCE(fee, 0)) as total_fees_collected,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_transactions,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_transactions,
                COUNT(CASE WHEN status = 'failed' THEN 1 END) as failed_transactions
            FROM transactions
        ");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get wallet statistics
     */
    private function getWalletStats() {
        $stmt = $this->conn->prepare("
            SELECT 
                COUNT(*) as total_wallets,
                SUM(balance) as total_crypto_balance,
                COUNT(DISTINCT user_id) as users_with_wallets
            FROM crypto_wallets
        ");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get all users with pagination
     */
    public function getUsers($limit = 50, $offset = 0, $filters = []) {
        try {
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['search'])) {
                $whereClause .= " AND (first_name LIKE ? OR last_name LIKE ? OR email LIKE ?)";
                $searchTerm = "%{$filters['search']}%";
                $params[] = $searchTerm;
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            if (!empty($filters['verification_status'])) {
                $whereClause .= " AND is_verified = ?";
                $params[] = $filters['verification_status'];
            }
            
            if (!empty($filters['is_active'])) {
                $whereClause .= " AND is_active = ?";
                $params[] = $filters['is_active'];
            }
            
            // Get total count
            $countStmt = $this->conn->prepare("SELECT COUNT(*) as total FROM users $whereClause");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
            
            // Get users
            $stmt = $this->conn->prepare("
                SELECT 
                    u.*,
                    COALESCE(SUM(cw.balance), 0) as crypto_balance,
                    COALESCE(SUM(ufb.balance), 0) as fiat_balance
                FROM users u
                LEFT JOIN crypto_wallets cw ON u.id = cw.user_id
                LEFT JOIN user_fiat_balances ufb ON u.id = ufb.user_id
                $whereClause
                GROUP BY u.id
                ORDER BY u.created_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'users' => $users,
                'total' => $total
            ];
            
        } catch (Exception $e) {
            error_log("Admin users error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load users'
            ];
        }
    }
    
    /**
     * Get all transactions with pagination
     */
    public function getTransactions($limit = 50, $offset = 0, $filters = []) {
        try {
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['type'])) {
                $whereClause .= " AND type = ?";
                $params[] = $filters['type'];
            }
            
            if (!empty($filters['status'])) {
                $whereClause .= " AND status = ?";
                $params[] = $filters['status'];
            }
            
            // Get total count
            $countStmt = $this->conn->prepare("SELECT COUNT(*) as total FROM transactions $whereClause");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
            
            // Get transactions
            $stmt = $this->conn->prepare("
                SELECT 
                    t.*,
                    u.first_name,
                    u.last_name,
                    u.email
                FROM transactions t
                LEFT JOIN users u ON t.user_id = u.id
                $whereClause
                ORDER BY t.created_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'transactions' => $transactions,
                'total' => $total
            ];
            
        } catch (Exception $e) {
            error_log("Admin transactions error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load transactions'
            ];
        }
    }
    
    /**
     * Get all crypto wallets
     */
    public function getCryptoWallets($filters = []) {
        try {
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['network'])) {
                $whereClause .= " AND cw.network = ?";
                $params[] = $filters['network'];
            }
            
            if (!empty($filters['currency'])) {
                $whereClause .= " AND cw.currency = ?";
                $params[] = $filters['currency'];
            }

            // Get total count
            $countStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM crypto_wallets cw
                LEFT JOIN users u ON cw.user_id = u.id
                $whereClause
            ");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];

            // Get pagination parameters
            $limit = intval($filters['limit'] ?? 50);
            $offset = intval($filters['offset'] ?? 0);
            
            $stmt = $this->conn->prepare("
                SELECT 
                    cw.*,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone
                FROM crypto_wallets cw
                LEFT JOIN users u ON cw.user_id = u.id
                $whereClause
                ORDER BY cw.created_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'wallets' => $wallets,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
            
        } catch (Exception $e) {
            error_log("Admin wallets error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load wallets'
            ];
        }
    }
    
    /**
     * Get all payments
     */
    public function getPayments($filters = []) {
        try {
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['status'])) {
                $whereClause .= " AND p.status = ?";
                $params[] = $filters['status'];
            }

            // Get total count
            $countStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM payments p
                LEFT JOIN users u ON p.user_id = u.id
                $whereClause
            ");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];

            // Get pagination parameters
            $limit = intval($filters['limit'] ?? 50);
            $offset = intval($filters['offset'] ?? 0);
            
            $stmt = $this->conn->prepare("
                SELECT 
                    p.*,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone
                FROM payments p
                LEFT JOIN users u ON p.user_id = u.id
                $whereClause
                ORDER BY p.created_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'payments' => $payments,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
            
        } catch (Exception $e) {
            error_log("Admin payments error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load payments'
            ];
        }
    }
    
    
    
    
    /**
     * Get blockchain status
     */
    public function getBlockchainStatus() {
        try {
            // Get recent crypto transactions
            $stmt = $this->conn->prepare("
                SELECT * FROM crypto_transactions 
                ORDER BY created_at DESC 
                LIMIT 10
            ");
            $stmt->execute();
            $recentTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'status' => 'Active',
                'last_check' => date('Y-m-d H:i:s'),
                'transactions_found' => count($recentTransactions),
                'recent_transactions' => $recentTransactions
            ];
            
        } catch (Exception $e) {
            error_log("Admin blockchain status error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load blockchain status'
            ];
        }
    }
    
    /**
     * Get system statistics
     */
    public function getSystemStats() {
        try {
            // Get database size
            $dbSize = $this->getDatabaseSize();
            
            // Get active users (last 24 hours)
            $stmt = $this->conn->prepare("
                SELECT COUNT(*) as active_users 
                FROM users 
                WHERE last_login_at > datetime('now', '-1 day')
            ");
            $stmt->execute();
            $activeUsers = $stmt->fetch(PDO::FETCH_ASSOC)['active_users'];
            
            // Get pending transactions
            $stmt = $this->conn->prepare("
                SELECT COUNT(*) as pending_transactions 
                FROM transactions 
                WHERE status = 'pending'
            ");
            $stmt->execute();
            $pendingTransactions = $stmt->fetch(PDO::FETCH_ASSOC)['pending_transactions'];
            
            return [
                'success' => true,
                'db_size' => $dbSize,
                'active_users' => $activeUsers,
                'pending_transactions' => $pendingTransactions,
                'uptime' => $this->getSystemUptime()
            ];
            
        } catch (Exception $e) {
            error_log("Admin system stats error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to load system statistics'
            ];
        }
    }
    
    /**
     * Get database size
     */
    private function getDatabaseSize() {
        $dbFile = __DIR__ . '/../database/crypto_exchange.db';
        if (file_exists($dbFile)) {
            $size = filesize($dbFile);
            return $this->formatBytes($size);
        }
        return 'Unknown';
    }
    
    /**
     * Format bytes to human readable
     */
    private function formatBytes($size, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
            $size /= 1024;
        }
        return round($size, $precision) . ' ' . $units[$i];
    }
    
    /**
     * Get system uptime
     */
    private function getSystemUptime() {
        $uptime = time() - filemtime(__DIR__ . '/../database/crypto_exchange.db');
        return $this->formatUptime($uptime);
    }
    
    /**
     * Format uptime to human readable
     */
    private function formatUptime($seconds) {
        $days = floor($seconds / 86400);
        $hours = floor(($seconds % 86400) / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        
        if ($days > 0) {
            return "{$days}d {$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h {$minutes}m";
        } else {
            return "{$minutes}m";
        }
    }
    
    /**
     * Check all payments
     */
    public function checkAllPayments() {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM payments 
                WHERE status = 'pending' 
                ORDER BY created_at DESC
            ");
            $stmt->execute();
            $pendingPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $checkedCount = count($pendingPayments);
            $updatedCount = 0;
            
            foreach ($pendingPayments as $payment) {
                // Here you would check with Bitmuk API
                // For now, we'll simulate some updates
                if (rand(1, 5) === 1) { // 20% chance of status change
                    $newStatus = rand(1, 2) === 1 ? 'completed' : 'failed';
                    
                    $updateStmt = $this->conn->prepare("
                        UPDATE payments 
                        SET status = ?, updated_at = datetime('now')
                        WHERE id = ?
                    ");
                    $updateStmt->execute([$newStatus, $payment['id']]);
                    $updatedCount++;
                }
            }
            
            return [
                'success' => true,
                'checked_count' => $checkedCount,
                'updated_count' => $updatedCount
            ];
            
        } catch (Exception $e) {
            error_log("Admin check payments error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to check payments'
            ];
        }
    }

    /**
     * Update user status (active, verified, etc.)
     */
    public function updateUserStatus($userId, $status, $value) {
        try {
            // Validate status field
            $allowedStatuses = ['is_active', 'is_verified', 'email_verified', 'phone_verified'];
            if (!in_array($status, $allowedStatuses)) {
                return [
                    'success' => false,
                    'error' => 'Invalid status field. Allowed: ' . implode(', ', $allowedStatuses)
                ];
            }

            // Validate value (should be 0 or 1)
            if (!in_array($value, [0, 1, true, false])) {
                return [
                    'success' => false,
                    'error' => 'Invalid value. Must be 0 or 1'
                ];
            }

            // Convert boolean to int
            $intValue = $value ? 1 : 0;

            // Update user status
            $stmt = $this->conn->prepare("UPDATE users SET $status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
            $stmt->execute([$intValue, $userId]);

            if ($stmt->rowCount() > 0) {
                // Log the status change
                $statusText = $intValue ? 'enabled' : 'disabled';
                error_log("Admin updated user $userId: $status set to $statusText");

                return [
                    'success' => true,
                    'message' => "User $status updated successfully",
                    'user_id' => $userId,
                    'status' => $status,
                    'value' => $intValue
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'User not found or no changes made'
                ];
            }

        } catch (Exception $e) {
            error_log("Update user status error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to update user status'
            ];
        }
    }

    /**
     * Delete user permanently
     */
    public function deleteUser($userId) {
        try {
            $this->conn->beginTransaction();

            // Delete user's crypto wallets
            $stmt = $this->conn->prepare("DELETE FROM crypto_wallets WHERE user_id = ?");
            $stmt->execute([$userId]);

            // Delete user's fiat balances
            $stmt = $this->conn->prepare("DELETE FROM user_fiat_balances WHERE user_id = ?");
            $stmt->execute([$userId]);

            // Delete user's transactions
            $stmt = $this->conn->prepare("DELETE FROM transactions WHERE user_id = ?");
            $stmt->execute([$userId]);

            // Delete user's payments
            $stmt = $this->conn->prepare("DELETE FROM payments WHERE user_id = ?");
            $stmt->execute([$userId]);

            // Finally delete the user
            $stmt = $this->conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->execute([$userId]);

            if ($stmt->rowCount() > 0) {
                $this->conn->commit();
                error_log("Admin permanently deleted user: $userId");
                return [
                    'success' => true,
                    'message' => 'User deleted successfully',
                    'user_id' => $userId
                ];
            } else {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'User not found'
                ];
            }

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Delete user error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to delete user'
            ];
        }
    }

    /**
     * Ban user (set is_active to 0 and add ban reason)
     */
    public function banUser($userId, $reason) {
        try {
            // Update user status to inactive
            $stmt = $this->conn->prepare("UPDATE users SET is_active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
            $stmt->execute([$userId]);

            if ($stmt->rowCount() > 0) {
                // Log the ban action (you might want to create a user_actions table for this)
                error_log("Admin banned user $userId. Reason: $reason");
                
                return [
                    'success' => true,
                    'message' => 'User banned successfully',
                    'user_id' => $userId,
                    'reason' => $reason
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'User not found'
                ];
            }

        } catch (Exception $e) {
            error_log("Ban user error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to ban user'
            ];
        }
    }

    /**
     * Reset user password
     */
    public function resetUserPassword($userId, $newPassword = null) {
        try {
            // Generate a random password if none provided
            if (!$newPassword) {
                $newPassword = bin2hex(random_bytes(8)); // 16 character random password
            }

            // Hash the new password
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);

            // Update user password
            $stmt = $this->conn->prepare("UPDATE users SET password_hash = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
            $stmt->execute([$hashedPassword, $userId]);

            if ($stmt->rowCount() > 0) {
                error_log("Admin reset password for user: $userId");
                
                return [
                    'success' => true,
                    'message' => 'Password reset successfully',
                    'user_id' => $userId,
                    'new_password' => $newPassword // Return plain password for admin to share with user
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'User not found'
                ];
            }

        } catch (Exception $e) {
            error_log("Reset password error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to reset password'
            ];
        }
    }

    /**
     * Get detailed user information
     */
    public function getUserDetails($userId) {
        try {
            // Get user basic info
            $stmt = $this->conn->prepare("
                SELECT 
                    u.*,
                    COALESCE(SUM(cw.balance), 0) as total_crypto_balance,
                    COALESCE(SUM(ufb.balance), 0) as total_fiat_balance
                FROM users u
                LEFT JOIN crypto_wallets cw ON u.id = cw.user_id
                LEFT JOIN user_fiat_balances ufb ON u.id = ufb.user_id
                WHERE u.id = ?
                GROUP BY u.id
            ");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                return [
                    'success' => false,
                    'error' => 'User not found'
                ];
            }

            // Get user's crypto wallets
            $stmt = $this->conn->prepare("
                SELECT * FROM crypto_wallets WHERE user_id = ? ORDER BY created_at DESC
            ");
            $stmt->execute([$userId]);
            $cryptoWallets = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get user's fiat balances
            $stmt = $this->conn->prepare("
                SELECT * FROM user_fiat_balances WHERE user_id = ?
            ");
            $stmt->execute([$userId]);
            $fiatBalances = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get recent transactions
            $stmt = $this->conn->prepare("
                SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 10
            ");
            $stmt->execute([$userId]);
            $recentTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get recent payments
            $stmt = $this->conn->prepare("
                SELECT * FROM payments WHERE user_id = ? ORDER BY created_at DESC LIMIT 10
            ");
            $stmt->execute([$userId]);
            $recentPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'user' => $user,
                'crypto_wallets' => $cryptoWallets,
                'fiat_balances' => $fiatBalances,
                'recent_transactions' => $recentTransactions,
                'recent_payments' => $recentPayments,
                'total_wallets' => count($cryptoWallets),
                'total_balances' => count($fiatBalances)
            ];

        } catch (Exception $e) {
            error_log("Get user details error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get user details'
            ];
        }
    }

    /**
     * Update user profile
     */
    public function updateUser($userId, $updateData) {
        try {
            // Remove user_id from update data to prevent conflicts
            unset($updateData['user_id']);

            // Allowed fields for update
            $allowedFields = ['first_name', 'last_name', 'email', 'phone', 'country_code', 'is_verified', 'is_active', 'email_verified', 'phone_verified'];

            $updateFields = [];
            $values = [];

            foreach ($updateData as $field => $value) {
                if (in_array($field, $allowedFields)) {
                    $updateFields[] = "$field = ?";
                    $values[] = $value;
                }
            }

            if (empty($updateFields)) {
                return [
                    'success' => false,
                    'error' => 'No valid fields to update'
                ];
            }

            // Add updated_at and user_id
            $updateFields[] = "updated_at = CURRENT_TIMESTAMP";
            $values[] = $userId;

            $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = ?";
            $stmt = $this->conn->prepare($sql);
            $stmt->execute($values);

            if ($stmt->rowCount() > 0) {
                error_log("Admin updated user profile: $userId");
                return [
                    'success' => true,
                    'message' => 'User updated successfully',
                    'user_id' => $userId,
                    'updated_fields' => array_keys($updateData)
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'User not found or no changes made'
                ];
            }

        } catch (Exception $e) {
            error_log("Update user error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to update user'
            ];
        }
    }

    /**
     * Get detailed transaction information
     */
    public function getTransactionDetails($transactionId) {
        try {
            // Get transaction with user info
            $stmt = $this->conn->prepare("
                SELECT 
                    t.*,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone
                FROM transactions t
                LEFT JOIN users u ON t.user_id = u.id
                WHERE t.id = ?
            ");
            $stmt->execute([$transactionId]);
            $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$transaction) {
                return [
                    'success' => false,
                    'error' => 'Transaction not found'
                ];
            }

            return [
                'success' => true,
                'transaction' => $transaction
            ];

        } catch (Exception $e) {
            error_log("Get transaction details error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get transaction details'
            ];
        }
    }

    /**
     * Confirm a transaction
     */
    public function confirmTransaction($transactionId) {
        try {
            $this->conn->beginTransaction();

            // Get transaction details
            $stmt = $this->conn->prepare("SELECT * FROM transactions WHERE id = ? AND status = 'pending'");
            $stmt->execute([$transactionId]);
            $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$transaction) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Transaction not found or already processed'
                ];
            }

            // Update transaction status
            $stmt = $this->conn->prepare("
                UPDATE transactions 
                SET status = 'confirmed', updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$transactionId]);

            // Process the transaction based on type
            $result = $this->processConfirmedTransaction($transaction);
            
            if ($result['success']) {
                $this->conn->commit();
                error_log("Admin confirmed transaction: $transactionId");
                return [
                    'success' => true,
                    'message' => 'Transaction confirmed successfully',
                    'transaction_id' => $transactionId,
                    'processed' => $result['processed']
                ];
            } else {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => $result['error']
                ];
            }

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Confirm transaction error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to confirm transaction'
            ];
        }
    }

    /**
     * Reject a transaction
     */
    public function rejectTransaction($transactionId, $reason = 'Admin rejection') {
        try {
            // Update transaction status
            $stmt = $this->conn->prepare("
                UPDATE transactions 
                SET status = 'failed', 
                    description = CONCAT(COALESCE(description, ''), ' - Rejected: ', ?),
                    updated_at = CURRENT_TIMESTAMP 
                WHERE id = ? AND status = 'pending'
            ");
            $stmt->execute([$reason, $transactionId]);

            if ($stmt->rowCount() > 0) {
                error_log("Admin rejected transaction: $transactionId. Reason: $reason");
                return [
                    'success' => true,
                    'message' => 'Transaction rejected successfully',
                    'transaction_id' => $transactionId,
                    'reason' => $reason
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'Transaction not found or already processed'
                ];
            }

        } catch (Exception $e) {
            error_log("Reject transaction error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to reject transaction'
            ];
        }
    }

    /**
     * Refund a transaction
     */
    public function refundTransaction($transactionId) {
        try {
            $this->conn->beginTransaction();

            // Get transaction details
            $stmt = $this->conn->prepare("SELECT * FROM transactions WHERE id = ? AND status = 'confirmed'");
            $stmt->execute([$transactionId]);
            $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$transaction) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Transaction not found or not confirmed'
                ];
            }

            // Process refund based on transaction type
            $result = $this->processRefund($transaction);
            
            if ($result['success']) {
                // Update transaction status
                $stmt = $this->conn->prepare("
                    UPDATE transactions 
                    SET status = 'refunded', 
                        description = CONCAT(COALESCE(description, ''), ' - Refunded by admin'),
                        updated_at = CURRENT_TIMESTAMP 
                    WHERE id = ?
                ");
                $stmt->execute([$transactionId]);

                $this->conn->commit();
                error_log("Admin refunded transaction: $transactionId");
                return [
                    'success' => true,
                    'message' => 'Transaction refunded successfully',
                    'transaction_id' => $transactionId,
                    'refund_amount' => $result['refund_amount']
                ];
            } else {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => $result['error']
                ];
            }

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Refund transaction error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to refund transaction'
            ];
        }
    }

    /**
     * Process confirmed transaction based on type
     */
    private function processConfirmedTransaction($transaction) {
        try {
            switch ($transaction['type']) {
                case 'buy':
                    return $this->processBuyConfirmation($transaction);
                case 'sell':
                    return $this->processSellConfirmation($transaction);
                case 'send':
                    return $this->processSendConfirmation($transaction);
                case 'deposit':
                    return $this->processDepositConfirmation($transaction);
                case 'withdrawal':
                    return $this->processWithdrawalConfirmation($transaction);
                case 'convert':
                    return $this->processConvertConfirmation($transaction);
                default:
                    return ['success' => true, 'processed' => false];
            }
        } catch (Exception $e) {
            error_log("Process confirmed transaction error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to process transaction'];
        }
    }

    /**
     * Process refund based on transaction type
     */
    private function processRefund($transaction) {
        try {
            switch ($transaction['type']) {
                case 'buy':
                    return $this->processBuyRefund($transaction);
                case 'sell':
                    return $this->processSellRefund($transaction);
                case 'deposit':
                    return $this->processDepositRefund($transaction);
                default:
                    return ['success' => false, 'error' => 'Refund not supported for this transaction type'];
            }
        } catch (Exception $e) {
            error_log("Process refund error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to process refund'];
        }
    }

    /**
     * Process buy transaction confirmation
     */
    private function processBuyConfirmation($transaction) {
        // For buy transactions, credit the crypto wallet
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO crypto_wallets (user_id, crypto, network, balance, wallet_address)
            VALUES (?, ?, ?, 
                COALESCE((SELECT balance FROM crypto_wallets WHERE user_id = ? AND crypto = ? AND network = ?), 0) + ?,
                COALESCE((SELECT wallet_address FROM crypto_wallets WHERE user_id = ? AND crypto = ? AND network = ?), 'admin-generated')
            )
        ");
        
        $stmt->execute([
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default',
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default',
            $transaction['amount'],
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default'
        ]);

        return ['success' => true, 'processed' => true];
    }

    /**
     * Process sell transaction confirmation
     */
    private function processSellConfirmation($transaction) {
        // For sell transactions, credit the fiat wallet
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO user_fiat_balances (user_id, currency, balance)
            VALUES (?, ?, 
                COALESCE((SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency = ?), 0) + ?
            )
        ");
        
        $stmt->execute([
            $transaction['user_id'], $transaction['currency'],
            $transaction['user_id'], $transaction['currency'],
            $transaction['amount']
        ]);

        return ['success' => true, 'processed' => true];
    }

    /**
     * Process send transaction confirmation
     */
    private function processSendConfirmation($transaction) {
        // Send transactions are usually already processed when initiated
        return ['success' => true, 'processed' => false];
    }

    /**
     * Process deposit transaction confirmation
     */
    private function processDepositConfirmation($transaction) {
        // Credit fiat balance
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO user_fiat_balances (user_id, currency, balance)
            VALUES (?, ?, 
                COALESCE((SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency = ?), 0) + ?
            )
        ");
        
        $stmt->execute([
            $transaction['user_id'], $transaction['currency'],
            $transaction['user_id'], $transaction['currency'],
            $transaction['amount']
        ]);

        return ['success' => true, 'processed' => true];
    }

    /**
     * Process withdrawal transaction confirmation
     */
    private function processWithdrawalConfirmation($transaction) {
        // Withdrawal confirmations are usually handled by the payment processor
        return ['success' => true, 'processed' => false];
    }

    /**
     * Process convert transaction confirmation
     */
    private function processConvertConfirmation($transaction) {
        // Convert transactions involve debiting one currency and crediting another
        // This would need to be implemented based on your conversion logic
        return ['success' => true, 'processed' => false];
    }

    /**
     * Process buy transaction refund
     */
    private function processBuyRefund($transaction) {
        // Refund the fiat amount that was paid
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO user_fiat_balances (user_id, currency, balance)
            VALUES (?, ?, 
                COALESCE((SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency = ?), 0) + ?
            )
        ");
        
        // Assuming the original currency was fiat (like UGX, KES, etc.)
        $originalCurrency = $transaction['from_currency'] ?? $transaction['currency'];
        $refundAmount = $transaction['amount'] + ($transaction['fee'] ?? 0);
        
        $stmt->execute([
            $transaction['user_id'], $originalCurrency,
            $transaction['user_id'], $originalCurrency,
            $refundAmount
        ]);

        return ['success' => true, 'refund_amount' => $refundAmount];
    }

    /**
     * Process sell transaction refund
     */
    private function processSellRefund($transaction) {
        // Refund the crypto that was sold
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO crypto_wallets (user_id, crypto, network, balance, wallet_address)
            VALUES (?, ?, ?, 
                COALESCE((SELECT balance FROM crypto_wallets WHERE user_id = ? AND crypto = ? AND network = ?), 0) + ?,
                COALESCE((SELECT wallet_address FROM crypto_wallets WHERE user_id = ? AND crypto = ? AND network = ?), 'admin-generated')
            )
        ");
        
        $stmt->execute([
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default',
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default',
            $transaction['amount'],
            $transaction['user_id'], $transaction['currency'], $transaction['network'] ?? 'default'
        ]);

        return ['success' => true, 'refund_amount' => $transaction['amount']];
    }

    /**
     * Process deposit transaction refund
     */
    private function processDepositRefund($transaction) {
        // Deposits are usually not refunded as they represent incoming funds
        return ['success' => false, 'error' => 'Deposit refunds are not supported'];
    }

    /**
     * Get detailed wallet information
     */
    public function getWalletDetails($walletId) {
        try {
            // Get wallet with user info
            $stmt = $this->conn->prepare("
                SELECT 
                    cw.*,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone
                FROM crypto_wallets cw
                LEFT JOIN users u ON cw.user_id = u.id
                WHERE cw.id = ?
            ");
            $stmt->execute([$walletId]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$wallet) {
                return [
                    'success' => false,
                    'error' => 'Wallet not found'
                ];
            }

            return [
                'success' => true,
                'wallet' => $wallet
            ];

        } catch (Exception $e) {
            error_log("Get wallet details error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get wallet details'
            ];
        }
    }

    /**
     * Adjust wallet balance
     */
    public function adjustWalletBalance($walletId, $action, $amount) {
        try {
            $this->conn->beginTransaction();

            // Get current wallet balance
            $stmt = $this->conn->prepare("SELECT * FROM crypto_wallets WHERE id = ?");
            $stmt->execute([$walletId]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$wallet) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Wallet not found'
                ];
            }

            // Calculate new balance
            $currentBalance = floatval($wallet['balance']);
            $adjustmentAmount = floatval($amount);

            if ($action === 'add') {
                $newBalance = $currentBalance + $adjustmentAmount;
            } else {
                $newBalance = $currentBalance - $adjustmentAmount;
                if ($newBalance < 0) {
                    $this->conn->rollback();
                    return [
                        'success' => false,
                        'error' => 'Insufficient balance for subtraction'
                    ];
                }
            }

            // Update wallet balance
            $stmt = $this->conn->prepare("
                UPDATE crypto_wallets 
                SET balance = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$newBalance, $walletId]);

            // Log the adjustment
            $stmt = $this->conn->prepare("
                INSERT INTO transactions (
                    user_id, type, currency, amount, description, status, network
                ) VALUES (?, ?, ?, ?, ?, 'confirmed', ?)
            ");
            $description = "Admin balance adjustment: {$action} {$amount} {$wallet['currency']}";
            $stmt->execute([
                $wallet['user_id'], 
                'admin_adjustment', 
                $wallet['currency'], 
                $adjustmentAmount, 
                $description,
                $wallet['network']
            ]);

            $this->conn->commit();
            error_log("Admin adjusted wallet $walletId: {$action} {$amount} {$wallet['currency']}. New balance: {$newBalance}");

            return [
                'success' => true,
                'message' => 'Wallet balance adjusted successfully',
                'wallet_id' => $walletId,
                'action' => $action,
                'amount' => $adjustmentAmount,
                'new_balance' => $newBalance
            ];

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Adjust wallet balance error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to adjust wallet balance'
            ];
        }
    }

    /**
     * Check wallet balance against blockchain
     */
    public function checkWalletBalance($walletId) {
        try {
            // Get wallet details
            $stmt = $this->conn->prepare("SELECT * FROM crypto_wallets WHERE id = ?");
            $stmt->execute([$walletId]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$wallet) {
                return [
                    'success' => false,
                    'error' => 'Wallet not found'
                ];
            }

            // For now, return simulated blockchain balance
            // In a real implementation, this would query the actual blockchain
            $blockchainBalance = $this->getBlockchainBalance($wallet['address'], $wallet['currency'], $wallet['network']);
            $localBalance = floatval($wallet['balance']);
            
            $synced = abs($blockchainBalance - $localBalance) < 0.000001; // Consider synced if difference is less than 0.000001

            return [
                'success' => true,
                'wallet_id' => $walletId,
                'blockchain_balance' => $blockchainBalance,
                'local_balance' => $localBalance,
                'synced' => $synced,
                'difference' => $blockchainBalance - $localBalance
            ];

        } catch (Exception $e) {
            error_log("Check wallet balance error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to check wallet balance'
            ];
        }
    }

    /**
     * Sync wallet balance with blockchain
     */
    public function syncWalletBalance($walletId) {
        try {
            $this->conn->beginTransaction();

            // Get wallet details
            $stmt = $this->conn->prepare("SELECT * FROM crypto_wallets WHERE id = ?");
            $stmt->execute([$walletId]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$wallet) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Wallet not found'
                ];
            }

            // Get blockchain balance
            $blockchainBalance = $this->getBlockchainBalance($wallet['address'], $wallet['currency'], $wallet['network']);
            $localBalance = floatval($wallet['balance']);
            
            if (abs($blockchainBalance - $localBalance) < 0.000001) {
                $this->conn->rollback();
                return [
                    'success' => true,
                    'message' => 'Balances are already synced',
                    'new_balance' => $localBalance
                ];
            }

            // Update local balance to match blockchain
            $stmt = $this->conn->prepare("
                UPDATE crypto_wallets 
                SET balance = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$blockchainBalance, $walletId]);

            // Log the sync operation
            $stmt = $this->conn->prepare("
                INSERT INTO transactions (
                    user_id, type, currency, amount, description, status, network
                ) VALUES (?, ?, ?, ?, ?, 'confirmed', ?)
            ");
            $description = "Admin balance sync: {$wallet['currency']} wallet synced with blockchain";
            $stmt->execute([
                $wallet['user_id'], 
                'admin_sync', 
                $wallet['currency'], 
                $blockchainBalance - $localBalance, 
                $description,
                $wallet['network']
            ]);

            $this->conn->commit();
            error_log("Admin synced wallet $walletId: {$localBalance} -> {$blockchainBalance} {$wallet['currency']}");

            return [
                'success' => true,
                'message' => 'Wallet balance synced successfully',
                'wallet_id' => $walletId,
                'old_balance' => $localBalance,
                'new_balance' => $blockchainBalance,
                'difference' => $blockchainBalance - $localBalance
            ];

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Sync wallet balance error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to sync wallet balance'
            ];
        }
    }

    /**
     * Freeze wallet (disable transactions)
     */
    public function freezeWallet($walletId) {
        try {
            // Update wallet status to frozen
            $stmt = $this->conn->prepare("
                UPDATE crypto_wallets 
                SET is_frozen = 1, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$walletId]);

            if ($stmt->rowCount() > 0) {
                // Log the freeze action
                $stmt = $this->conn->prepare("SELECT user_id, currency, network FROM crypto_wallets WHERE id = ?");
                $stmt->execute([$walletId]);
                $wallet = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($wallet) {
                    $stmt = $this->conn->prepare("
                        INSERT INTO transactions (
                            user_id, type, currency, amount, description, status, network
                        ) VALUES (?, ?, ?, ?, ?, 'confirmed', ?)
                    ");
                    $description = "Admin freeze: {$wallet['currency']} wallet frozen";
                    $stmt->execute([
                        $wallet['user_id'], 
                        'admin_freeze', 
                        $wallet['currency'], 
                        0, 
                        $description,
                        $wallet['network']
                    ]);
                }

                error_log("Admin froze wallet: $walletId");
                return [
                    'success' => true,
                    'message' => 'Wallet frozen successfully',
                    'wallet_id' => $walletId
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'Wallet not found'
                ];
            }

        } catch (Exception $e) {
            error_log("Freeze wallet error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to freeze wallet'
            ];
        }
    }

    /**
     * Get blockchain balance (real blockchain data)
     * Uses actual blockchain APIs to get real balances
     */
    private function getBlockchainBalance($address, $currency, $network) {
        try {
            require_once __DIR__ . '/blockchain_apis.php';
            $blockchainAPI = new BlockchainAPIs();
            
            // Get real balance from blockchain
            $balanceData = $blockchainAPI->getBalance($address, $currency, $network);
            
            if ($balanceData && isset($balanceData['balance'])) {
                return floatval($balanceData['balance']);
            }
            
            return 0.0;
            
        } catch (Exception $e) {
            error_log("Error getting real blockchain balance: " . $e->getMessage());
            return 0.0;
        }
    }

    /**
     * Get detailed payment information
     */
    public function getPaymentDetails($transactionId) {
        try {
            // Get payment with user info
            $stmt = $this->conn->prepare("
                SELECT 
                    p.*,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone
                FROM payments p
                LEFT JOIN users u ON p.user_id = u.id
                WHERE p.transaction_id = ?
            ");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$payment) {
                return [
                    'success' => false,
                    'error' => 'Payment not found'
                ];
            }

            return [
                'success' => true,
                'payment' => $payment
            ];

        } catch (Exception $e) {
            error_log("Get payment details error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get payment details'
            ];
        }
    }

    /**
     * Check payment status against Bitmuk
     */
    public function checkPaymentStatus($transactionId) {
        try {
            // Get payment details
            $stmt = $this->conn->prepare("SELECT * FROM payments WHERE transaction_id = ?");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$payment) {
                return [
                    'success' => false,
                    'error' => 'Payment not found'
                ];
            }

            // For now, return simulated Bitmuk status
            // In a real implementation, this would query the actual Bitmuk API
            $simulatedBitmukStatus = $this->getSimulatedBitmukStatus($payment['bitmuk_reference']);
            $localStatus = $payment['status'];
            
            $synced = $simulatedBitmukStatus === $localStatus;

            return [
                'success' => true,
                'transaction_id' => $transactionId,
                'local_status' => $localStatus,
                'bitmuk_status' => $simulatedBitmukStatus,
                'synced' => $synced
            ];

        } catch (Exception $e) {
            error_log("Check payment status error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to check payment status'
            ];
        }
    }

    /**
     * Retry a failed payment
     */
    public function retryPayment($transactionId) {
        try {
            $this->conn->beginTransaction();

            // Get payment details
            $stmt = $this->conn->prepare("SELECT * FROM payments WHERE transaction_id = ? AND status = 'failed'");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$payment) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Payment not found or not failed'
                ];
            }

            // Update payment status to pending for retry
            $stmt = $this->conn->prepare("
                UPDATE payments 
                SET status = 'pending', updated_at = CURRENT_TIMESTAMP 
                WHERE transaction_id = ?
            ");
            $stmt->execute([$transactionId]);

            // Log the retry action
            $stmt = $this->conn->prepare("
                INSERT INTO transactions (
                    user_id, type, currency, amount, description, status
                ) VALUES (?, ?, ?, ?, ?, 'confirmed')
            ");
            $description = "Admin retry: Payment retry initiated for {$payment['transaction_id']}";
            $stmt->execute([
                $payment['user_id'], 
                'admin_retry', 
                $payment['currency'], 
                $payment['amount'], 
                $description
            ]);

            $this->conn->commit();
            error_log("Admin retried payment: $transactionId");

            return [
                'success' => true,
                'message' => 'Payment retry initiated successfully',
                'transaction_id' => $transactionId,
                'new_status' => 'pending'
            ];

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Retry payment error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to retry payment'
            ];
        }
    }

    /**
     * Update payment status
     */
    public function updatePaymentStatus($transactionId, $status) {
        try {
            $this->conn->beginTransaction();

            // Get payment details
            $stmt = $this->conn->prepare("SELECT * FROM payments WHERE transaction_id = ?");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$payment) {
                $this->conn->rollback();
                return [
                    'success' => false,
                    'error' => 'Payment not found'
                ];
            }

            // Update payment status
            $stmt = $this->conn->prepare("
                UPDATE payments 
                SET status = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE transaction_id = ?
            ");
            $stmt->execute([$status, $transactionId]);

            // If status is completed, credit the user's account
            if ($status === 'completed') {
                $this->creditUserAccount($payment['user_id'], $payment['amount'], $payment['currency']);
            }

            // Log the status change
            $stmt = $this->conn->prepare("
                INSERT INTO transactions (
                    user_id, type, currency, amount, description, status
                ) VALUES (?, ?, ?, ?, ?, 'confirmed')
            ");
            $description = "Admin status update: Payment {$payment['transaction_id']} status changed to {$status}";
            $stmt->execute([
                $payment['user_id'], 
                'admin_status_update', 
                $payment['currency'], 
                $payment['amount'], 
                $description
            ]);

            $this->conn->commit();
            error_log("Admin updated payment status: $transactionId -> $status");

            return [
                'success' => true,
                'message' => 'Payment status updated successfully',
                'transaction_id' => $transactionId,
                'new_status' => $status
            ];

        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Update payment status error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to update payment status'
            ];
        }
    }

    /**
     * Credit user account after successful payment
     */
    private function creditUserAccount($userId, $amount, $currency) {
        try {
            // Check if user already has a balance for this currency
            $stmt = $this->conn->prepare("
                SELECT * FROM user_fiat_balances 
                WHERE user_id = ? AND currency_code = ?
            ");
            $stmt->execute([$userId, $currency]);
            $existingBalance = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($existingBalance) {
                // Update existing balance
                $newBalance = $existingBalance['balance'] + $amount;
                $stmt = $this->conn->prepare("
                    UPDATE user_fiat_balances 
                    SET balance = ?, updated_at = CURRENT_TIMESTAMP 
                    WHERE user_id = ? AND currency_code = ?
                ");
                $stmt->execute([$newBalance, $userId, $currency]);
            } else {
                // Create new balance record
                $stmt = $this->conn->prepare("
                    INSERT INTO user_fiat_balances (user_id, currency_code, balance, created_at, updated_at) 
                    VALUES (?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                ");
                $stmt->execute([$userId, $currency, $amount]);
            }

            error_log("Credited user $userId account: +$amount $currency");

        } catch (Exception $e) {
            error_log("Credit user account error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get simulated Bitmuk status
     */
    private function getSimulatedBitmukStatus($bitmukReference) {
        // For now, return a simulated status
        // In production, this would query the actual Bitmuk API
        
        $statuses = ['pending', 'completed', 'failed'];
        return $statuses[array_rand($statuses)];
    }

    /**
     * Get audit logs with filters
     */
    public function getAuditLogs($filters = []) {
        try {
            // Check if audit_logs table exists
            $tableCheck = $this->conn->query("SELECT name FROM sqlite_master WHERE type='table' AND name='audit_logs'");
            if (!$tableCheck->fetch()) {
                // Return empty result if table doesn't exist
                return [
                    'success' => true,
                    'logs' => [],
                    'total' => 0,
                    'limit' => intval($filters['limit'] ?? 50),
                    'offset' => intval($filters['offset'] ?? 0)
                ];
            }

            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['user_id'])) {
                $whereClause .= " AND al.user_id = ?";
                $params[] = $filters['user_id'];
            }
            
            if (!empty($filters['action'])) {
                $whereClause .= " AND al.action LIKE ?";
                $params[] = "%{$filters['action']}%";
            }
            
            if (!empty($filters['table_name'])) {
                $whereClause .= " AND al.table_name = ?";
                $params[] = $filters['table_name'];
            }
            
            if (!empty($filters['date_from'])) {
                $whereClause .= " AND DATE(al.created_at) >= ?";
                $params[] = $filters['date_from'];
            }
            
            if (!empty($filters['date_to'])) {
                $whereClause .= " AND DATE(al.created_at) <= ?";
                $params[] = $filters['date_to'];
            }
            
            // Get total count
            $countStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM audit_logs al
                LEFT JOIN users u ON al.user_id = u.id
                $whereClause
            ");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
            
            // Get pagination parameters
            $limit = intval($filters['limit'] ?? 50);
            $offset = intval($filters['offset'] ?? 0);
            
            // Get logs with user information
            $stmt = $this->conn->prepare("
                SELECT 
                    al.*,
                    u.first_name,
                    u.last_name,
                    u.email
                FROM audit_logs al
                LEFT JOIN users u ON al.user_id = u.id
                $whereClause
                ORDER BY al.created_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'logs' => $logs,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
            
        } catch (Exception $e) {
            error_log("Get audit logs error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get audit logs'
            ];
        }
    }

    /**
     * Toggle payment system on/off
     */
    public function togglePaymentSystem($enabled) {
        try {
            // This would typically update a system setting
            // For now, we'll just log the action
            error_log("Payment system toggled: " . ($enabled ? 'enabled' : 'disabled'));
            
            return [
                'success' => true,
                'message' => 'Payment system ' . ($enabled ? 'enabled' : 'disabled'),
                'enabled' => $enabled
            ];
            
        } catch (Exception $e) {
            error_log("Toggle payment system error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to toggle payment system'
            ];
        }
    }

    /**
     * Clear all transactions (dangerous operation)
     */
    public function clearAllTransactions() {
        try {
            // This is a dangerous operation - in production you'd want additional safeguards
            $this->conn->beginTransaction();
            
            // Count transactions before deletion
            $countStmt = $this->conn->prepare("SELECT COUNT(*) as count FROM transactions");
            $countStmt->execute();
            $count = $countStmt->fetch(PDO::FETCH_ASSOC)['count'];
            
            // Delete all transactions
            $this->conn->exec("DELETE FROM transactions");
            
            $this->conn->commit();
            
            error_log("All transactions cleared: $count transactions deleted");
            
            return [
                'success' => true,
                'message' => "Cleared $count transactions",
                'count' => $count
            ];
            
        } catch (Exception $e) {
            if (isset($this->conn)) {
                $this->conn->rollback();
            }
            error_log("Clear all transactions error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to clear transactions'
            ];
        }
    }

    /**
     * Get KYC applications with filters
     */
    public function getKYCApplications($filters = []) {
        try {
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Apply filters
            if (!empty($filters['status'])) {
                $whereClause .= " AND u.kyc_status = ?";
                $params[] = $filters['status'];
            }
            
            if (!empty($filters['submitted_only']) && $filters['submitted_only'] === 'true') {
                $whereClause .= " AND u.kyc_documents_submitted = 1";
            }
            
            if (!empty($filters['date_from'])) {
                $whereClause .= " AND DATE(u.updated_at) >= ?";
                $params[] = $filters['date_from'];
            }
            
            if (!empty($filters['date_to'])) {
                $whereClause .= " AND DATE(u.updated_at) <= ?";
                $params[] = $filters['date_to'];
            }
            
            // Get total count
            $countStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM users u
                $whereClause
                AND u.kyc_documents_submitted = 1
            ");
            $countStmt->execute($params);
            $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
            
            // Get pagination parameters
            $limit = intval($filters['limit'] ?? 50);
            $offset = intval($filters['offset'] ?? 0);
            
            // Get KYC applications
            $stmt = $this->conn->prepare("
                SELECT 
                    u.id,
                    u.first_name,
                    u.last_name,
                    u.email,
                    u.phone,
                    u.kyc_status,
                    u.kyc_documents_submitted,
                    u.kyc_document_type,
                    u.kyc_document_number,
                    u.kyc_verified_at,
                    u.created_at,
                    u.updated_at
                FROM users u
                $whereClause
                AND u.kyc_documents_submitted = 1
                ORDER BY u.updated_at DESC
                LIMIT ? OFFSET ?
            ");
            
            $params[] = $limit;
            $params[] = $offset;
            $stmt->execute($params);
            $applications = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'applications' => $applications,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
            
        } catch (Exception $e) {
            error_log("Get KYC applications error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get KYC applications'
            ];
        }
    }

    /**
     * Verify KYC application
     */
    public function verifyKYC($userId, $reason = '') {
        try {
            $this->conn->beginTransaction();
            
            // Update user KYC status
            $stmt = $this->conn->prepare("
                UPDATE users 
                SET 
                    kyc_status = 'verified',
                    is_verified = 1,
                    kyc_verified_at = CURRENT_TIMESTAMP,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            
            // Log the verification action
            $this->logAdminAction('kyc_verified', 'users', $userId, [
                'reason' => $reason,
                'admin_id' => 'admin'
            ]);
            
            $this->conn->commit();
            
            return [
                'success' => true,
                'message' => 'KYC application verified successfully'
            ];
            
        } catch (Exception $e) {
            if (isset($this->conn)) {
                $this->conn->rollback();
            }
            error_log("Verify KYC error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to verify KYC application'
            ];
        }
    }

    /**
     * Reject KYC application
     */
    public function rejectKYC($userId, $reason) {
        try {
            $this->conn->beginTransaction();
            
            // Update user KYC status
            $stmt = $this->conn->prepare("
                UPDATE users 
                SET 
                    kyc_status = 'rejected',
                    is_verified = 0,
                    kyc_documents_submitted = 0,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
            
            // Log the rejection action
            $this->logAdminAction('kyc_rejected', 'users', $userId, [
                'reason' => $reason,
                'admin_id' => 'admin'
            ]);
            
            $this->conn->commit();
            
            return [
                'success' => true,
                'message' => 'KYC application rejected successfully'
            ];
            
        } catch (Exception $e) {
            if (isset($this->conn)) {
                $this->conn->rollback();
            }
            error_log("Reject KYC error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to reject KYC application'
            ];
        }
    }

    /**
     * Log admin action
     */
    private function logAdminAction($action, $table, $recordId, $details = []) {
        try {
            // Check if audit_logs table exists
            $tableCheck = $this->conn->query("SELECT name FROM sqlite_master WHERE type='table' AND name='audit_logs'");
            if (!$tableCheck->fetch()) {
                return; // Skip logging if table doesn't exist
            }
            
            $stmt = $this->conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, details, created_at)
                VALUES (?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([
                'admin',
                $action,
                $table,
                $recordId,
                json_encode($details)
            ]);
            
        } catch (Exception $e) {
            error_log("Log admin action error: " . $e->getMessage());
        }
    }
}
?>
