<?php
/**
 * Auto Wallet Generator Service
 * Automatically generates crypto wallets for new users
 */

require_once __DIR__ . '/wallet_generator.php';

class AutoWalletGenerator {
    private $conn;
    private $walletGenerator;
    
    // Supported networks and currencies for auto-generation
    private $supportedNetworks = [
        'erc20' => ['USDT', 'USDC'],
        'trc20' => ['USDT', 'USDC'],
        'bep20' => ['USDT', 'USDC', 'BNB'],
        'polygon' => ['USDT', 'USDC', 'MATIC'],
        'solana' => ['USDT', 'USDC', 'SOL'],
        'base' => ['USDT', 'USDC']
    ];
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->walletGenerator = new WalletGenerator($connection);
    }
    
    /**
     * Generate all supported wallets for a new user
     */
    public function generateUserWallets($userId) {
        $generatedWallets = [];
        $errors = [];
        
        try {
            error_log("Starting automatic wallet generation for user ID: $userId");
            
            // Generate wallets for each supported network
            foreach ($this->supportedNetworks as $network => $currencies) {
                foreach ($currencies as $currency) {
                    try {
                        $result = $this->walletGenerator->generateWallet($userId, $network, $currency);
                        
                        if ($result['success']) {
                            $generatedWallets[] = [
                                'network' => $network,
                                'currency' => $currency,
                                'address' => $result['wallet']['address'],
                                'id' => $result['wallet']['id']
                            ];
                            error_log("Generated $network $currency wallet for user $userId: {$result['wallet']['address']}");
                        } else {
                            $errors[] = "Failed to generate $network $currency wallet: " . $result['error'];
                            error_log("Failed to generate $network $currency wallet for user $userId: " . $result['error']);
                        }
                        
                        // Small delay to prevent overwhelming the system
                        usleep(100000); // 0.1 second delay
                        
                    } catch (Exception $e) {
                        $errorMsg = "Exception generating $network $currency wallet: " . $e->getMessage();
                        $errors[] = $errorMsg;
                        error_log($errorMsg);
                    }
                }
            }
            
            // Create initial fiat balance entry
            $this->createInitialFiatBalance($userId);
            
            error_log("Wallet generation completed for user $userId. Generated: " . count($generatedWallets) . " wallets, Errors: " . count($errors));
            
            return [
                'success' => true,
                'generated_wallets' => $generatedWallets,
                'total_wallets' => count($generatedWallets),
                'errors' => $errors,
                'message' => 'Wallets generated successfully'
            ];
            
        } catch (Exception $e) {
            error_log("Critical error in auto wallet generation for user $userId: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to generate wallets: ' . $e->getMessage(),
                'generated_wallets' => $generatedWallets,
                'errors' => $errors
            ];
        }
    }
    
    /**
     * Create initial fiat balance entry for user
     */
    private function createInitialFiatBalance($userId) {
        try {
            // Check if user_fiat_balances table exists
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS user_fiat_balances (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    currency TEXT NOT NULL,
                    balance REAL DEFAULT 0.00000000,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    UNIQUE(user_id, currency)
                )
            ");
            
            // Get user's country to determine default currency
            $stmt = $this->conn->prepare("SELECT country_code FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                $currency_map = [
                    'UG' => 'UGX',
                    'KE' => 'KES',
                    'TZ' => 'TZS',
                    'RW' => 'RWF',
                    'CD' => 'CDF',
                    'NG' => 'NGN',
                    'US' => 'USD'
                ];
                $defaultCurrency = $currency_map[$user['country_code']] ?? 'UGX';
                
                // Create initial balance entry
                $stmt = $this->conn->prepare("
                    INSERT OR IGNORE INTO user_fiat_balances (user_id, currency_code, balance) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$userId, $defaultCurrency, 0.00000000]);
                
                error_log("Created initial fiat balance for user $userId: $defaultCurrency");
            }
            
        } catch (Exception $e) {
            error_log("Error creating initial fiat balance for user $userId: " . $e->getMessage());
        }
    }
    
    /**
     * Get wallet generation summary for a user
     */
    public function getUserWalletSummary($userId) {
        try {
            $stmt = $this->conn->prepare("
                SELECT network, currency, address, balance, created_at 
                FROM crypto_wallets 
                WHERE user_id = ? 
                ORDER BY network, currency
            ");
            $stmt->execute([$userId]);
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $summary = [];
            foreach ($wallets as $wallet) {
                $summary[] = [
                    'network' => $wallet['network'],
                    'currency' => $wallet['currency'],
                    'address' => $wallet['address'],
                    'balance' => $wallet['balance'],
                    'created_at' => $wallet['created_at']
                ];
            }
            
            return [
                'success' => true,
                'user_id' => $userId,
                'total_wallets' => count($summary),
                'wallets' => $summary
            ];
            
        } catch (Exception $e) {
            error_log("Error getting wallet summary for user $userId: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get wallet summary: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Regenerate wallets for existing user (admin function)
     */
    public function regenerateUserWallets($userId) {
        try {
            // First, delete existing wallets
            $stmt = $this->conn->prepare("DELETE FROM crypto_wallets WHERE user_id = ?");
            $stmt->execute([$userId]);
            
            error_log("Deleted existing wallets for user $userId, regenerating...");
            
            // Generate new wallets
            return $this->generateUserWallets($userId);
            
        } catch (Exception $e) {
            error_log("Error regenerating wallets for user $userId: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to regenerate wallets: ' . $e->getMessage()
            ];
        }
    }
}
