/**
 * Backup and Monitoring Service
 * Handles database backups and system monitoring
 */

class BackupService {
    private static $backup_dir;
    private static $max_backups = 7; // Keep 7 days of backups
    
    public static function initialize() {
        self::$backup_dir = __DIR__ . '/../backups/';
        
        // Create backup directory if it doesn't exist
        if (!is_dir(self::$backup_dir)) {
            mkdir(self::$backup_dir, 0755, true);
        }
    }
    
    public static function createDatabaseBackup() {
        try {
            require_once __DIR__ . '/../config/env.php';
            EnvConfig::load(__DIR__ . '/../config.env');
            
            $db_host = EnvConfig::get('DB_HOST', 'localhost');
            $db_name = EnvConfig::get('DB_NAME', 'crypto_exchange');
            $db_user = EnvConfig::get('DB_USER', 'root');
            $db_pass = EnvConfig::get('DB_PASS', '');
            
            $timestamp = date('Y-m-d_H-i-s');
            $backup_file = self::$backup_dir . "backup_{$db_name}_{$timestamp}.sql";
            
            // Create mysqldump command
            $command = "mysqldump -h {$db_host} -u {$db_user}";
            if ($db_pass) {
                $command .= " -p{$db_pass}";
            }
            $command .= " {$db_name} > {$backup_file}";
            
            // Execute backup
            exec($command, $output, $return_code);
            
            if ($return_code === 0 && file_exists($backup_file)) {
                // Compress the backup
                $compressed_file = $backup_file . '.gz';
                exec("gzip {$backup_file}");
                
                // Clean old backups
                self::cleanOldBackups();
                
                error_log("Database backup created: {$compressed_file}");
                return $compressed_file;
            } else {
                error_log("Database backup failed with return code: {$return_code}");
                return false;
            }
            
        } catch (Exception $e) {
            error_log("Backup creation error: " . $e->getMessage());
            return false;
        }
    }
    
    private static function cleanOldBackups() {
        try {
            $files = glob(self::$backup_dir . "backup_*.sql.gz");
            
            if (count($files) > self::$max_backups) {
                // Sort by modification time (oldest first)
                usort($files, function($a, $b) {
                    return filemtime($a) - filemtime($b);
                });
                
                // Remove oldest backups
                $files_to_remove = array_slice($files, 0, count($files) - self::$max_backups);
                foreach ($files_to_remove as $file) {
                    unlink($file);
                    error_log("Removed old backup: {$file}");
                }
            }
            
        } catch (Exception $e) {
            error_log("Backup cleanup error: " . $e->getMessage());
        }
    }
    
    public static function restoreDatabaseBackup($backup_file) {
        try {
            require_once __DIR__ . '/../config/env.php';
            EnvConfig::load(__DIR__ . '/../config.env');
            
            $db_host = EnvConfig::get('DB_HOST', 'localhost');
            $db_name = EnvConfig::get('DB_NAME', 'crypto_exchange');
            $db_user = EnvConfig::get('DB_USER', 'root');
            $db_pass = EnvConfig::get('DB_PASS', '');
            
            // Decompress if needed
            if (strpos($backup_file, '.gz') !== false) {
                $temp_file = str_replace('.gz', '', $backup_file);
                exec("gunzip -c {$backup_file} > {$temp_file}");
                $backup_file = $temp_file;
            }
            
            // Create restore command
            $command = "mysql -h {$db_host} -u {$db_user}";
            if ($db_pass) {
                $command .= " -p{$db_pass}";
            }
            $command .= " {$db_name} < {$backup_file}";
            
            // Execute restore
            exec($command, $output, $return_code);
            
            // Clean up temp file
            if (isset($temp_file) && file_exists($temp_file)) {
                unlink($temp_file);
            }
            
            if ($return_code === 0) {
                error_log("Database restored from: {$backup_file}");
                return true;
            } else {
                error_log("Database restore failed with return code: {$return_code}");
                return false;
            }
            
        } catch (Exception $e) {
            error_log("Backup restore error: " . $e->getMessage());
            return false;
        }
    }
    
    public static function getBackupList() {
        try {
            $files = glob(self::$backup_dir . "backup_*.sql.gz");
            $backups = [];
            
            foreach ($files as $file) {
                $backups[] = [
                    'filename' => basename($file),
                    'size' => filesize($file),
                    'created' => date('Y-m-d H:i:s', filemtime($file)),
                    'path' => $file
                ];
            }
            
            // Sort by creation time (newest first)
            usort($backups, function($a, $b) {
                return strtotime($b['created']) - strtotime($a['created']);
            });
            
            return $backups;
            
        } catch (Exception $e) {
            error_log("Backup list error: " . $e->getMessage());
            return [];
        }
    }
}

class MonitoringService {
    private static $log_file;
    private static $metrics = [];
    
    public static function initialize() {
        self::$log_file = __DIR__ . '/../logs/monitoring.log';
        
        // Create logs directory if it doesn't exist
        $log_dir = dirname(self::$log_file);
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
    }
    
    public static function logMetric($metric_name, $value, $tags = []) {
        try {
            $timestamp = date('Y-m-d H:i:s');
            $log_entry = [
                'timestamp' => $timestamp,
                'metric' => $metric_name,
                'value' => $value,
                'tags' => $tags
            ];
            
            file_put_contents(self::$log_file, json_encode($log_entry) . "\n", FILE_APPEND | LOCK_EX);
            
            // Store in memory for quick access
            self::$metrics[$metric_name] = [
                'value' => $value,
                'timestamp' => $timestamp,
                'tags' => $tags
            ];
            
        } catch (Exception $e) {
            error_log("Metric logging error: " . $e->getMessage());
        }
    }
    
    public static function getSystemMetrics() {
        try {
            $metrics = [
                'timestamp' => date('Y-m-d H:i:s'),
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true),
                'memory_limit' => ini_get('memory_limit'),
                'disk_free' => disk_free_space(__DIR__ . '/../'),
                'disk_total' => disk_total_space(__DIR__ . '/../'),
                'load_average' => function_exists('sys_getloadavg') ? sys_getloadavg() : null,
                'uptime' => self::getSystemUptime(),
                'database_status' => self::checkDatabaseStatus(),
                'blockchain_service_status' => self::checkBlockchainServiceStatus()
            ];
            
            return $metrics;
            
        } catch (Exception $e) {
            error_log("System metrics error: " . $e->getMessage());
            return [];
        }
    }
    
    private static function getSystemUptime() {
        try {
            if (function_exists('sys_getloadavg')) {
                return [
                    'load_average' => sys_getloadavg(),
                    'memory_limit' => ini_get('memory_limit'),
                    'max_execution_time' => ini_get('max_execution_time')
                ];
            }
        } catch (Exception $e) {
            // Ignore
        }
        
        return ['status' => 'unknown'];
    }
    
    private static function checkDatabaseStatus() {
        try {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            $start_time = microtime(true);
            $stmt = $conn->query("SELECT 1");
            $end_time = microtime(true);
            
            return [
                'status' => 'healthy',
                'response_time' => round(($end_time - $start_time) * 1000, 2) . 'ms'
            ];
            
        } catch (Exception $e) {
            return [
                'status' => 'unhealthy',
                'error' => $e->getMessage()
            ];
        }
    }
    
    private static function checkBlockchainServiceStatus() {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'http://localhost:3001/health');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            
            $start_time = microtime(true);
            $response = curl_exec($ch);
            $end_time = microtime(true);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                return [
                    'status' => 'healthy',
                    'response_time' => round(($end_time - $start_time) * 1000, 2) . 'ms'
                ];
            } else {
                return [
                    'status' => 'unhealthy',
                    'http_code' => $httpCode
                ];
            }
            
        } catch (Exception $e) {
            return [
                'status' => 'unhealthy',
                'error' => $e->getMessage()
            ];
        }
    }
    
    public static function sendAlert($alert_type, $message, $severity = 'warning') {
        try {
            $alert = [
                'timestamp' => date('Y-m-d H:i:s'),
                'type' => $alert_type,
                'message' => $message,
                'severity' => $severity
            ];
            
            // Log the alert
            error_log("ALERT: " . json_encode($alert));
            
            // In production, you would send to monitoring service (e.g., PagerDuty, Slack)
            // For now, just log it
            
            return true;
            
        } catch (Exception $e) {
            error_log("Alert sending error: " . $e->getMessage());
            return false;
        }
    }
}

// Initialize services
BackupService::initialize();
MonitoringService::initialize();
