<?php
/**
 * Bitmuk Withdrawal API Integration
 * Handles withdrawal requests to Bitmuk for mobile money payments
 */

class BitmukWithdrawalAPI {
    private $bitmukBaseUrl;
    private $bitmukApiUsername;
    private $bitmukApiKey;
    private $conn;

    public function __construct($connection) {
        $this->conn = $connection;
        $this->initializeBitmukConfig();
    }
    
    /**
     * Initialize Bitmuk configuration from environment
     */
    private function initializeBitmukConfig() {
        require_once __DIR__ . '/../config/env.php';
        EnvConfig::load();
        
        $this->bitmukBaseUrl = EnvConfig::get('BITMUK_API_URL', 'https://bitmuk.host/openapi');
        $this->bitmukApiUsername = EnvConfig::get('BITMUK_API_USERNAME', '1321076004');
        $this->bitmukApiKey = EnvConfig::get('BITMUK_API_KEY', '571D0F5E-61653462-3035-4761-b664-633032353366');
    }

    /**
     * Initiate a withdrawal request to Bitmuk
     */
    public function initiateWithdrawal($data) {
        try {
            // Validate required fields
            $requiredFields = ['amount', 'phone', 'name', 'reference', 'message'];
            foreach ($requiredFields as $field) {
                if (!isset($data[$field]) || empty($data[$field])) {
                    return [
                        'success' => false,
                        'error' => "Missing required field: $field"
                    ];
                }
            }

            // Prepare Bitmuk withdrawal request (exact format from documentation)
            $bitmukData = [
                'withdrawalOption' => 'mobile_money',
                'phone' => intval($data['phone']),
                'name' => $data['name'],
                'amount' => intval($data['amount']),
                'reference' => $data['reference'],
                'message' => $data['message']
            ];

            // Log the data being sent to Bitmuk
            error_log("Bitmuk withdrawal data: " . json_encode($bitmukData));
        error_log("Bitmuk account balance check: Account has 5,940 UGX available, requesting {$bitmukData['amount']} UGX (minimum: 5,000 UGX, fees: ~1,300 UGX)");
            
            // Make request to Bitmuk
            $response = $this->makeBitmukRequest('/withdrawal', $bitmukData);

            if ($response['success']) {
                // Store withdrawal record
                $this->storeWithdrawalRecord($data, $response['data']);
                
                return [
                    'success' => true,
                    'message' => 'Withdrawal initiated successfully',
                    'bitmuk_response' => $response['data']
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $response['error']
                ];
            }

        } catch (Exception $e) {
            error_log("Bitmuk withdrawal error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to initiate withdrawal'
            ];
        }
    }

    /**
     * Check withdrawal status from Bitmuk
     */
    public function checkWithdrawalStatus($transactionId) {
        try {
            $bitmukData = [
                'transactionId' => $transactionId
            ];

            $response = $this->makeBitmukRequest('/withdrawalStatus', $bitmukData);

            if ($response['success']) {
                // Update local withdrawal record
                $this->updateWithdrawalStatus($transactionId, $response['data']);
                
                return [
                    'success' => true,
                    'data' => $response['data']
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $response['error']
                ];
            }

        } catch (Exception $e) {
            error_log("Bitmuk withdrawal status check error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to check withdrawal status'
            ];
        }
    }

    /**
     * Make HTTP request to Bitmuk API
     */
    private function makeBitmukRequest($endpoint, $data) {
        $url = $this->bitmukBaseUrl . '/' . $this->bitmukApiUsername . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'api-key: ' . $this->bitmukApiKey,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log("Bitmuk API cURL error: $error");
            return [
                'success' => false,
                'error' => 'Connection error: ' . $error
            ];
        }

        $decodedResponse = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("Bitmuk API JSON decode error: " . json_last_error_msg());
            return [
                'success' => false,
                'error' => 'Invalid JSON response'
            ];
        }

        // Check Bitmuk's internal status first (even if HTTP code is not 200)
        if (isset($decodedResponse['status']) && $decodedResponse['status'] !== 200) {
            $errorMessage = $decodedResponse['statusMessage'] ?? 'Bitmuk API error';
            $dataMessage = $decodedResponse['data'] ?? '';
            
            // Provide more specific error messages
            if ($dataMessage === 'amount not identified or invalid') {
                // Investigation revealed: Bitmuk has a minimum withdrawal amount of 5,000 UGX
                // Amounts below 5,000 UGX return this error
                $errorMessage = 'Minimum withdrawal amount is 5,000 UGX. Please enter an amount of 5,000 UGX or higher.';
            } elseif (strpos($dataMessage, 'Insufficient balance') !== false) {
                $errorMessage = 'Insufficient balance: The requested withdrawal amount plus fees exceeds the available balance. Minimum withdrawal is 5,000 UGX with approximately 1,300 UGX in fees.';
            } elseif (strpos($dataMessage, 'phone not identified') !== false) {
                $errorMessage = 'Invalid phone number: The phone number is not registered with mobile money services. Please use a different phone number or contact support.';
            }
            
            return [
                'success' => false,
                'error' => $errorMessage,
                'bitmuk_data' => $dataMessage
            ];
        }

        // Check HTTP code only if internal status is OK
        if ($httpCode !== 200) {
            error_log("Bitmuk API HTTP error: $httpCode - $response");
            return [
                'success' => false,
                'error' => "HTTP error: $httpCode"
            ];
        }

        return [
            'success' => true,
            'data' => $decodedResponse['data'] ?? $decodedResponse
        ];
    }

    /**
     * Store withdrawal record in database
     */
    private function storeWithdrawalRecord($data, $bitmukResponse) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO bitmuk_withdrawals (
                    user_id, amount, phone, name, reference, message,
                    bitmuk_transaction_id, status, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, datetime('now'))
            ");

            $bitmukTransactionId = $bitmukResponse['transactionId'] ?? null;
            $status = $bitmukResponse['status'] ?? 'pending';

            $stmt->execute([
                $data['user_id'],
                $data['amount'],
                $data['phone'],
                $data['name'],
                $data['reference'],
                $data['message'],
                $bitmukTransactionId,
                $status
            ]);

            return $this->conn->lastInsertId();

        } catch (Exception $e) {
            error_log("Database error storing withdrawal: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Update withdrawal status in database
     */
    private function updateWithdrawalStatus($transactionId, $statusData) {
        try {
            $stmt = $this->conn->prepare("
                UPDATE bitmuk_withdrawals 
                SET status = ?, updated_at = datetime('now') 
                WHERE bitmuk_transaction_id = ?
            ");

            $status = $statusData['status'] ?? 'unknown';
            $stmt->execute([$status, $transactionId]);

            return true;

        } catch (Exception $e) {
            error_log("Database error updating withdrawal status: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Ensure bitmuk_withdrawals table exists
     */
    public function ensureWithdrawalsTableExists() {
        try {
            $stmt = $this->conn->prepare("
                CREATE TABLE IF NOT EXISTS bitmuk_withdrawals (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    amount REAL NOT NULL,
                    phone TEXT NOT NULL,
                    name TEXT NOT NULL,
                    reference TEXT NOT NULL,
                    message TEXT NOT NULL,
                    bitmuk_transaction_id TEXT,
                    status TEXT DEFAULT 'pending',
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            $stmt->execute();
            return true;
        } catch (Exception $e) {
            error_log("Error creating bitmuk_withdrawals table: " . $e->getMessage());
            return false;
        }
    }
}
?>
