<?php
/**
 * Real Blockchain API Integration
 * Production-ready blockchain API integration for multiple networks
 */

class BlockchainAPIs {
    private $conn;
    private $apiKeys;
    private $networkConfigs;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->initializeAPIKeys();
        $this->initializeNetworkConfigs();
    }
    
    /**
     * Initialize API keys from environment or config
     */
    private function initializeAPIKeys() {
        // Load environment configuration
        require_once __DIR__ . '/../config/env.php';
        EnvConfig::load();
        
        $this->apiKeys = [
            'etherscan' => EnvConfig::get('ETHERSCAN_API_KEY', 'YourEtherscanAPIKey'),
            'bscscan' => EnvConfig::get('BSCSCAN_API_KEY', 'YourBSCScanAPIKey'),
            'polygonscan' => EnvConfig::get('POLYGONSCAN_API_KEY', 'YourPolygonScanAPIKey'),
            'basescan' => EnvConfig::get('BASESCAN_API_KEY', 'YourBaseScanAPIKey'),
            'tronscan' => EnvConfig::get('TRONSCAN_API_KEY', 'YourTronScanAPIKey'),
            'trongrid' => EnvConfig::get('TRONGRID_API_KEY', 'YourTronGridAPIKey'),
            'solscan' => EnvConfig::get('SOLSCAN_API_KEY', 'YourSolscanAPIKey'),
            'moralis' => EnvConfig::get('MORALIS_API_KEY', 'YourMoralisAPIKey'),
            'alchemy' => EnvConfig::get('ALCHEMY_API_KEY', 'YourAlchemyAPIKey'),
            'infura' => EnvConfig::get('INFURA_API_KEY', 'YourInfuraAPIKey')
        ];
    }
    
    /**
     * Initialize network configurations
     */
    private function initializeNetworkConfigs() {
        $this->networkConfigs = [
            'ethereum' => [
                'name' => 'Ethereum',
                'chainId' => 1,
                'symbol' => 'ETH',
                'rpc_url' => 'https://mainnet.infura.io/v3/' . $this->apiKeys['infura'],
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://etherscan.io',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'alchemy_url' => 'https://eth-mainnet.g.alchemy.com/v2/' . $this->apiKeys['alchemy'],
                'supported_tokens' => ['USDT', 'USDC', 'DAI', 'WBTC'],
                'token_contracts' => [
                    'USDT' => '0xdAC17F958D2ee523a2206206994597C13D831ec7',
                    'USDC' => '0xA0b86a33E6441c8C06DdD0C5E4E8f4a2F2B5b5b5'
                ],
                'min_confirmations' => 12,
                'avg_block_time' => 13 // seconds
            ],
            'bsc' => [
                'name' => 'Binance Smart Chain',
                'chainId' => 56,
                'symbol' => 'BNB',
                'rpc_url' => 'https://bsc-dataseed.binance.org/',
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://bscscan.com',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&chainid=56&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'supported_tokens' => ['USDT', 'USDC', 'BUSD', 'CAKE'],
                'token_contracts' => [
                    'USDT' => '0x55d398326f99059fF775485246999027B3197955',
                    'USDC' => '0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d'
                ],
                'min_confirmations' => 3,
                'avg_block_time' => 3 // seconds
            ],
            'polygon' => [
                'name' => 'Polygon',
                'chainId' => 137,
                'symbol' => 'MATIC',
                'rpc_url' => 'https://polygon-rpc.com/',
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://polygonscan.com',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&chainid=137&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'supported_tokens' => ['USDT', 'USDC', 'DAI', 'WETH'],
                'token_contracts' => [
                    'USDT' => '0xc2132D05D31c914a87C6611C10748AEb04B58e8F',
                    'USDC' => '0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174'
                ],
                'min_confirmations' => 30,
                'avg_block_time' => 2 // seconds
            ],
            'base' => [
                'name' => 'Base',
                'chainId' => 8453,
                'symbol' => 'ETH',
                'rpc_url' => 'https://mainnet.base.org',
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://basescan.org',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&chainid=8453&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'supported_tokens' => ['USDC', 'DAI', 'USDT'],
                'token_contracts' => [
                    'USDC' => '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913',
                    'USDT' => '0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb'
                ],
                'min_confirmations' => 2,
                'avg_block_time' => 2 // seconds
            ],
            'arbitrum' => [
                'name' => 'Arbitrum One',
                'chainId' => 42161,
                'symbol' => 'ETH',
                'rpc_url' => 'https://arb1.arbitrum.io/rpc',
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://arbiscan.io',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&chainid=42161&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'supported_tokens' => ['USDC', 'USDT', 'DAI', 'ARB'],
                'token_contracts' => [
                    'USDC' => '0xaf88d065e77c8cC2239327C5EDb3A432268e5831',
                    'USDT' => '0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9',
                    'ARB' => '0x912CE59144191C1204E64559FE8253a0e49E6548'
                ],
                'min_confirmations' => 1,
                'avg_block_time' => 1 // seconds
            ],
            'optimism' => [
                'name' => 'Optimism',
                'chainId' => 10,
                'symbol' => 'ETH',
                'rpc_url' => 'https://mainnet.optimism.io',
                'explorer_api' => 'https://api.etherscan.io/v2/api',
                'explorer_url' => 'https://optimistic.etherscan.io',
                'gas_tracker' => 'https://api.etherscan.io/v2/api?module=gastracker&action=gasoracle&chainid=10&apikey=' . $this->apiKeys['etherscan'],
                'api_key' => $this->apiKeys['etherscan'],
                'supported_tokens' => ['USDC', 'USDT', 'DAI', 'OP'],
                'token_contracts' => [
                    'USDC' => '0x0b2C639c533813f4Aa9D7837CAf62653d097Ff85',
                    'USDT' => '0x94b008aA00579c1307B0EF2c499aD98a8ce58e58',
                    'OP' => '0x4200000000000000000000000000000000000042'
                ],
                'min_confirmations' => 1,
                'avg_block_time' => 2 // seconds
            ],
            'tron' => [
                'name' => 'Tron',
                'chainId' => null, // Tron doesn't use chainId
                'symbol' => 'TRX',
                'rpc_url' => 'https://api.trongrid.io',
                'explorer_api' => 'https://api.trongrid.io',
                'explorer_url' => 'https://tronscan.org',
                'api_key' => $this->apiKeys['trongrid'],
                'trongrid_url' => 'https://api.trongrid.io',
                'supported_tokens' => ['USDT', 'USDC', 'USDD'],
                'token_contracts' => [
                    'USDT' => 'TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t',
                    'USDC' => 'TEkxiTehnzSmSe2XqrBj4w32RUN966rdz8'
                ],
                'min_confirmations' => 19,
                'avg_block_time' => 3 // seconds
            ],
            'solana' => [
                'name' => 'Solana',
                'chainId' => null,
                'symbol' => 'SOL',
                'rpc_url' => 'https://api.mainnet-beta.solana.com',
                'explorer_api' => 'https://api.solscan.io',
                'explorer_url' => 'https://solscan.io',
                'api_key' => $this->apiKeys['solscan'],
                'supported_tokens' => ['USDT', 'USDC', 'RAY', 'SRM'],
                'token_contracts' => [
                    'USDT' => 'Es9vMFrzaCERmJfrF4H2FYD4KCoNkY11McCe8BenwNYB',
                    'USDC' => 'EPjFWdd5AufqSSqeM2qN1xzybapC8G4wEGGkZwyTDt1v'
                ],
                'min_confirmations' => 32,
                'avg_block_time' => 0.4 // seconds
            ]
        ];
    }
    
    /**
     * Get network configuration
     */
    public function getNetworkConfig($network) {
        return $this->networkConfigs[$network] ?? null;
    }
    
    /**
     * Get supported networks
     */
    public function getSupportedNetworks() {
        return array_keys($this->networkConfigs);
    }
    
    /**
     * Get current gas prices for Ethereum-based networks
     */
    public function getGasPrices($network) {
        $config = $this->getNetworkConfig($network);
        if (!$config || !isset($config['gas_tracker'])) {
            return ['success' => false, 'error' => 'Network not supported for gas tracking'];
        }
        
        try {
            $response = $this->makeHTTPRequest($config['gas_tracker']);
            
            if ($response['success']) {
                $data = json_decode($response['data'], true);
                
                if ($data['status'] === '1') {
                    return [
                        'success' => true,
                        'slow' => intval($data['result']['SafeGasPrice']) * 1e9, // Convert to wei
                        'standard' => intval($data['result']['ProposeGasPrice']) * 1e9,
                        'fast' => intval($data['result']['FastGasPrice']) * 1e9,
                        'network' => $network
                    ];
                } else {
                    return ['success' => false, 'error' => 'Gas tracker API error: ' . $data['message']];
                }
            } else {
                return ['success' => false, 'error' => 'Failed to fetch gas prices: ' . $response['error']];
            }
        } catch (Exception $e) {
            error_log("Gas price fetch error for $network: " . $e->getMessage());
            return ['success' => false, 'error' => 'Exception: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get account balance for Ethereum-based networks
     */
    public function getAccountBalance($network, $address, $token = null) {
        $config = $this->getNetworkConfig($network);
        if (!$config) {
            return ['success' => false, 'error' => 'Network not supported'];
        }
        
        try {
            if ($token && isset($config['token_contracts'][$token])) {
                // Get ERC-20 token balance
                return $this->getTokenBalance($network, $address, $config['token_contracts'][$token]);
            } else {
                // Get native token balance
                return $this->getNativeBalance($network, $address);
            }
        } catch (Exception $e) {
            error_log("Balance fetch error for $network $address: " . $e->getMessage());
            
            // Handle specific error types
            if (strpos($e->getMessage(), 'blocked') !== false || strpos($e->getMessage(), 'Cloudflare') !== false) {
                return ['success' => false, 'error' => 'API temporarily blocked - please try again later'];
            } elseif (strpos($e->getMessage(), 'bcdiv') !== false) {
                return ['success' => false, 'error' => 'Invalid balance format received'];
            } else {
                return ['success' => false, 'error' => 'Failed to fetch balance: ' . $e->getMessage()];
            }
        }
    }
    
    /**
     * Get native token balance
     */
    private function getNativeBalance($network, $address) {
        $config = $this->getNetworkConfig($network);
        
        if ($network === 'tron') {
            return $this->getTronBalance($address);
        } elseif ($network === 'solana') {
            return $this->getSolanaBalance($address);
        } else {
            // Try multiple approaches for Ethereum-based networks
            
            // Approach 1: Try V2 API with chainid parameter
            if (isset($config['explorer_api']) && strpos($config['explorer_api'], 'v2') !== false) {
                $url = $config['explorer_api'] . '?' . http_build_query([
                    'module' => 'account',
                    'action' => 'balance',
                    'address' => $address,
                    'tag' => 'latest',
                    'chainid' => $config['chainId'],
                    'apikey' => $config['api_key']
                ]);
                
                $response = $this->makeHTTPRequest($url);
                
                if ($response['success']) {
                    $data = json_decode($response['data'], true);
                    
                    if ($data['status'] === '1') {
                        $balance = bcdiv($data['result'], '1000000000000000000', 18); // Convert from wei
                        return [
                            'success' => true,
                            'balance' => $balance,
                            'currency' => $config['symbol'],
                            'network' => $network
                        ];
                    }
                }
            }
            
            // Approach 2: Try Alchemy API for Ethereum
            if ($network === 'ethereum' && isset($config['alchemy_url'])) {
                $payload = json_encode([
                    'jsonrpc' => '2.0',
                    'method' => 'eth_getBalance',
                    'params' => [$address, 'latest'],
                    'id' => 1
                ]);
                
                $response = $this->makeHTTPRequest($config['alchemy_url'], ['Content-Type: application/json'], $payload);
                
                if ($response['success']) {
                    $data = json_decode($response['data'], true);
                    
                    if (isset($data['result']) && !isset($data['error'])) {
                        $balance = bcdiv($data['result'], '1000000000000000000', 18); // Convert from wei
                        return [
                            'success' => true,
                            'balance' => $balance,
                            'currency' => $config['symbol'],
                            'network' => $network
                        ];
                    }
                }
            }
            
            // Approach 3: Try public RPC endpoints
            if (isset($config['rpc_url'])) {
                $payload = json_encode([
                    'jsonrpc' => '2.0',
                    'method' => 'eth_getBalance',
                    'params' => [$address, 'latest'],
                    'id' => 1
                ]);
                
                $response = $this->makeHTTPRequest($config['rpc_url'], ['Content-Type: application/json'], $payload);
                
                if ($response['success']) {
                    $data = json_decode($response['data'], true);
                    
                    if (isset($data['result']) && !isset($data['error'])) {
                        // Convert hex result to decimal if needed
                        $result = $data['result'];
                        if (strpos($result, '0x') === 0) {
                            $result = hexdec($result);
                        }
                        $balance = bcdiv((string)$result, '1000000000000000000', 18); // Convert from wei
                        return [
                            'success' => true,
                            'balance' => $balance,
                            'currency' => $config['symbol'],
                            'network' => $network
                        ];
                    }
                }
            }
            
            return ['success' => false, 'error' => 'All balance fetch methods failed'];
        }
    }
    
    /**
     * Get ERC-20 token balance
     */
    private function getTokenBalance($network, $address, $contractAddress) {
        $config = $this->getNetworkConfig($network);
        
        if ($network === 'tron') {
            return $this->getTronTokenBalance($address, $contractAddress);
        }
        
        // Use Etherscan API v2 format for token balance
        $params = [
            'module' => 'account',
            'action' => 'tokenbalance',
            'contractaddress' => $contractAddress,
            'address' => $address,
            'tag' => 'latest',
            'apikey' => $config['api_key']
        ];
        
        // Add chainid for v2 API if available
        if (isset($config['chainId'])) {
            $params['chainid'] = $config['chainId'];
        }
        
        $url = $config['explorer_api'] . '?' . http_build_query($params);
        
        $response = $this->makeHTTPRequest($url);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if ($data['status'] === '1') {
                $balance = bcdiv($data['result'], '1000000', 6); // USDT/USDC have 6 decimals
                return [
                    'success' => true,
                    'balance' => $balance,
                    'contract' => $contractAddress,
                    'network' => $network
                ];
            } else {
                return ['success' => false, 'error' => 'API error: ' . $data['message']];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Tron balance
     */
    private function getTronBalance($address) {
        $config = $this->getNetworkConfig('tron');
        $url = $config['trongrid_url'] . '/v1/accounts/' . $address;
        
        $response = $this->makeHTTPRequest($url, [
            'TRON-PRO-API-KEY' => $this->apiKeys['trongrid']
        ]);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if (isset($data['data'][0]['balance'])) {
                $balance = bcdiv($data['data'][0]['balance'], '1000000', 6); // TRX has 6 decimals
                return [
                    'success' => true,
                    'balance' => $balance,
                    'currency' => 'TRX',
                    'network' => 'tron'
                ];
            } else {
                return ['success' => false, 'error' => 'Invalid response format'];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Tron token balance (TRC20)
     */
    private function getTronTokenBalance($address, $contractAddress) {
        try {
            // Use our TronWeb Node.js service for real blockchain data
            $nodeServiceUrl = 'http://localhost:3001/balance?address=' . urlencode($address);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $nodeServiceUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
            
            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($result === false) {
                throw new Exception("TronWeb service call failed: " . $error);
            }
            
            $response = json_decode($result, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Invalid JSON response from TronWeb service: " . $result);
            }
            
            if (!$response['success']) {
                throw new Exception("TronWeb service error: " . $response['error']);
            }
            
            // Return USDT balance (assuming USDT TRC20)
            return [
                'success' => true,
                'balance' => $response['balance'],
                'currency' => 'USDT',
                'network' => 'tron',
                'contract' => $contractAddress
            ];
            
        } catch (Exception $e) {
            error_log("TronWeb service error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get Tron balance: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get Solana balance
     */
    private function getSolanaBalance($address) {
        $url = 'https://api.mainnet-beta.solana.com';
        
        $payload = [
            'jsonrpc' => '2.0',
            'id' => 1,
            'method' => 'getBalance',
            'params' => [$address]
        ];
        
        $response = $this->makeHTTPRequest($url, [], json_encode($payload));
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if (isset($data['result']['value'])) {
                $balance = bcdiv($data['result']['value'], '1000000000', 9); // SOL has 9 decimals
                return [
                    'success' => true,
                    'balance' => $balance,
                    'currency' => 'SOL',
                    'network' => 'solana'
                ];
            } else {
                return ['success' => false, 'error' => 'Invalid response format'];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get transaction history for an address
     */
    public function getTransactionHistory($network, $address, $limit = 100) {
        $config = $this->getNetworkConfig($network);
        if (!$config) {
            return ['success' => false, 'error' => 'Network not supported'];
        }
        
        try {
            if ($network === 'tron') {
                return $this->getTronTransactions($address, $limit);
            } elseif ($network === 'solana') {
                return $this->getSolanaTransactions($address, $limit);
            } else {
                return $this->getEthereumTransactions($network, $address, $limit);
            }
        } catch (Exception $e) {
            error_log("Transaction history error for $network $address: " . $e->getMessage());
            return ['success' => false, 'error' => 'Exception: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get Ethereum-based network transactions
     */
    private function getEthereumTransactions($network, $address, $limit) {
        $config = $this->getNetworkConfig($network);
        
        $url = $config['explorer_api'] . '?' . http_build_query([
            'module' => 'account',
            'action' => 'txlist',
            'address' => $address,
            'startblock' => 0,
            'endblock' => 99999999,
            'page' => 1,
            'offset' => $limit,
            'sort' => 'desc',
            'apikey' => $config['api_key']
        ]);
        
        $response = $this->makeHTTPRequest($url);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if ($data['status'] === '1') {
                $transactions = [];
                foreach ($data['result'] as $tx) {
                    $transactions[] = [
                        'hash' => $tx['hash'],
                        'from' => $tx['from'],
                        'to' => $tx['to'],
                        'value' => bcdiv($tx['value'], '1000000000000000000', 18),
                        'gas_used' => $tx['gasUsed'],
                        'gas_price' => $tx['gasPrice'],
                        'timestamp' => $tx['timeStamp'],
                        'block_number' => $tx['blockNumber'],
                        'confirmations' => $tx['confirmations'],
                        'status' => $tx['isError'] === '0' ? 'success' : 'failed'
                    ];
                }
                
                return [
                    'success' => true,
                    'transactions' => $transactions,
                    'network' => $network
                ];
            } else {
                return ['success' => false, 'error' => 'API error: ' . $data['message']];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Tron transactions
     */
    private function getTronTransactions($address, $limit) {
        $url = 'https://api.trongrid.io/v1/accounts/' . $address . '/transactions/trc20?limit=' . $limit;
        
        $response = $this->makeHTTPRequest($url, [
            'TRON-PRO-API-KEY' => $this->apiKeys['tronscan']
        ]);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            $transactions = [];
            if (isset($data['data'])) {
                foreach ($data['data'] as $tx) {
                    $transactions[] = [
                        'hash' => $tx['transaction_id'],
                        'from' => $tx['from'],
                        'to' => $tx['to'],
                        'value' => bcdiv($tx['value'], '1000000', 6),
                        'token' => $tx['token_info']['symbol'],
                        'timestamp' => $tx['block_timestamp'],
                        'block_number' => $tx['block_number'],
                        'status' => 'success'
                    ];
                }
            }
            
            return [
                'success' => true,
                'transactions' => $transactions,
                'network' => 'tron'
            ];
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Solana transactions
     */
    private function getSolanaTransactions($address, $limit) {
        $url = 'https://api.mainnet-beta.solana.com';
        
        $payload = [
            'jsonrpc' => '2.0',
            'id' => 1,
            'method' => 'getSignaturesForAddress',
            'params' => [$address, ['limit' => $limit]]
        ];
        
        $response = $this->makeHTTPRequest($url, [], json_encode($payload));
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            $transactions = [];
            if (isset($data['result'])) {
                foreach ($data['result'] as $tx) {
                    $transactions[] = [
                        'hash' => $tx['signature'],
                        'slot' => $tx['slot'],
                        'timestamp' => $tx['blockTime'] * 1000, // Convert to milliseconds
                        'status' => $tx['err'] ? 'failed' : 'success',
                        'confirmation_status' => $tx['confirmationStatus']
                    ];
                }
            }
            
            return [
                'success' => true,
                'transactions' => $transactions,
                'network' => 'solana'
            ];
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get transaction details by hash
     */
    public function getTransactionDetails($network, $txHash) {
        $config = $this->getNetworkConfig($network);
        if (!$config) {
            return ['success' => false, 'error' => 'Network not supported'];
        }
        
        try {
            if ($network === 'tron') {
                return $this->getTronTransactionDetails($txHash);
            } elseif ($network === 'solana') {
                return $this->getSolanaTransactionDetails($txHash);
            } else {
                return $this->getEthereumTransactionDetails($network, $txHash);
            }
        } catch (Exception $e) {
            error_log("Transaction details error for $network $txHash: " . $e->getMessage());
            return ['success' => false, 'error' => 'Exception: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get Ethereum transaction details
     */
    private function getEthereumTransactionDetails($network, $txHash) {
        $config = $this->getNetworkConfig($network);
        
        $url = $config['explorer_api'] . '?' . http_build_query([
            'module' => 'proxy',
            'action' => 'eth_getTransactionByHash',
            'txhash' => $txHash,
            'apikey' => $config['api_key']
        ]);
        
        $response = $this->makeHTTPRequest($url);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if (isset($data['result']) && $data['result'] !== null) {
                $tx = $data['result'];
                return [
                    'success' => true,
                    'transaction' => [
                        'hash' => $tx['hash'],
                        'from' => $tx['from'],
                        'to' => $tx['to'],
                        'value' => hexdec($tx['value']) / 1e18,
                        'gas' => hexdec($tx['gas']),
                        'gas_price' => hexdec($tx['gasPrice']),
                        'nonce' => hexdec($tx['nonce']),
                        'block_number' => hexdec($tx['blockNumber']),
                        'network' => $network
                    ]
                ];
            } else {
                return ['success' => false, 'error' => 'Transaction not found'];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Tron transaction details
     */
    private function getTronTransactionDetails($txHash) {
        $url = 'https://api.trongrid.io/wallet/gettransactionbyid?value=' . $txHash;
        
        $response = $this->makeHTTPRequest($url);
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if (isset($data['txID'])) {
                return [
                    'success' => true,
                    'transaction' => [
                        'hash' => $data['txID'],
                        'from' => $data['raw_data']['contract'][0]['parameter']['value']['owner_address'],
                        'to' => $data['raw_data']['contract'][0]['parameter']['value']['to_address'],
                        'value' => $data['raw_data']['contract'][0]['parameter']['value']['amount'] / 1e6,
                        'block_number' => $data['blockNumber'],
                        'network' => 'tron'
                    ]
                ];
            } else {
                return ['success' => false, 'error' => 'Transaction not found'];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Get Solana transaction details
     */
    private function getSolanaTransactionDetails($txHash) {
        $url = 'https://api.mainnet-beta.solana.com';
        
        $payload = [
            'jsonrpc' => '2.0',
            'id' => 1,
            'method' => 'getTransaction',
            'params' => [$txHash, ['encoding' => 'json']]
        ];
        
        $response = $this->makeHTTPRequest($url, [], json_encode($payload));
        
        if ($response['success']) {
            $data = json_decode($response['data'], true);
            
            if (isset($data['result'])) {
                $tx = $data['result'];
                return [
                    'success' => true,
                    'transaction' => [
                        'hash' => $txHash,
                        'slot' => $tx['slot'],
                        'fee' => $tx['meta']['fee'],
                        'status' => $tx['meta']['err'] ? 'failed' : 'success',
                        'network' => 'solana'
                    ]
                ];
            } else {
                return ['success' => false, 'error' => 'Transaction not found'];
            }
        } else {
            return ['success' => false, 'error' => 'Request failed: ' . $response['error']];
        }
    }
    
    /**
     * Validate address format for a network
     */
    public function validateAddress($network, $address) {
        $config = $this->getNetworkConfig($network);
        if (!$config) {
            return ['success' => false, 'error' => 'Network not supported'];
        }
        
        switch ($network) {
            case 'ethereum':
            case 'bsc':
            case 'polygon':
            case 'base':
                // Ethereum-style addresses (0x followed by 40 hex characters)
                return [
                    'success' => preg_match('/^0x[a-fA-F0-9]{40}$/', $address),
                    'message' => 'Ethereum-style address required'
                ];
                
            case 'tron':
                // Tron addresses (T followed by 33 characters)
                return [
                    'success' => preg_match('/^T[A-Za-z1-9]{33}$/', $address),
                    'message' => 'Tron address required (starts with T)'
                ];
                
            case 'solana':
                // Solana addresses (base58, 32-44 characters)
                return [
                    'success' => preg_match('/^[1-9A-HJ-NP-Za-km-z]{32,44}$/', $address),
                    'message' => 'Solana address required'
                ];
                
            default:
                return ['success' => false, 'error' => 'Unknown network'];
        }
    }
    
    /**
     * Make HTTP request with error handling
     */
    private function makeHTTPRequest($url, $headers = [], $data = null) {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_USERAGENT => 'YELLOW-Crypto-Exchange/1.0'
        ]);
        
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, array_map(function($key, $value) {
                return "$key: $value";
            }, array_keys($headers), $headers));
        }
        
        if ($data !== null) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array_merge(
                curl_getinfo($ch, CURLINFO_HEADER_OUT) ?: [],
                ['Content-Type: application/json']
            ));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'error' => $error];
        }
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'data' => $response, 'http_code' => $httpCode];
        } else {
            // Check for API blocking (Cloudflare, etc.)
            if (strpos($response, 'Cloudflare') !== false || strpos($response, 'blocked') !== false) {
                return ['success' => false, 'error' => 'API temporarily blocked by security service'];
            } elseif ($httpCode === 403) {
                return ['success' => false, 'error' => 'Access forbidden - API key may be invalid or rate limited'];
            } elseif ($httpCode === 429) {
                return ['success' => false, 'error' => 'Rate limit exceeded - please try again later'];
            } else {
                return ['success' => false, 'error' => "HTTP $httpCode: " . substr($response, 0, 200)];
            }
        }
    }
    
    /**
     * Get network status and health
     */
    public function getNetworkStatus($network) {
        $config = $this->getNetworkConfig($network);
        if (!$config) {
            return ['success' => false, 'error' => 'Network not supported'];
        }
        
        try {
            // Test RPC connectivity
            $rpcResponse = $this->testRPCConnection($network);
            $apiResponse = $this->testAPIConnection($network);
            
            // Consider network online if either RPC or API is working
            $isOnline = $rpcResponse || $apiResponse;
            
            return [
                'success' => $isOnline,
                'network' => $network,
                'rpc_status' => $rpcResponse,
                'api_status' => $apiResponse,
                'overall_status' => $isOnline,
                'config' => [
                    'name' => $config['name'],
                    'chain_id' => $config['chainId'],
                    'symbol' => $config['symbol'],
                    'min_confirmations' => $config['min_confirmations'],
                    'avg_block_time' => $config['avg_block_time']
                ]
            ];
        } catch (Exception $e) {
            error_log("Network status error for $network: " . $e->getMessage());
            return ['success' => false, 'error' => 'Exception: ' . $e->getMessage()];
        }
    }
    
    /**
     * Test RPC connection
     */
    private function testRPCConnection($network) {
        $config = $this->getNetworkConfig($network);
        
        if ($network === 'solana') {
            $payload = [
                'jsonrpc' => '2.0',
                'id' => 1,
                'method' => 'getHealth'
            ];
            
            $response = $this->makeHTTPRequest($config['rpc_url'], [], json_encode($payload));
            return $response['success'] && $response['http_code'] === 200;
        } elseif ($network === 'tron') {
            // For Tron, test with a simple API call to get latest block
            $url = $config['trongrid_url'] . '/v1/blocks/latest';
            $response = $this->makeHTTPRequest($url, [
                'TRON-PRO-API-KEY' => $this->apiKeys['trongrid']
            ]);
            return $response['success'] && $response['http_code'] === 200;
        } else {
            // For Ethereum-based networks, we'll test with a simple eth_blockNumber call
            $payload = [
                'jsonrpc' => '2.0',
                'id' => 1,
                'method' => 'eth_blockNumber',
                'params' => []
            ];
            
            $response = $this->makeHTTPRequest($config['rpc_url'], [], json_encode($payload));
            return $response['success'] && $response['http_code'] === 200;
        }
    }
    
    /**
     * Test API connection
     */
    private function testAPIConnection($network) {
        $config = $this->getNetworkConfig($network);
        
        if ($network === 'tron') {
            $url = $config['trongrid_url'] . '/wallet/getnowblock';
            $response = $this->makeHTTPRequest($url, [
                'TRON-PRO-API-KEY' => $this->apiKeys['trongrid']
            ]);
        } elseif ($network === 'solana') {
            // Solana doesn't have a separate API, use RPC
            return true;
        } else {
            $url = $config['explorer_api'] . '?' . http_build_query([
                'module' => 'proxy',
                'action' => 'eth_blockNumber',
                'apikey' => $config['api_key']
            ]);
            
            $response = $this->makeHTTPRequest($url);
        }
        
        return $response['success'] && $response['http_code'] === 200;
    }
}
?>
