<?php
/**
 * Automated Blockchain Balance Sync System
 * Automatically syncs system wallet balances with real blockchain data
 */

require_once __DIR__ . '/../config/database.php';

class BlockchainBalanceSync {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    /**
     * Get TRON USDT balance from blockchain
     */
    private function getTronUsdtBalance($address) {
        try {
            // Get TRC20 token balance for USDT
            $usdtContractAddress = 'TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t'; // USDT TRC20 contract
            
            $url = "https://api.trongrid.io/v1/accounts/{$address}/transactions/trc20?limit=1&contract_address={$usdtContractAddress}";
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if (!$data || !$data['success'] || empty($data['data'])) {
                return 0; // No USDT transactions found
            }
            
            // Calculate balance from transaction history
            // This is a simplified approach - in production you'd use proper balance API
            $balance = 0;
            foreach ($data['data'] as $tx) {
                if ($tx['to'] === $address) {
                    $balance += floatval($tx['value']) / 1000000; // Convert from smallest unit
                } elseif ($tx['from'] === $address) {
                    $balance -= floatval($tx['value']) / 1000000;
                }
            }
            
            return max(0, $balance); // Ensure non-negative balance
            
        } catch (Exception $e) {
            error_log("TRON balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Bitcoin balance from blockchain
     */
    private function getBitcoinBalance($address) {
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'method' => 'GET'
                ]
            ]);
            
            $url = "https://blockstream.info/api/address/{$address}";
            $response = file_get_contents($url, false, $context);
            
            if ($response === false) {
                return false;
            }
            
            $data = json_decode($response, true);
            
            if ($data && isset($data['chain_stats']['funded_txo_sum'])) {
                $satoshi = $data['chain_stats']['funded_txo_sum'] - $data['chain_stats']['spent_txo_sum'];
                return $satoshi / 100000000; // Convert from satoshi to BTC
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Bitcoin balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Ethereum/ERC20 balance from blockchain
     */
    private function getEthereumBalance($address, $tokenContract = null) {
        try {
            $apiKey = $_ENV['ETHERSCAN_API_KEY'] ?? 'YourEtherscanAPIKey';
            
            if ($tokenContract) {
                // ERC20 token balance
                $url = "https://api.etherscan.io/api?module=account&action=tokenbalance&contractaddress={$tokenContract}&address={$address}&tag=latest&apikey={$apiKey}";
            } else {
                // ETH balance
                $url = "https://api.etherscan.io/api?module=account&action=balance&address={$address}&tag=latest&apikey={$apiKey}";
            }
            
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'method' => 'GET'
                ]
            ]);
            
            $response = file_get_contents($url, false, $context);
            
            if ($response === false) {
                return false;
            }
            
            $data = json_decode($response, true);
            
            if ($data && $data['status'] === '1') {
                $decimals = $tokenContract ? 6 : 18; // USDT has 6 decimals, ETH has 18
                return floatval($data['result']) / pow(10, $decimals);
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Ethereum balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get BSC/BEP20 balance from blockchain
     */
    private function getBSCBalance($address, $tokenContract = null) {
        try {
            $apiKey = $_ENV['BSCSCAN_API_KEY'] ?? 'YourBSCScanAPIKey';
            
            if ($tokenContract) {
                // BEP20 token balance
                $url = "https://api.bscscan.com/api?module=account&action=tokenbalance&contractaddress={$tokenContract}&address={$address}&tag=latest&apikey={$apiKey}";
            } else {
                // BNB balance
                $url = "https://api.bscscan.com/api?module=account&action=balance&address={$address}&tag=latest&apikey={$apiKey}";
            }
            
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if ($data && $data['status'] === '1') {
                $decimals = $tokenContract ? 6 : 18; // USDT has 6 decimals, BNB has 18
                return floatval($data['result']) / pow(10, $decimals);
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("BSC balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Polygon balance from blockchain
     */
    private function getPolygonBalance($address, $tokenContract = null) {
        try {
            $apiKey = $_ENV['POLYGONSCAN_API_KEY'] ?? 'YourPolygonScanAPIKey';
            
            if ($tokenContract) {
                // ERC20 token balance on Polygon
                $url = "https://api.polygonscan.com/api?module=account&action=tokenbalance&contractaddress={$tokenContract}&address={$address}&tag=latest&apikey={$apiKey}";
            } else {
                // MATIC balance
                $url = "https://api.polygonscan.com/api?module=account&action=balance&address={$address}&tag=latest&apikey={$apiKey}";
            }
            
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if ($data && $data['status'] === '1') {
                $decimals = $tokenContract ? 6 : 18; // USDT has 6 decimals, MATIC has 18
                return floatval($data['result']) / pow(10, $decimals);
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Polygon balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Solana balance from blockchain
     */
    private function getSolanaBalance($address, $tokenContract = null) {
        try {
            if ($tokenContract) {
                // SPL token balance
                $url = "https://api.mainnet-beta.solana.com";
                $data = [
                    'jsonrpc' => '2.0',
                    'id' => 1,
                    'method' => 'getTokenAccountsByOwner',
                    'params' => [
                        $address,
                        ['mint' => $tokenContract],
                        ['encoding' => 'jsonParsed']
                    ]
                ];
                
                $options = [
                    'http' => [
                        'header' => "Content-type: application/json\r\n",
                        'method' => 'POST',
                        'content' => json_encode($data)
                    ]
                ];
                
                $context = stream_context_create($options);
                $response = file_get_contents($url, false, $context);
                $result = json_decode($response, true);
                
                if ($result && isset($result['result']['value'][0]['account']['data']['parsed']['info']['tokenAmount']['uiAmount'])) {
                    return floatval($result['result']['value'][0]['account']['data']['parsed']['info']['tokenAmount']['uiAmount']);
                }
            } else {
                // SOL balance
                $url = "https://api.mainnet-beta.solana.com";
                $data = [
                    'jsonrpc' => '2.0',
                    'id' => 1,
                    'method' => 'getBalance',
                    'params' => [$address]
                ];
                
                $options = [
                    'http' => [
                        'header' => "Content-type: application/json\r\n",
                        'method' => 'POST',
                        'content' => json_encode($data)
                    ]
                ];
                
                $context = stream_context_create($options);
                $response = file_get_contents($url, false, $context);
                $result = json_decode($response, true);
                
                if ($result && isset($result['result']['value'])) {
                    return floatval($result['result']['value']) / 1000000000; // Convert from lamports to SOL
                }
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Solana balance sync error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Sync balance for a specific wallet
     */
    public function syncWalletBalance($walletId) {
        try {
            $conn = $this->db->getConnection();
            
            // Get wallet details
            $stmt = $conn->prepare("
                SELECT id, currency, network, address, balance 
                FROM crypto_wallets 
                WHERE id = ? AND user_id = 0
            ");
            $stmt->execute([$walletId]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$wallet) {
                return ['success' => false, 'error' => 'Wallet not found'];
            }
            
            $newBalance = false;
            $network = strtolower($wallet['network']);
            $currency = $wallet['currency'];
            
            // Sync based on network
            switch ($network) {
                case 'trc20':
                    if ($currency === 'USDT') {
                        $newBalance = $this->getTronUsdtBalance($wallet['address']);
                    }
                    break;
                    
                case 'erc20':
                    if ($currency === 'USDT') {
                        $usdtContract = '0xdAC17F958D2ee523a2206206994597C13D831ec7';
                        $newBalance = $this->getEthereumBalance($wallet['address'], $usdtContract);
                    } elseif ($currency === 'USDC') {
                        $usdcContract = '0xA0b86a33E6441b8C4C8C0e4b0b8C4C8C0e4b0b8C';
                        $newBalance = $this->getEthereumBalance($wallet['address'], $usdcContract);
                    } elseif ($currency === 'ETH') {
                        $newBalance = $this->getEthereumBalance($wallet['address']);
                    }
                    break;
                    
                case 'bep20':
                    if ($currency === 'USDT') {
                        $usdtContract = '0x55d398326f99059fF775485246999027B3197955';
                        $newBalance = $this->getBSCBalance($wallet['address'], $usdtContract);
                    }
                    break;
                    
                case 'polygon':
                    if ($currency === 'USDT') {
                        $usdtContract = '0xc2132D05D31c914a87C6611C10748AEb04B58e8F';
                        $newBalance = $this->getPolygonBalance($wallet['address'], $usdtContract);
                    }
                    break;
                    
                case 'btc':
                    if ($currency === 'BTC') {
                        $newBalance = $this->getBitcoinBalance($wallet['address']);
                    }
                    break;
                    
                case 'solana':
                    if ($currency === 'SOL') {
                        $newBalance = $this->getSolanaBalance($wallet['address']);
                    }
                    break;
                    
                default:
                    return ['success' => false, 'error' => 'Unsupported network: ' . $network];
            }
            
            if ($newBalance === false) {
                return ['success' => false, 'error' => 'Failed to fetch blockchain balance'];
            }
            
            // Update database if balance changed
            $oldBalance = floatval($wallet['balance']);
            if (abs($newBalance - $oldBalance) > 0.000001) { // Only update if significant change
                $stmt = $conn->prepare("
                    UPDATE crypto_wallets 
                    SET balance = ?, updated_at = CURRENT_TIMESTAMP 
                    WHERE id = ?
                ");
                $stmt->execute([$newBalance, $walletId]);
                
                return [
                    'success' => true,
                    'wallet_id' => $walletId,
                    'currency' => $currency,
                    'network' => $wallet['network'],
                    'old_balance' => $oldBalance,
                    'new_balance' => $newBalance,
                    'updated' => true
                ];
            }
            
            return [
                'success' => true,
                'wallet_id' => $walletId,
                'currency' => $currency,
                'network' => $wallet['network'],
                'balance' => $newBalance,
                'updated' => false
            ];
            
        } catch (Exception $e) {
            error_log("Wallet sync error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Sync all system wallet balances
     */
    public function syncAllBalances() {
        try {
            // Set execution time limit for web requests
            set_time_limit(60);
            
            $conn = $this->db->getConnection();
            
            // Get all system wallets
            $stmt = $conn->prepare("
                SELECT id, currency, network, address, balance 
                FROM crypto_wallets 
                WHERE user_id = 0
                ORDER BY currency, network
            ");
            $stmt->execute();
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $results = [];
            $updatedCount = 0;
            
            foreach ($wallets as $wallet) {
                $result = $this->syncWalletBalance($wallet['id']);
                $results[] = $result;
                
                if ($result['success'] && isset($result['updated']) && $result['updated']) {
                    $updatedCount++;
                }
                
                // Add small delay to prevent API rate limiting
                usleep(100000); // 0.1 second delay
            }
            
            return [
                'success' => true,
                'total_wallets' => count($wallets),
                'updated_wallets' => $updatedCount,
                'results' => $results
            ];
            
        } catch (Exception $e) {
            error_log("Sync all balances error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Quick sync for admin interface (only working networks)
     */
    public function quickSyncBalances() {
        try {
            set_time_limit(30);
            
            $conn = $this->db->getConnection();
            
            // Only sync networks that are known to work
            $stmt = $conn->prepare("
                SELECT id, currency, network, address, balance 
                FROM crypto_wallets 
                WHERE user_id = 0 
                AND network IN ('TRC20', 'BTC', 'solana')
                ORDER BY currency, network
            ");
            $stmt->execute();
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $results = [];
            $updatedCount = 0;
            
            foreach ($wallets as $wallet) {
                $result = $this->syncWalletBalance($wallet['id']);
                $results[] = $result;
                
                if ($result['success'] && isset($result['updated']) && $result['updated']) {
                    $updatedCount++;
                }
            }
            
            return [
                'success' => true,
                'total_wallets' => count($wallets),
                'updated_wallets' => $updatedCount,
                'results' => $results,
                'note' => 'Quick sync - only synced working networks (TRC20, BTC, Solana)'
            ];
            
        } catch (Exception $e) {
            error_log("Quick sync balances error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Create a cron job entry for automated syncing
     */
    public function createCronJob() {
        $cronCommand = "*/5 * * * * cd " . __DIR__ . " && php -f sync_balances_cron.php >> sync_balances.log 2>&1";
        
        // This would typically be added to crontab
        // For now, we'll just return the command
        return $cronCommand;
    }
}

// CLI usage
if (php_sapi_name() === 'cli') {
    $sync = new BlockchainBalanceSync();
    
    if (isset($argv[1]) && is_numeric($argv[1])) {
        // Sync specific wallet
        $result = $sync->syncWalletBalance($argv[1]);
        echo json_encode($result, JSON_PRETTY_PRINT) . "\n";
    } else {
        // Sync all wallets
        $result = $sync->syncAllBalances();
        echo "Blockchain Balance Sync Results:\n";
        echo "================================\n";
        echo "Total Wallets: " . $result['total_wallets'] . "\n";
        echo "Updated Wallets: " . $result['updated_wallets'] . "\n\n";
        
        foreach ($result['results'] as $walletResult) {
            if ($walletResult['success']) {
                if (isset($walletResult['updated']) && $walletResult['updated']) {
                    echo "✅ {$walletResult['currency']} {$walletResult['network']}: ";
                    echo "{$walletResult['old_balance']} → {$walletResult['new_balance']}\n";
                } else {
                    echo "ℹ️  {$walletResult['currency']} {$walletResult['network']}: ";
                    echo "{$walletResult['balance']} (no change)\n";
                }
            } else {
                echo "❌ {$walletResult['currency']} {$walletResult['network']}: ";
                echo "Error - {$walletResult['error']}\n";
            }
        }
    }
}
