<?php
/**
 * Crypto Prices API
 * Fetches real-time cryptocurrency prices from CoinGecko API
 */

// Only set headers if running in web context
if (isset($_SERVER['REQUEST_METHOD'])) {
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');

    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        exit(0);
    }
}

class CryptoPricesAPI {
    private $baseUrl = 'https://api.coingecko.com/api/v3';
    private $cache = [];
    private $cacheTimeout = 600; // 10 minutes cache (increased to avoid rate limits)
    private $conn;
    private $cacheFile;
    
    public function __construct($dbConnection = null) {
        $this->conn = $dbConnection;
        // Initialize cache file
        $this->cacheFile = __DIR__ . '/../tmp/price_cache.json';
        $this->loadCache();
    }
    
    /**
     * Load cached prices
     */
    private function loadCache() {
        if (file_exists($this->cacheFile)) {
            $data = json_decode(file_get_contents($this->cacheFile), true);
            if ($data && isset($data['timestamp'])) {
                // Check if cache is still valid
                if (time() - $data['timestamp'] < $this->cacheTimeout) {
                    $this->cache = $data['prices'] ?? [];
                }
            }
        }
    }
    
    /**
     * Save prices to cache
     */
    private function saveCache($prices) {
        $data = [
            'timestamp' => time(),
            'prices' => $prices
        ];
        
        // Ensure tmp directory exists
        $tmpDir = dirname($this->cacheFile);
        if (!is_dir($tmpDir)) {
            mkdir($tmpDir, 0755, true);
        }
        
        file_put_contents($this->cacheFile, json_encode($data));
    }
    
    /**
     * Fetch current cryptocurrency prices
     */
    public function getCurrentPrices() {
        try {
            // Return cached prices if available and recent
            if (!empty($this->cache)) {
                error_log("Returning cached prices to avoid rate limiting");
                return [
                    'success' => true,
                    'prices' => $this->cache,
                    'timestamp' => time(),
                    'cached' => true
                ];
            }
            
            // CoinGecko API endpoint for current prices
            $url = $this->baseUrl . '/simple/price?' . http_build_query([
                'ids' => 'bitcoin,ethereum,binancecoin,tether,usd-coin,solana,matic-network,tron',
                'vs_currencies' => 'usd',
                'include_24hr_change' => 'true'
            ]);
            
            $response = $this->makeHTTPRequest($url);
            
            if ($response['success']) {
                $data = json_decode($response['data'], true);
                
                // Map CoinGecko IDs to our symbols
                $priceMap = [
                    'bitcoin' => 'BTC',
                    'ethereum' => 'ETH',
                    'binancecoin' => 'BNB',
                    'tether' => 'USDT',
                    'usd-coin' => 'USDC',
                    'solana' => 'SOL',
                    'matic-network' => 'MATIC',
                    'tron' => 'TRX'
                ];
                
                $formattedPrices = [];
                foreach ($priceMap as $coingeckoId => $symbol) {
                    if (isset($data[$coingeckoId])) {
                        $formattedPrices[$symbol] = [
                            'price_usd' => $data[$coingeckoId]['usd'],
                            'price_change_24h' => $data[$coingeckoId]['usd_24h_change'] ?? 0,
                            'symbol' => $symbol,
                            'timestamp' => time()
                        ];
                    }
                }
                
                // Cache the prices
                $this->saveCache($formattedPrices);
                
                return [
                    'success' => true,
                    'prices' => $formattedPrices,
                    'timestamp' => time()
                ];
            } else {
                throw new Exception('Failed to fetch prices from CoinGecko API');
            }
            
        } catch (Exception $e) {
            error_log("Crypto prices API error: " . $e->getMessage());
            
            // Return cached prices if available
            if (!empty($this->cache)) {
                return [
                    'success' => true,
                    'prices' => $this->cache,
                    'timestamp' => time(),
                    'cached' => true
                ];
            }
            
            // Return fallback prices with success=true to keep app working
            $fallbackPrices = $this->getFallbackPrices();
            $this->cache = $fallbackPrices;
            $this->saveCache($fallbackPrices);
            
            return [
                'success' => true,
                'prices' => $fallbackPrices,
                'timestamp' => time(),
                'fallback' => true
            ];
        }
    }
    
    /**
     * Get fallback prices when API fails
     */
    private function getFallbackPrices() {
        return [
            'BTC' => ['price_usd' => 65000, 'price_change_24h' => 0, 'symbol' => 'BTC'],
            'ETH' => ['price_usd' => 3500, 'price_change_24h' => 0, 'symbol' => 'ETH'],
            'BNB' => ['price_usd' => 320, 'price_change_24h' => 0, 'symbol' => 'BNB'],
            'USDT' => ['price_usd' => 1.00, 'price_change_24h' => 0, 'symbol' => 'USDT'],
            'USDC' => ['price_usd' => 1.00, 'price_change_24h' => 0, 'symbol' => 'USDC'],
            'SOL' => ['price_usd' => 180, 'price_change_24h' => 0, 'symbol' => 'SOL'],
            'MATIC' => ['price_usd' => 0.85, 'price_change_24h' => 0, 'symbol' => 'MATIC'],
            'TRX' => ['price_usd' => 0.08, 'price_change_24h' => 0, 'symbol' => 'TRX']
        ];
    }
    
    /**
     * Convert crypto price to local currency
     */
    public function convertToLocalCurrency($cryptoPrice, $localCurrency = 'UGX') {
        // Exchange rates (you can integrate with a forex API for real rates)
        $exchangeRates = [
            'UGX' => 3700, // 1 USD = 3700 UGX (approximate)
            'KES' => 130,  // 1 USD = 130 KES (approximate)
            'TZS' => 2500, // 1 USD = 2500 TZS (approximate)
            'RWF' => 1200, // 1 USD = 1200 RWF (approximate)
            'CDF' => 2500, // 1 USD = 2500 CDF (approximate)
            'NGN' => 750,  // 1 USD = 750 NGN (approximate)
            'USD' => 1
        ];
        
        $rate = $exchangeRates[$localCurrency] ?? 1;
        return $cryptoPrice * $rate;
    }
    
    /**
     * Make HTTP request with retry logic
     */
    private function makeHTTPRequest($url, $retryCount = 0) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_USERAGENT, 'CryptoExchange/1.0');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'error' => $error];
        }
        
        // Handle rate limiting with exponential backoff
        if ($httpCode === 429 && $retryCount < 2) {
            $waitTime = pow(2, $retryCount) * 2; // 2s, 4s
            sleep($waitTime);
            return $this->makeHTTPRequest($url, $retryCount + 1);
        }
        
        if ($httpCode !== 200) {
            return ['success' => false, 'error' => "HTTP $httpCode"];
        }
        
        return ['success' => true, 'data' => $response];
    }
    
    /**
     * Get market overview data
     */
    public function getMarketOverview() {
        try {
            // Check cache first for market overview
            $cacheKey = 'market_overview';
            $marketCacheFile = __DIR__ . '/../tmp/market_cache.json';
            
            if (file_exists($marketCacheFile)) {
                $cacheData = json_decode(file_get_contents($marketCacheFile), true);
                if ($cacheData && isset($cacheData['timestamp'])) {
                    if (time() - $cacheData['timestamp'] < $this->cacheTimeout) {
                        error_log("Returning cached market overview to avoid rate limiting");
                        return $cacheData['data'];
                    }
                }
            }
            
            // Get current prices for major cryptocurrencies
            $cryptos = ['bitcoin', 'ethereum', 'tether', 'usd-coin', 'binancecoin', 'cardano', 'solana', 'matic-network', 'tron'];
            $url = $this->baseUrl . '/simple/price?' . http_build_query([
                'ids' => implode(',', $cryptos),
                'vs_currencies' => 'usd',
                'include_24hr_change' => 'true',
                'include_market_cap' => 'true',
                'include_24hr_vol' => 'true'
            ]);
            
            $response = $this->makeHTTPRequest($url);
            
            if (!$response['success']) {
                // Return cached data or fallback
                if (file_exists($marketCacheFile)) {
                    $cacheData = json_decode(file_get_contents($marketCacheFile), true);
                    if ($cacheData && isset($cacheData['data'])) {
                        error_log("API failed, returning stale cache");
                        return $cacheData['data'];
                    }
                }
                
                return $this->getFallbackMarketOverview();
            }
            
            $data = json_decode($response['data'], true);
            
            if (!$data) {
                return [
                    'success' => false,
                    'error' => 'Failed to parse API response'
                ];
            }
            
            $overview = [];
            
            // Process each cryptocurrency
            foreach ($data as $id => $info) {
                $symbol = strtoupper($this->getIdToSymbol($id));
                $overview[] = [
                    'id' => $id,
                    'symbol' => $symbol,
                    'name' => $this->getIdToName($id),
                    'price_usd' => $info['usd'],
                    'market_cap' => $info['usd_market_cap'] ?? 0,
                    'volume_24h' => $info['usd_24h_vol'] ?? 0,
                    'change_24h' => $info['usd_24h_change'] ?? 0,
                    'change_24h_percent' => round($info['usd_24h_change'] ?? 0, 2)
                ];
            }
            
            // Sort by market cap
            usort($overview, function($a, $b) {
                return $b['market_cap'] - $a['market_cap'];
            });
            
            $result = [
                'success' => true,
                'data' => [
                    'cryptocurrencies' => $overview,
                    'total_market_cap' => array_sum(array_column($overview, 'market_cap')),
                    'total_volume_24h' => array_sum(array_column($overview, 'volume_24h')),
                    'last_updated' => date('Y-m-d H:i:s')
                ]
            ];
            
            // Cache the result
            $marketCacheFile = __DIR__ . '/../tmp/market_cache.json';
            $tmpDir = dirname($marketCacheFile);
            if (!is_dir($tmpDir)) {
                mkdir($tmpDir, 0755, true);
            }
            file_put_contents($marketCacheFile, json_encode([
                'timestamp' => time(),
                'data' => $result
            ]));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Market overview error: " . $e->getMessage());
            
            // Return fallback market data
            return $this->getFallbackMarketOverview();
        }
    }
    
    /**
     * Get fallback market overview when API fails
     */
    private function getFallbackMarketOverview() {
        $fallbackData = [
            ['id' => 'bitcoin', 'symbol' => 'BTC', 'name' => 'Bitcoin', 'price_usd' => 67500, 'market_cap' => 1320000000000, 'volume_24h' => 28000000000, 'change_24h' => 2.5, 'change_24h_percent' => 2.5],
            ['id' => 'ethereum', 'symbol' => 'ETH', 'name' => 'Ethereum', 'price_usd' => 3800, 'market_cap' => 456000000000, 'volume_24h' => 15000000000, 'change_24h' => 1.8, 'change_24h_percent' => 1.8],
            ['id' => 'tether', 'symbol' => 'USDT', 'name' => 'Tether', 'price_usd' => 1.00, 'market_cap' => 95000000000, 'volume_24h' => 50000000000, 'change_24h' => 0.01, 'change_24h_percent' => 0.01],
            ['id' => 'binancecoin', 'symbol' => 'BNB', 'name' => 'BNB', 'price_usd' => 610, 'market_cap' => 89000000000, 'volume_24h' => 1800000000, 'change_24h' => 1.2, 'change_24h_percent' => 1.2],
            ['id' => 'solana', 'symbol' => 'SOL', 'name' => 'Solana', 'price_usd' => 185, 'market_cap' => 78000000000, 'volume_24h' => 3200000000, 'change_24h' => 3.5, 'change_24h_percent' => 3.5],
            ['id' => 'usd-coin', 'symbol' => 'USDC', 'name' => 'USD Coin', 'price_usd' => 1.00, 'market_cap' => 34000000000, 'volume_24h' => 6500000000, 'change_24h' => 0.0, 'change_24h_percent' => 0.0],
        ];
        
        return [
            'success' => true,
            'data' => [
                'cryptocurrencies' => $fallbackData,
                'total_market_cap' => array_sum(array_column($fallbackData, 'market_cap')),
                'total_volume_24h' => array_sum(array_column($fallbackData, 'volume_24h')),
                'last_updated' => date('Y-m-d H:i:s'),
                'fallback' => true
            ]
        ];
    }
    
    /**
     * Get detailed coin information
     */
    public function getCoinDetails($crypto, $currency = 'USD') {
        try {
            $coinId = $this->getSymbolToId(strtolower($crypto));
            
            if (!$coinId) {
                return [
                    'success' => false,
                    'error' => 'Cryptocurrency not found'
                ];
            }
            
            // Get detailed coin information
            $url = $this->baseUrl . "/coins/$coinId?" . http_build_query([
                'localization' => 'false',
                'tickers' => 'false',
                'market_data' => 'true',
                'community_data' => 'false',
                'developer_data' => 'false',
                'sparkline' => 'false'
            ]);
            
            $response = $this->makeHTTPRequest($url);
            
            if (!$response['success']) {
                return [
                    'success' => false,
                    'error' => $response['error']
                ];
            }
            
            $data = json_decode($response['data'], true);
            
            if (!$data) {
                return [
                    'success' => false,
                    'error' => 'Failed to parse API response'
                ];
            }
            
            $marketData = $data['market_data'];
            
            // Get price in requested currency
            $currencyLower = strtolower($currency);
            $price = $marketData['current_price'][$currencyLower] ?? $marketData['current_price']['usd'];
            
            $details = [
                'id' => $data['id'],
                'symbol' => strtoupper($data['symbol']),
                'name' => $data['name'],
                'description' => $data['description']['en'] ?? '',
                'image' => $data['image']['large'] ?? '',
                'price' => $price,
                'currency' => $currency,
                'market_cap' => $marketData['market_cap'][$currencyLower] ?? $marketData['market_cap']['usd'],
                'total_volume' => $marketData['total_volume'][$currencyLower] ?? $marketData['total_volume']['usd'],
                'price_change_24h' => $marketData['price_change_24h'],
                'price_change_percentage_24h' => round($marketData['price_change_percentage_24h'], 2),
                'price_change_percentage_7d' => round($marketData['price_change_percentage_7d_in_currency'][$currencyLower] ?? $marketData['price_change_percentage_7d_in_currency']['usd'], 2),
                'price_change_percentage_30d' => round($marketData['price_change_percentage_30d_in_currency'][$currencyLower] ?? $marketData['price_change_percentage_30d_in_currency']['usd'], 2),
                'market_cap_rank' => $marketData['market_cap_rank'],
                'circulating_supply' => $marketData['circulating_supply'],
                'total_supply' => $marketData['total_supply'],
                'max_supply' => $marketData['max_supply'],
                'ath' => $marketData['ath'][$currencyLower] ?? $marketData['ath']['usd'],
                'ath_change_percentage' => round($marketData['ath_change_percentage'][$currencyLower] ?? $marketData['ath_change_percentage']['usd'], 2),
                'atl' => $marketData['atl'][$currencyLower] ?? $marketData['atl']['usd'],
                'atl_change_percentage' => round($marketData['atl_change_percentage'][$currencyLower] ?? $marketData['atl_change_percentage']['usd'], 2),
                'last_updated' => $data['last_updated']
            ];
            
            return [
                'success' => true,
                'data' => $details
            ];
            
        } catch (Exception $e) {
            error_log("Coin details error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to fetch coin details'
            ];
        }
    }
    
    /**
     * Convert coin ID to symbol
     */
    private function getIdToSymbol($id) {
        $mapping = [
            'bitcoin' => 'BTC',
            'ethereum' => 'ETH',
            'tether' => 'USDT',
            'usd-coin' => 'USDC',
            'binancecoin' => 'BNB',
            'cardano' => 'ADA',
            'solana' => 'SOL',
            'matic-network' => 'MATIC',
            'tron' => 'TRX'
        ];
        return $mapping[$id] ?? strtoupper($id);
    }
    
    /**
     * Convert coin ID to name
     */
    private function getIdToName($id) {
        $mapping = [
            'bitcoin' => 'Bitcoin',
            'ethereum' => 'Ethereum',
            'tether' => 'Tether',
            'usd-coin' => 'USD Coin',
            'binancecoin' => 'Binance Coin',
            'cardano' => 'Cardano',
            'solana' => 'Solana',
            'matic-network' => 'Polygon',
            'tron' => 'TRON'
        ];
        return $mapping[$id] ?? ucfirst($id);
    }
    
    /**
     * Convert symbol to coin ID
     */
    private function getSymbolToId($symbol) {
        $mapping = [
            'btc' => 'bitcoin',
            'eth' => 'ethereum',
            'usdt' => 'tether',
            'usdc' => 'usd-coin',
            'bnb' => 'binancecoin',
            'ada' => 'cardano',
            'sol' => 'solana',
            'matic' => 'matic-network',
            'trx' => 'tron'
        ];
        return $mapping[$symbol] ?? $symbol;
    }
}

// This file is now used as a class library through the API router
// Direct execution is disabled to prevent conflicts
?>
