<?php
/**
 * Automated Deposit Detection System
 * Monitors blockchain for incoming deposits and credits user accounts
 * Similar to how Binance detects and processes deposits
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/blockchain_apis.php';

class DepositMonitor {
    private $db;
    private $blockchainAPIs;
    private $minConfirmations = 6; // Minimum confirmations before crediting
    
    public function __construct() {
        $this->db = new Database();
        $this->blockchainAPIs = new BlockchainAPIs($this->db->getConnection());
    }
    
    /**
     * Main function to check all user wallets for deposits
     */
    public function checkAllDeposits() {
        $startTime = microtime(true);
        $results = [
            'success' => true,
            'checked' => 0,
            'deposits_found' => 0,
            'deposits_credited' => 0,
            'errors' => [],
            'deposits' => []
        ];
        
        try {
            error_log("🔍 Starting deposit detection scan...");
            
            // Get all user wallets that can receive deposits
            $wallets = $this->getUserWallets();
            $results['checked'] = count($wallets);
            
            error_log("📊 Found {$results['checked']} wallets to monitor");
            
            foreach ($wallets as $wallet) {
                try {
                    $depositResult = $this->checkWalletDeposits($wallet);
                    
                    if ($depositResult['deposits_found'] > 0) {
                        $results['deposits_found'] += $depositResult['deposits_found'];
                        $results['deposits_credited'] += $depositResult['deposits_credited'];
                        $results['deposits'] = array_merge($results['deposits'], $depositResult['deposits']);
                    }
                    
                } catch (Exception $e) {
                    $error = "Error checking wallet {$wallet['address']}: " . $e->getMessage();
                    error_log("❌ " . $error);
                    $results['errors'][] = $error;
                }
            }
            
            $duration = round(microtime(true) - $startTime, 2);
            error_log("✅ Deposit scan completed in {$duration}s. Found: {$results['deposits_found']}, Credited: {$results['deposits_credited']}");
            
        } catch (Exception $e) {
            $results['success'] = false;
            $results['error'] = $e->getMessage();
            error_log("❌ Deposit scan failed: " . $e->getMessage());
        }
        
        return $results;
    }
    
    /**
     * Get all user wallets that need monitoring
     */
    private function getUserWallets() {
        $conn = $this->db->getConnection();
        
        $stmt = $conn->prepare("
            SELECT 
                id, user_id, currency, network, address, balance, 
                last_deposit_check, last_deposit_block
            FROM crypto_wallets 
            WHERE user_id > 0 
            AND address IS NOT NULL 
            AND address != ''
            AND is_real_blockchain = 1
            ORDER BY user_id, network, currency
        ");
        
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Check a specific wallet for new deposits
     */
    private function checkWalletDeposits($wallet) {
        $result = [
            'deposits_found' => 0,
            'deposits_credited' => 0,
            'deposits' => []
        ];
        
        try {
            // Get current blockchain balance
            $blockchainBalance = $this->getBlockchainBalance($wallet['address'], $wallet['currency'], $wallet['network']);
            
            if ($blockchainBalance === null) {
                error_log("⚠️ Could not fetch balance for {$wallet['address']} ({$wallet['network']})");
                return $result;
            }
            
            $dbBalance = floatval($wallet['balance']);
            
            // Check if blockchain balance is greater than database balance
            if ($blockchainBalance > $dbBalance) {
                $depositAmount = $blockchainBalance - $dbBalance;
                
                error_log("💰 DEPOSIT DETECTED!");
                error_log("   Address: {$wallet['address']}");
                error_log("   Network: {$wallet['network']}");
                error_log("   Currency: {$wallet['currency']}");
                error_log("   Amount: {$depositAmount}");
                error_log("   Blockchain: {$blockchainBalance}, Database: {$dbBalance}");
                
                $result['deposits_found']++;
                
                // Credit the deposit
                $credited = $this->creditDeposit($wallet, $depositAmount, $blockchainBalance);
                
                if ($credited) {
                    $result['deposits_credited']++;
                    $result['deposits'][] = [
                        'user_id' => $wallet['user_id'],
                        'address' => $wallet['address'],
                        'currency' => $wallet['currency'],
                        'network' => $wallet['network'],
                        'amount' => $depositAmount,
                        'new_balance' => $blockchainBalance
                    ];
                }
            }
            
            // Update last check time
            $this->updateLastCheck($wallet['id']);
            
        } catch (Exception $e) {
            error_log("❌ Error checking wallet {$wallet['address']}: " . $e->getMessage());
            throw $e;
        }
        
        return $result;
    }
    
    /**
     * Get blockchain balance for an address
     */
    private function getBlockchainBalance($address, $currency, $network) {
        try {
            $network = strtolower($network);
            
            // Map database network names to API network names
            $networkMap = [
                'trc20' => 'tron',
                'erc20' => 'ethereum',
                'bep20' => 'bsc',
                'polygon' => 'polygon',
                'solana' => 'solana',
                'base' => 'base',
                'celo' => 'celo',
                'btc' => 'bitcoin'
            ];
            
            $apiNetwork = $networkMap[$network] ?? $network;
            
            // Use BlockchainAPIs unified method
            $token = ($currency === 'USDT' || $currency === 'USDC') ? $currency : null;
            $result = $this->blockchainAPIs->getAccountBalance($apiNetwork, $address, $token);
            
            // Handle response format
            if (is_array($result)) {
                if (isset($result['success']) && $result['success'] && isset($result['balance'])) {
                    return floatval($result['balance']);
                } elseif (isset($result['error'])) {
                    error_log("⚠️ API error for {$address}: " . $result['error']);
                    return null;
                }
            } elseif (is_numeric($result)) {
                return floatval($result);
            }
            
            return null;
            
        } catch (Exception $e) {
            error_log("❌ Error fetching blockchain balance for {$address} ({$network}): " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Credit deposit to user's account
     */
    private function creditDeposit($wallet, $depositAmount, $newBalance) {
        $conn = $this->db->getConnection();
        
        try {
            $conn->beginTransaction();
            
            // Update user's wallet balance
            $stmt = $conn->prepare("
                UPDATE crypto_wallets 
                SET balance = ?, 
                    updated_at = datetime('now', 'localtime')
                WHERE id = ?
            ");
            $stmt->execute([$newBalance, $wallet['id']]);
            
            // Record deposit transaction
            $txHash = 'deposit_' . $wallet['id'] . '_' . time() . '_' . uniqid();
            $stmt = $conn->prepare("
                INSERT INTO crypto_transactions (
                    user_id, wallet_id, type, currency, network, 
                    amount, address, to_address, tx_hash, status, created_at
                ) VALUES (?, ?, 'deposit', ?, ?, ?, ?, ?, ?, 'completed', datetime('now', 'localtime'))
            ");
            $stmt->execute([
                $wallet['user_id'],
                $wallet['id'],
                $wallet['currency'],
                $wallet['network'],
                $depositAmount,
                'external', // From external wallet
                $wallet['address'], // To user's deposit address
                $txHash
            ]);
            
            $transactionId = $conn->lastInsertId();
            
            // Add system wallet balance (for custodial accounting)
            // When user deposits, system's total holdings increase
            $this->updateSystemWalletBalance($wallet['currency'], $wallet['network'], $depositAmount);
            
            $conn->commit();
            
            error_log("✅ Deposit credited successfully!");
            error_log("   Transaction ID: {$transactionId}");
            error_log("   User ID: {$wallet['user_id']}");
            error_log("   Amount: {$depositAmount} {$wallet['currency']}");
            
            // TODO: Send notification to user
            $this->notifyUser($wallet['user_id'], $depositAmount, $wallet['currency'], $wallet['network']);
            
            return true;
            
        } catch (Exception $e) {
            $conn->rollBack();
            error_log("❌ Failed to credit deposit: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update system wallet balance for accounting
     */
    private function updateSystemWalletBalance($currency, $network, $amount) {
        $conn = $this->db->getConnection();
        
        try {
            // Get system wallet for this currency/network
            $stmt = $conn->prepare("
                SELECT id FROM crypto_wallets 
                WHERE user_id = 0 
                AND currency = ? 
                AND (network = ? OR network = ?)
                LIMIT 1
            ");
            $stmt->execute([$currency, $network, strtoupper($network)]);
            $systemWallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($systemWallet) {
                // Increase system wallet balance (accounting)
                $stmt = $conn->prepare("
                    UPDATE crypto_wallets 
                    SET balance = balance + ?, 
                        updated_at = datetime('now', 'localtime')
                    WHERE id = ?
                ");
                $stmt->execute([$amount, $systemWallet['id']]);
                
                error_log("📊 System wallet balance updated: +{$amount} {$currency}");
            }
            
        } catch (Exception $e) {
            error_log("⚠️ Could not update system wallet balance: " . $e->getMessage());
            // Non-critical, continue
        }
    }
    
    /**
     * Update last deposit check timestamp
     */
    private function updateLastCheck($walletId) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                UPDATE crypto_wallets 
                SET last_deposit_check = datetime('now', 'localtime') 
                WHERE id = ?
            ");
            $stmt->execute([$walletId]);
        } catch (Exception $e) {
            error_log("⚠️ Could not update last check time: " . $e->getMessage());
        }
    }
    
    /**
     * Notify user about deposit
     */
    private function notifyUser($userId, $amount, $currency, $network) {
        try {
            // TODO: Implement notification system
            // - Email notification
            // - Push notification
            // - In-app notification
            
            error_log("📧 Notification sent to user {$userId}: Deposit of {$amount} {$currency} on {$network}");
            
        } catch (Exception $e) {
            error_log("⚠️ Could not send notification: " . $e->getMessage());
        }
    }
    
    /**
     * Check deposits for a specific user
     */
    public function checkUserDeposits($userId) {
        $results = [
            'success' => true,
            'deposits_found' => 0,
            'deposits_credited' => 0,
            'deposits' => []
        ];
        
        try {
            $conn = $this->db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id, user_id, currency, network, address, balance, 
                    last_deposit_check, last_deposit_block
                FROM crypto_wallets 
                WHERE user_id = ? 
                AND address IS NOT NULL 
                AND address != ''
                AND is_real_blockchain = 1
            ");
            
            $stmt->execute([$userId]);
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($wallets as $wallet) {
                $depositResult = $this->checkWalletDeposits($wallet);
                
                if ($depositResult['deposits_found'] > 0) {
                    $results['deposits_found'] += $depositResult['deposits_found'];
                    $results['deposits_credited'] += $depositResult['deposits_credited'];
                    $results['deposits'] = array_merge($results['deposits'], $depositResult['deposits']);
                }
            }
            
        } catch (Exception $e) {
            $results['success'] = false;
            $results['error'] = $e->getMessage();
            error_log("❌ Error checking user deposits: " . $e->getMessage());
        }
        
        return $results;
    }
}

// If run directly from command line (not included)
if (php_sapi_name() === 'cli' && realpath($_SERVER['SCRIPT_FILENAME']) === __FILE__) {
    echo "🚀 Starting Deposit Monitor...\n";
    
    $monitor = new DepositMonitor();
    $results = $monitor->checkAllDeposits();
    
    echo "\n📊 DEPOSIT SCAN RESULTS:\n";
    echo "========================\n";
    echo "Wallets Checked: {$results['checked']}\n";
    echo "Deposits Found: {$results['deposits_found']}\n";
    echo "Deposits Credited: {$results['deposits_credited']}\n";
    
    if (count($results['deposits']) > 0) {
        echo "\n💰 DEPOSITS:\n";
        foreach ($results['deposits'] as $deposit) {
            echo "  - User {$deposit['user_id']}: {$deposit['amount']} {$deposit['currency']} ({$deposit['network']})\n";
            echo "    Address: {$deposit['address']}\n";
        }
    }
    
    if (count($results['errors']) > 0) {
        echo "\n❌ ERRORS:\n";
        foreach ($results['errors'] as $error) {
            echo "  - {$error}\n";
        }
    }
    
    echo "\n✅ Scan completed!\n";
}
