<?php
/**
 * Dynamic Pricing System
 * Automatically calculates buy/sell prices based on market prices + spreads
 * Similar to Yellow Card's approach
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/market_prices.php';

class DynamicPricingSystem {
    private $conn;
    private $marketPricesAPI;

    public function __construct($connection) {
        $this->conn = $connection;
        $this->marketPricesAPI = new MarketPricesAPI($connection);
    }

    /**
     * Get dynamic buy/sell prices for a crypto/currency pair
     * This is the main method that calculates prices like Yellow Card
     */
    public function getDynamicPrices($crypto, $currency) {
        try {
            // 1. Get spread configuration
            $spreadConfig = $this->getSpreadConfig($crypto, $currency);
            if (!$spreadConfig) {
                return [
                    'success' => false,
                    'error' => 'Spread configuration not found'
                ];
            }

            // 2. Get current market price with timeout
            $marketPrice = $this->getMarketPriceWithTimeout($crypto, $currency);
            if (!$marketPrice['success']) {
                return [
                    'success' => false,
                    'error' => 'Unable to fetch market price: ' . $marketPrice['error']
                ];
            }

            $basePrice = $marketPrice['price'];
            $spreadPercentage = $spreadConfig['spread_percentage'] / 100;

            // 3. Calculate buy/sell prices (Yellow Card approach)
            $buyPrice = $basePrice * (1 + $spreadPercentage);
            $sellPrice = $basePrice * (1 - $spreadPercentage);

            // 4. Apply min/max spread limits
            $minBuyPrice = $basePrice * (1 + $spreadConfig['min_spread'] / 100);
            $maxBuyPrice = $basePrice * (1 + $spreadConfig['max_spread'] / 100);
            $minSellPrice = $basePrice * (1 - $spreadConfig['max_spread'] / 100);
            $maxSellPrice = $basePrice * (1 - $spreadConfig['min_spread'] / 100);

            // Ensure prices are within limits
            $buyPrice = max($minBuyPrice, min($maxBuyPrice, $buyPrice));
            $sellPrice = max($minSellPrice, min($maxSellPrice, $sellPrice));

            // 5. Calculate effective spread
            $effectiveSpread = (($buyPrice - $sellPrice) / $basePrice) * 100;

            return [
                'success' => true,
                'crypto' => $crypto,
                'currency' => $currency,
                'market_price' => $basePrice,
                'market_price_usd' => $marketPrice['price_usd'],
                'exchange_rate' => $marketPrice['exchange_rate'],
                'buy_price' => round($buyPrice, 2),
                'sell_price' => round($sellPrice, 2),
                'spread_percentage' => $spreadConfig['spread_percentage'],
                'effective_spread' => round($effectiveSpread, 2),
                'source' => $marketPrice['source'],
                'last_updated' => $marketPrice['last_updated'] ?? date('Y-m-d H:i:s'),
                'spread_config' => $spreadConfig
            ];

        } catch (Exception $e) {
            error_log("Dynamic pricing error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Price calculation failed'
            ];
        }
    }

    /**
     * Get market price with timeout protection
     */
    private function getMarketPriceWithTimeout($crypto, $currency) {
        // Set a timeout for the market price request
        $timeout = 5; // 5 seconds timeout
        
        // Use pcntl_alarm if available (Unix systems)
        if (function_exists('pcntl_alarm')) {
            pcntl_alarm($timeout);
            pcntl_signal(SIGALRM, function() {
                throw new Exception('Market price request timed out');
            });
        }
        
        try {
            $result = $this->marketPricesAPI->getMarketPrice($crypto, $currency);
            
            // Cancel alarm if request completed
            if (function_exists('pcntl_alarm')) {
                pcntl_alarm(0);
            }
            
            return $result;
            
        } catch (Exception $e) {
            // Cancel alarm
            if (function_exists('pcntl_alarm')) {
                pcntl_alarm(0);
            }
            
            // Return fallback price if request fails
            return $this->getFallbackPrice($crypto, $currency);
        }
    }
    
    /**
     * Get fallback price when API requests fail
     */
    private function getFallbackPrice($crypto, $currency) {
        // Fallback prices in USD
        $fallbackPrices = [
            'BTC' => 65000,
            'ETH' => 3500,
            'BNB' => 320,
            'USDT' => 1.00,
            'USDC' => 1.00,
            'SOL' => 180,
            'MATIC' => 0.85,
            'TRX' => 0.08
        ];
        
        $priceUsd = $fallbackPrices[$crypto] ?? 1.00;
        
        // Convert to local currency (simplified conversion)
        $exchangeRates = [
            'UGX' => 3700,
            'KES' => 150,
            'NGN' => 800,
            'TZS' => 2500,
            'RWF' => 1200,
            'CDF' => 2500,
            'USD' => 1.00
        ];
        
        $rate = $exchangeRates[$currency] ?? 3700;
        $priceLocal = $priceUsd * $rate;
        
        return [
            'success' => true,
            'price' => $priceLocal,
            'price_usd' => $priceUsd,
            'exchange_rate' => $rate,
            'source' => 'fallback',
            'last_updated' => date('Y-m-d H:i:s')
        ];
    }

    /**
     * Get spread configuration for a crypto/currency pair
     */
    private function getSpreadConfig($crypto, $currency) {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM pricing_spreads 
                WHERE crypto = ? AND currency = ? AND is_active = 1
            ");
            $stmt->execute([$crypto, $currency]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Spread config fetch error: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Update spread configuration
     */
    public function updateSpreadConfig($crypto, $currency, $spreadPercentage, $minSpread = null, $maxSpread = null) {
        try {
            // Validate spread percentage
            if ($spreadPercentage < 0.1 || $spreadPercentage > 10) {
                return [
                    'success' => false,
                    'error' => 'Spread percentage must be between 0.1% and 10%'
                ];
            }

            // Set defaults for min/max if not provided
            if ($minSpread === null) {
                $minSpread = max(0.1, $spreadPercentage * 0.5);
            }
            if ($maxSpread === null) {
                $maxSpread = min(10, $spreadPercentage * 2);
            }

            $stmt = $this->conn->prepare("
                INSERT OR REPLACE INTO pricing_spreads 
                (crypto, currency, spread_percentage, min_spread, max_spread, updated_at)
                VALUES (?, ?, ?, ?, ?, datetime('now'))
            ");
            $stmt->execute([$crypto, $currency, $spreadPercentage, $minSpread, $maxSpread]);

            return [
                'success' => true,
                'message' => 'Spread configuration updated successfully'
            ];

        } catch (Exception $e) {
            error_log("Spread config update error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to update spread configuration'
            ];
        }
    }

    /**
     * Get all spread configurations
     */
    public function getAllSpreadConfigs() {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM pricing_spreads 
                WHERE is_active = 1
                ORDER BY crypto, currency
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Get all spreads error: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Calculate profit margin for a transaction
     */
    public function calculateProfitMargin($crypto, $currency, $transactionType, $amount) {
        try {
            $prices = $this->getDynamicPrices($crypto, $currency);
            if (!$prices['success']) {
                return [
                    'success' => false,
                    'error' => $prices['error']
                ];
            }

            $marketPrice = $prices['market_price'];
            $transactionPrice = $transactionType === 'buy' ? $prices['buy_price'] : $prices['sell_price'];
            
            // Calculate profit per unit
            $profitPerUnit = abs($transactionPrice - $marketPrice);
            $totalProfit = $profitPerUnit * $amount;
            
            // Calculate profit percentage
            $profitPercentage = ($profitPerUnit / $marketPrice) * 100;

            return [
                'success' => true,
                'transaction_type' => $transactionType,
                'amount' => $amount,
                'market_price' => $marketPrice,
                'transaction_price' => $transactionPrice,
                'profit_per_unit' => $profitPerUnit,
                'total_profit' => $totalProfit,
                'profit_percentage' => round($profitPercentage, 2)
            ];

        } catch (Exception $e) {
            error_log("Profit margin calculation error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Profit margin calculation failed'
            ];
        }
    }

    /**
     * Get pricing statistics for admin dashboard
     */
    public function getPricingStats() {
        try {
            $stats = [];

            // Total active spread configurations
            $stmt = $this->conn->prepare("SELECT COUNT(*) as count FROM pricing_spreads WHERE is_active = 1");
            $stmt->execute();
            $stats['total_configs'] = $stmt->fetch()['count'];

            // Average spread percentage
            $stmt = $this->conn->prepare("SELECT AVG(spread_percentage) as avg_spread FROM pricing_spreads WHERE is_active = 1");
            $stmt->execute();
            $stats['average_spread'] = round($stmt->fetch()['avg_spread'], 2);

            // Most traded pairs (based on spread configs)
            $stmt = $this->conn->prepare("
                SELECT crypto, COUNT(*) as pair_count 
                FROM pricing_spreads 
                WHERE is_active = 1 
                GROUP BY crypto 
                ORDER BY pair_count DESC 
                LIMIT 5
            ");
            $stmt->execute();
            $stats['top_cryptos'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Cache status
            $stmt = $this->conn->prepare("
                SELECT COUNT(*) as cached_prices,
                       COUNT(CASE WHEN expires_at > datetime('now') THEN 1 END) as valid_prices
                FROM market_prices
            ");
            $stmt->execute();
            $cacheStats = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['cache_status'] = $cacheStats;

            return [
                'success' => true,
                'stats' => $stats
            ];

        } catch (Exception $e) {
            error_log("Pricing stats error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Failed to get pricing statistics'
            ];
        }
    }

    /**
     * Bulk update all prices (for cron jobs)
     */
    public function updateAllPrices() {
        try {
            $configs = $this->getAllSpreadConfigs();
            $updated = 0;
            $errors = 0;

            foreach ($configs as $config) {
                $result = $this->getDynamicPrices($config['crypto'], $config['currency']);
                if ($result['success']) {
                    $updated++;
                } else {
                    $errors++;
                    error_log("Failed to update price for {$config['crypto']}/{$config['currency']}: {$result['error']}");
                }
            }

            return [
                'success' => true,
                'updated' => $updated,
                'errors' => $errors,
                'total' => count($configs)
            ];

        } catch (Exception $e) {
            error_log("Bulk price update error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Bulk price update failed'
            ];
        }
    }

    /**
     * Compare with Yellow Card-style pricing
     */
    public function compareWithCompetitor($crypto, $currency, $competitorBuyPrice = null, $competitorSellPrice = null) {
        try {
            $ourPrices = $this->getDynamicPrices($crypto, $currency);
            if (!$ourPrices['success']) {
                return [
                    'success' => false,
                    'error' => $ourPrices['error']
                ];
            }

            $comparison = [
                'success' => true,
                'crypto' => $crypto,
                'currency' => $currency,
                'our_prices' => [
                    'buy_price' => $ourPrices['buy_price'],
                    'sell_price' => $ourPrices['sell_price'],
                    'spread' => $ourPrices['effective_spread']
                ]
            ];

            if ($competitorBuyPrice && $competitorSellPrice) {
                $competitorSpread = (($competitorBuyPrice - $competitorSellPrice) / (($competitorBuyPrice + $competitorSellPrice) / 2)) * 100;
                
                $comparison['competitor_prices'] = [
                    'buy_price' => $competitorBuyPrice,
                    'sell_price' => $competitorSellPrice,
                    'spread' => round($competitorSpread, 2)
                ];

                $comparison['competitive_analysis'] = [
                    'buy_price_difference' => $ourPrices['buy_price'] - $competitorBuyPrice,
                    'sell_price_difference' => $ourPrices['sell_price'] - $competitorSellPrice,
                    'spread_difference' => $ourPrices['effective_spread'] - $competitorSpread,
                    'we_are_competitive' => abs($ourPrices['buy_price'] - $competitorBuyPrice) < ($competitorBuyPrice * 0.05) // Within 5%
                ];
            }

            return $comparison;

        } catch (Exception $e) {
            error_log("Competitor comparison error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Competitor comparison failed'
            ];
        }
    }
}
?>

