<?php
/**
 * Comprehensive Error Handling and Recovery System
 * Production-ready error handling with automatic recovery
 */

class ErrorHandler {
    private $conn;
    private $errorConfig;
    private $recoveryStrategies;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->errorConfig = $this->getErrorConfig();
        $this->recoveryStrategies = $this->getRecoveryStrategies();
        $this->setupErrorHandling();
    }
    
    /**
     * Setup error handling
     */
    private function setupErrorHandling() {
        // Set custom error handler
        set_error_handler([$this, 'handleError']);
        
        // Set custom exception handler
        set_exception_handler([$this, 'handleException']);
        
        // Set shutdown handler for fatal errors
        register_shutdown_function([$this, 'handleShutdown']);
    }
    
    /**
     * Get error handling configuration
     */
    private function getErrorConfig() {
        return [
            'log_levels' => [
                'E_ERROR' => 'CRITICAL',
                'E_WARNING' => 'WARNING',
                'E_PARSE' => 'CRITICAL',
                'E_NOTICE' => 'INFO',
                'E_CORE_ERROR' => 'CRITICAL',
                'E_CORE_WARNING' => 'WARNING',
                'E_COMPILE_ERROR' => 'CRITICAL',
                'E_COMPILE_WARNING' => 'WARNING',
                'E_USER_ERROR' => 'ERROR',
                'E_USER_WARNING' => 'WARNING',
                'E_USER_NOTICE' => 'INFO',
                'E_STRICT' => 'INFO',
                'E_RECOVERABLE_ERROR' => 'ERROR',
                'E_DEPRECATED' => 'WARNING',
                'E_USER_DEPRECATED' => 'WARNING'
            ],
            'recovery_enabled' => true,
            'auto_retry' => true,
            'max_retries' => 3,
            'retry_delay' => 5, // seconds
            'circuit_breaker_threshold' => 10,
            'circuit_breaker_timeout' => 300, // 5 minutes
            'alert_thresholds' => [
                'CRITICAL' => 1,
                'ERROR' => 5,
                'WARNING' => 20
            ]
        ];
    }
    
    /**
     * Get recovery strategies
     */
    private function getRecoveryStrategies() {
        return [
            'database_connection' => [
                'retry_count' => 3,
                'retry_delay' => 2,
                'fallback' => 'use_readonly_connection'
            ],
            'api_timeout' => [
                'retry_count' => 2,
                'retry_delay' => 5,
                'fallback' => 'use_cached_data'
            ],
            'blockchain_api_error' => [
                'retry_count' => 3,
                'retry_delay' => 10,
                'fallback' => 'mark_as_pending'
            ],
            'payment_api_error' => [
                'retry_count' => 5,
                'retry_delay' => 30,
                'fallback' => 'manual_review'
            ],
            'file_system_error' => [
                'retry_count' => 2,
                'retry_delay' => 1,
                'fallback' => 'use_temp_directory'
            ]
        ];
    }
    
    /**
     * Handle PHP errors
     */
    public function handleError($severity, $message, $file, $line) {
        $errorData = [
            'type' => 'PHP_ERROR',
            'severity' => $severity,
            'message' => $message,
            'file' => $file,
            'line' => $line,
            'timestamp' => time(),
            'level' => $this->getErrorLevel($severity)
        ];
        
        $this->logError($errorData);
        $this->attemptRecovery($errorData);
        
        // Don't execute PHP internal error handler
        return true;
    }
    
    /**
     * Handle uncaught exceptions
     */
    public function handleException($exception) {
        $errorData = [
            'type' => 'EXCEPTION',
            'class' => get_class($exception),
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'timestamp' => time(),
            'level' => 'ERROR'
        ];
        
        $this->logError($errorData);
        $this->attemptRecovery($errorData);
        
        // Send error response
        $this->sendErrorResponse($errorData);
    }
    
    /**
     * Handle shutdown errors (fatal errors)
     */
    public function handleShutdown() {
        $error = error_get_last();
        
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            $errorData = [
                'type' => 'FATAL_ERROR',
                'severity' => $error['type'],
                'message' => $error['message'],
                'file' => $error['file'],
                'line' => $error['line'],
                'timestamp' => time(),
                'level' => 'CRITICAL'
            ];
            
            $this->logError($errorData);
            $this->attemptRecovery($errorData);
            $this->sendErrorResponse($errorData);
        }
    }
    
    /**
     * Get error level from severity
     */
    private function getErrorLevel($severity) {
        $severityName = array_search($severity, array_flip($this->errorConfig['log_levels']));
        return $this->errorConfig['log_levels'][$severityName] ?? 'ERROR';
    }
    
    /**
     * Log error to database and file
     */
    private function logError($errorData) {
        try {
            // Log to database
            $this->logToDatabase($errorData);
            
            // Log to file
            $this->logToFile($errorData);
            
            // Check if alerts should be sent
            $this->checkAlertThresholds($errorData);
            
        } catch (Exception $e) {
            // Fallback to file logging only
            error_log("Error logging failed: " . $e->getMessage() . " | Original error: " . json_encode($errorData));
        }
    }
    
    /**
     * Log error to database
     */
    private function logToDatabase($errorData) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO error_logs (
                    error_type, error_level, error_message, error_file, 
                    error_line, stack_trace, context_data, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $stmt->execute([
                $errorData['type'],
                $errorData['level'],
                $errorData['message'],
                $errorData['file'] ?? null,
                $errorData['line'] ?? null,
                $errorData['trace'] ?? null,
                json_encode($errorData)
            ]);
            
        } catch (Exception $e) {
            error_log("Database error logging failed: " . $e->getMessage());
        }
    }
    
    /**
     * Log error to file
     */
    private function logToFile($errorData) {
        $logFile = __DIR__ . '/../logs/errors.log';
        $logDir = dirname($logFile);
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logEntry = sprintf(
            "[%s] %s: %s in %s:%d\n%s\n\n",
            date('Y-m-d H:i:s'),
            $errorData['level'],
            $errorData['message'],
            $errorData['file'] ?? 'unknown',
            $errorData['line'] ?? 0,
            $errorData['trace'] ?? 'No stack trace available'
        );
        
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Check alert thresholds
     */
    private function checkAlertThresholds($errorData) {
        $level = $errorData['level'];
        $threshold = $this->errorConfig['alert_thresholds'][$level] ?? null;
        
        if (!$threshold) {
            return;
        }
        
        try {
            // Count errors in the last hour
            $stmt = $this->conn->prepare("
                SELECT COUNT(*) FROM error_logs 
                WHERE error_level = ? AND created_at >= datetime('now', '-1 hour')
            ");
            $stmt->execute([$level]);
            $count = $stmt->fetchColumn();
            
            if ($count >= $threshold) {
                $this->sendAlert($level, $count, $errorData);
            }
            
        } catch (Exception $e) {
            error_log("Alert threshold check failed: " . $e->getMessage());
        }
    }
    
    /**
     * Send alert
     */
    private function sendAlert($level, $count, $errorData) {
        // In production, integrate with alerting systems like:
        // - Email notifications
        // - Slack/Discord webhooks
        // - PagerDuty
        // - SMS alerts
        
        $alertMessage = sprintf(
            "ALERT: %d %s errors detected in the last hour. Latest error: %s",
            $count,
            $level,
            $errorData['message']
        );
        
        error_log("ALERT: " . $alertMessage);
        
        // Log alert to database
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO alerts (alert_type, alert_level, alert_message, created_at)
                VALUES (?, ?, ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute(['error_threshold', $level, $alertMessage]);
        } catch (Exception $e) {
            error_log("Alert logging failed: " . $e->getMessage());
        }
    }
    
    /**
     * Attempt error recovery
     */
    private function attemptRecovery($errorData) {
        if (!$this->errorConfig['recovery_enabled']) {
            return;
        }
        
        $recoveryType = $this->determineRecoveryType($errorData);
        $strategy = $this->recoveryStrategies[$recoveryType] ?? null;
        
        if (!$strategy) {
            return;
        }
        
        try {
            $this->executeRecovery($recoveryType, $strategy, $errorData);
        } catch (Exception $e) {
            error_log("Recovery attempt failed: " . $e->getMessage());
        }
    }
    
    /**
     * Determine recovery type based on error
     */
    private function determineRecoveryType($errorData) {
        $message = strtolower($errorData['message']);
        
        if (strpos($message, 'database') !== false || strpos($message, 'connection') !== false) {
            return 'database_connection';
        } elseif (strpos($message, 'timeout') !== false) {
            return 'api_timeout';
        } elseif (strpos($message, 'blockchain') !== false || strpos($message, 'etherscan') !== false) {
            return 'blockchain_api_error';
        } elseif (strpos($message, 'payment') !== false || strpos($message, 'bitmuk') !== false) {
            return 'payment_api_error';
        } elseif (strpos($message, 'file') !== false || strpos($message, 'directory') !== false) {
            return 'file_system_error';
        }
        
        return null;
    }
    
    /**
     * Execute recovery strategy
     */
    private function executeRecovery($recoveryType, $strategy, $errorData) {
        $maxRetries = $strategy['retry_count'];
        $retryDelay = $strategy['retry_delay'];
        $fallback = $strategy['fallback'];
        
        for ($i = 0; $i < $maxRetries; $i++) {
            if ($this->attemptRetry($recoveryType, $errorData)) {
                $this->logRecoverySuccess($recoveryType, $i + 1);
                return;
            }
            
            if ($i < $maxRetries - 1) {
                sleep($retryDelay);
            }
        }
        
        // All retries failed, use fallback
        $this->executeFallback($recoveryType, $fallback, $errorData);
    }
    
    /**
     * Attempt retry
     */
    private function attemptRetry($recoveryType, $errorData) {
        switch ($recoveryType) {
            case 'database_connection':
                return $this->retryDatabaseConnection();
                
            case 'api_timeout':
                return $this->retryAPICall($errorData);
                
            case 'blockchain_api_error':
                return $this->retryBlockchainAPI($errorData);
                
            case 'payment_api_error':
                return $this->retryPaymentAPI($errorData);
                
            default:
                return false;
        }
    }
    
    /**
     * Retry database connection
     */
    private function retryDatabaseConnection() {
        try {
            // Close existing connection
            $this->conn = null;
            
            // Create new connection
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $this->conn = $db->getConnection();
            
            // Test connection
            $stmt = $this->conn->query("SELECT 1");
            return $stmt !== false;
            
        } catch (Exception $e) {
            error_log("Database retry failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Retry API call
     */
    private function retryAPICall($errorData) {
        // This would implement specific API retry logic
        // For now, just return true as a placeholder
        return true;
    }
    
    /**
     * Retry blockchain API
     */
    private function retryBlockchainAPI($errorData) {
        // This would implement blockchain API retry logic
        return true;
    }
    
    /**
     * Retry payment API
     */
    private function retryPaymentAPI($errorData) {
        // This would implement payment API retry logic
        return true;
    }
    
    /**
     * Execute fallback strategy
     */
    private function executeFallback($recoveryType, $fallback, $errorData) {
        switch ($fallback) {
            case 'use_readonly_connection':
                $this->useReadonlyConnection();
                break;
                
            case 'use_cached_data':
                $this->useCachedData($errorData);
                break;
                
            case 'mark_as_pending':
                $this->markAsPending($errorData);
                break;
                
            case 'manual_review':
                $this->flagForManualReview($errorData);
                break;
                
            case 'use_temp_directory':
                $this->useTempDirectory();
                break;
        }
        
        $this->logFallbackUsed($recoveryType, $fallback, $errorData);
    }
    
    /**
     * Use readonly database connection
     */
    private function useReadonlyConnection() {
        // Implement readonly connection logic
        error_log("Using readonly database connection as fallback");
    }
    
    /**
     * Use cached data
     */
    private function useCachedData($errorData) {
        // Implement cached data logic
        error_log("Using cached data as fallback for: " . $errorData['message']);
    }
    
    /**
     * Mark transaction as pending
     */
    private function markAsPending($errorData) {
        // Implement pending marking logic
        error_log("Marking transaction as pending due to: " . $errorData['message']);
    }
    
    /**
     * Flag for manual review
     */
    private function flagForManualReview($errorData) {
        // Implement manual review flagging
        error_log("Flagging for manual review due to: " . $errorData['message']);
    }
    
    /**
     * Use temporary directory
     */
    private function useTempDirectory() {
        // Implement temp directory logic
        error_log("Using temporary directory as fallback");
    }
    
    /**
     * Log recovery success
     */
    private function logRecoverySuccess($recoveryType, $attempts) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO recovery_logs (recovery_type, attempts, status, created_at)
                VALUES (?, ?, 'success', CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$recoveryType, $attempts]);
        } catch (Exception $e) {
            error_log("Recovery success logging failed: " . $e->getMessage());
        }
    }
    
    /**
     * Log fallback usage
     */
    private function logFallbackUsed($recoveryType, $fallback, $errorData) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO recovery_logs (recovery_type, attempts, status, fallback_used, error_message, created_at)
                VALUES (?, 0, 'fallback', ?, ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$recoveryType, $fallback, $errorData['message']]);
        } catch (Exception $e) {
            error_log("Fallback logging failed: " . $e->getMessage());
        }
    }
    
    /**
     * Send error response to client
     */
    private function sendErrorResponse($errorData) {
        if (php_sapi_name() === 'cli') {
            return; // Don't send HTTP response for CLI
        }
        
        http_response_code(500);
        
        $response = [
            'success' => false,
            'error' => 'Internal server error',
            'error_id' => uniqid('err_'),
            'timestamp' => date('c')
        ];
        
        // Include error details in development mode
        if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
            $response['debug'] = [
                'message' => $errorData['message'],
                'file' => $errorData['file'] ?? null,
                'line' => $errorData['line'] ?? null
            ];
        }
        
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
    
    /**
     * Create error logs table if it doesn't exist
     */
    public function ensureErrorTablesExist() {
        try {
            // Create error_logs table
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS error_logs (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    error_type TEXT NOT NULL,
                    error_level TEXT NOT NULL,
                    error_message TEXT NOT NULL,
                    error_file TEXT,
                    error_line INTEGER,
                    stack_trace TEXT,
                    context_data TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Create recovery_logs table
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS recovery_logs (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    recovery_type TEXT NOT NULL,
                    attempts INTEGER DEFAULT 0,
                    status TEXT NOT NULL,
                    fallback_used TEXT,
                    error_message TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Create alerts table
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS alerts (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    alert_type TEXT NOT NULL,
                    alert_level TEXT NOT NULL,
                    alert_message TEXT NOT NULL,
                    acknowledged BOOLEAN DEFAULT FALSE,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Create indexes
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_error_logs_level ON error_logs(error_level)");
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_error_logs_created ON error_logs(created_at)");
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_recovery_logs_type ON recovery_logs(recovery_type)");
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_alerts_level ON alerts(alert_level)");
            
        } catch (Exception $e) {
            error_log("Error creating error tables: " . $e->getMessage());
        }
    }
    
    /**
     * Get error statistics
     */
    public function getErrorStatistics($timeframe = '24h') {
        try {
            $timeCondition = $this->getTimeframeCondition($timeframe);
            
            // Get error counts by level
            $stmt = $this->conn->prepare("
                SELECT 
                    error_level,
                    COUNT(*) as count
                FROM error_logs 
                WHERE created_at >= $timeCondition
                GROUP BY error_level
                ORDER BY count DESC
            ");
            $stmt->execute();
            $errorCounts = $stmt->fetchAll();
            
            // Get recovery statistics
            $stmt = $this->conn->prepare("
                SELECT 
                    recovery_type,
                    status,
                    COUNT(*) as count
                FROM recovery_logs 
                WHERE created_at >= $timeCondition
                GROUP BY recovery_type, status
                ORDER BY count DESC
            ");
            $stmt->execute();
            $recoveryStats = $stmt->fetchAll();
            
            // Get recent alerts
            $stmt = $this->conn->prepare("
                SELECT *
                FROM alerts 
                WHERE created_at >= $timeCondition
                ORDER BY created_at DESC
                LIMIT 10
            ");
            $stmt->execute();
            $recentAlerts = $stmt->fetchAll();
            
            return [
                'success' => true,
                'timeframe' => $timeframe,
                'error_counts' => $errorCounts,
                'recovery_stats' => $recoveryStats,
                'recent_alerts' => $recentAlerts
            ];
            
        } catch (Exception $e) {
            error_log("Error statistics failed: " . $e->getMessage());
            return ['success' => false, 'error' => 'Statistics failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get timeframe condition for SQL
     */
    private function getTimeframeCondition($timeframe) {
        switch ($timeframe) {
            case '1h':
                return "datetime('now', '-1 hour')";
            case '24h':
                return "datetime('now', '-1 day')";
            case '7d':
                return "datetime('now', '-7 days')";
            case '30d':
                return "datetime('now', '-30 days')";
            default:
                return "datetime('now', '-1 day')";
        }
    }
}
?>

