<?php
/**
 * Event-Driven Payment Status Checker
 * 
 * This system automatically checks payment statuses without requiring cron jobs.
 * It uses intelligent triggering based on user actions and system events.
 */

require_once __DIR__ . '/../config/database.php';

class EventDrivenPaymentChecker {
    private $conn;
    private $lastCheckTime;
    private $checkInterval = 300; // 5 minutes minimum between checks
    
    public function __construct() {
        $db = new Database();
        $this->conn = $db->getConnection();
        $this->lastCheckTime = $this->getLastCheckTime();
        $this->ensureSystemSettingsTable();
    }
    
    /**
     * Main entry point - called by various events
     */
    public function checkIfNeeded($forceCheck = false) {
        error_log("Event-driven payment checker disabled - system now completely depends on Bitmuk's status");
        
        // Return early - no automated checking
        return [
            'success' => true,
            'message' => 'Event-driven checker disabled - system depends on Bitmuk status',
            'processed' => 0,
            'completed' => 0
        ];
        
        // OLD CODE (DISABLED):
        // $now = time();
        // 
        // // Only check if enough time has passed or if forced
        // if (!$forceCheck && ($now - $this->lastCheckTime) < $this->checkInterval) {
        //     return ['skipped' => true, 'message' => 'Check not needed yet'];
        // }
        // 
        // // Perform the actual check
        // $result = $this->performCheck();
        // 
        // // Update last check time
        // $this->updateLastCheckTime($now);
        
        return $result;
    }
    
    /**
     * Perform the actual payment status check
     */
    private function performCheck() {
        error_log("[" . date('Y-m-d H:i:s') . "] Event-driven payment check started");
        
        // Get all pending payments older than 5 minutes
        $stmt = $this->conn->prepare("
            SELECT * FROM payments 
            WHERE status = 'pending' 
            AND created_at < datetime('now', '-5 minutes')
            ORDER BY created_at ASC
        ");
        $stmt->execute();
        $pendingPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $processedCount = 0;
        $completedCount = 0;
        
        foreach ($pendingPayments as $payment) {
            $processedCount++;
            
            // DISABLED: Automatic completion logic removed to prevent incorrect status updates
            // Payments should only be marked as completed through:
            // 1. Bitmuk API confirmation
            // 2. Manual verification
            // 3. Time-based failure (after reasonable timeout)
            
            // Intelligent fallback logic for Bitmuk API failures
            $createdTime = new DateTime($payment['created_at']);
            $now = new DateTime();
            $minutesSinceCreated = $now->diff($createdTime)->i;
            $hoursSinceCreated = $now->diff($createdTime)->h;
            
            // If payment is older than 15 minutes and still pending, likely failed
            // This handles cases where Bitmuk API is unreliable
            if (($minutesSinceCreated >= 15 || $hoursSinceCreated >= 1) && $payment['status'] === 'pending') {
                error_log("Intelligent fallback: Payment {$payment['transaction_id']} is older than 15 minutes, marking as failed");
                $this->updatePaymentStatus($payment, 'failed');
            }
        }
        
        $message = "Processed $processedCount payments, completed $completedCount";
        error_log("[" . date('Y-m-d H:i:s') . "] Event-driven payment check: $message");
        
        return [
            'success' => true,
            'processed' => $processedCount,
            'completed' => $completedCount,
            'message' => $message
        ];
    }
    
    /**
     * Update payment status and credit account with duplicate prevention
     */
    private function updatePaymentStatus($payment, $status) {
        $transactionId = $payment['transaction_id'];
        $userId = $payment['user_id'];
        $amount = $payment['amount'];
        $currency = $payment['currency'];
        
        // CRITICAL: Check if payment is already completed to prevent duplicate crediting
        if ($payment['status'] === 'completed') {
            error_log("Payment $transactionId is already completed, skipping update");
            return;
        }
        
        // Use database transaction to ensure atomicity
        $this->conn->beginTransaction();
        
        try {
            // Double-check payment status within transaction (prevent race conditions)
            $stmt = $this->conn->prepare("
                SELECT status FROM payments 
                WHERE transaction_id = ? AND status != 'completed'
            ");
            $stmt->execute([$transactionId]);
            $currentStatus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$currentStatus || $currentStatus['status'] === 'completed') {
                $this->conn->rollback();
                error_log("Payment $transactionId already processed, aborting update");
                return;
            }
            
            // Update payment status
            $stmt = $this->conn->prepare("
                UPDATE payments 
                SET status = ?, updated_at = datetime('now')
                WHERE transaction_id = ? AND status != 'completed'
            ");
            $result = $stmt->execute([$status, $transactionId]);
            
            if ($stmt->rowCount() === 0) {
                $this->conn->rollback();
                error_log("Payment $transactionId was already updated by another process");
                return;
            }
            
            // If completed, credit user account
            if ($status === 'completed') {
                $this->creditUserAccount($userId, $amount, $currency, $transactionId);
            }
            
            // Commit the transaction
            $this->conn->commit();
            error_log("Payment $transactionId successfully updated to $status");
            
        } catch (Exception $e) {
            $this->conn->rollback();
            error_log("Error updating payment $transactionId: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Credit user account with duplicate prevention
     */
    private function creditUserAccount($userId, $amount, $currency, $transactionId) {
        // CRITICAL: Check if this payment has already been credited
        $stmt = $this->conn->prepare("
            SELECT COUNT(*) as count FROM transactions 
            WHERE user_id = ? AND type = 'deposit' AND currency = ? 
            AND amount = ? AND description LIKE ?
        ");
        $stmt->execute([$userId, $currency, $amount, "%$transactionId%"]);
        $existingTransaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingTransaction['count'] > 0) {
            error_log("Payment $transactionId already credited to user $userId, skipping");
            return;
        }
        
        // Check if user already has this currency balance
        $stmt = $this->conn->prepare("
            SELECT balance FROM user_fiat_balances 
            WHERE user_id = ? AND currency_code = ?
        ");
        $stmt->execute([$userId, $currency]);
        $existingBalance = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingBalance) {
            // Update existing balance
            $stmt = $this->conn->prepare("
                UPDATE user_fiat_balances 
                SET balance = balance + ?, updated_at = datetime('now')
                WHERE user_id = ? AND currency_code = ?
            ");
            $stmt->execute([$amount, $userId, $currency]);
        } else {
            // Create new balance record
            $stmt = $this->conn->prepare("
                INSERT INTO user_fiat_balances (user_id, currency_code, balance, created_at, updated_at)
                VALUES (?, ?, ?, datetime('now'), datetime('now'))
            ");
            $stmt->execute([$userId, $currency, $amount]);
        }
        
        // Add transaction record with payment reference for duplicate detection
        $stmt = $this->conn->prepare("
            INSERT INTO transactions (user_id, type, currency, amount, status, description, created_at)
            VALUES (?, 'deposit', ?, ?, 'completed', ?, datetime('now'))
        ");
        $description = "Deposit via payment $transactionId";
        $stmt->execute([$userId, $currency, $amount, $description]);
        
        error_log("Successfully credited user $userId with $amount $currency for payment $transactionId");
    }
    
    /**
     * Get last check time from database
     */
    private function getLastCheckTime() {
        $stmt = $this->conn->prepare("
            SELECT value FROM system_settings 
            WHERE setting_key = 'last_payment_check'
        ");
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result ? (int)$result['value'] : 0;
    }
    
    /**
     * Update last check time in database
     */
    private function updateLastCheckTime($timestamp) {
        $stmt = $this->conn->prepare("
            INSERT OR REPLACE INTO system_settings (setting_key, value, updated_at)
            VALUES ('last_payment_check', ?, datetime('now'))
        ");
        $stmt->execute([$timestamp]);
    }
    
    /**
     * Create system_settings table if it doesn't exist
     */
    public function ensureSystemSettingsTable() {
        $stmt = $this->conn->prepare("
            CREATE TABLE IF NOT EXISTS system_settings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                setting_key TEXT UNIQUE NOT NULL,
                value TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        $stmt->execute();
    }
}

?>
