<?php
/**
 * Fund Protection and Insurance System
 * Protects user funds with insurance and recovery mechanisms
 */

class FundProtection {
    private $db;
    private $insurance_percentage = 0.02; // 2% of funds are insured
    private $recovery_fund_minimum = 100000; // Minimum $100k recovery fund
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * Calculate insurance coverage for user funds
     */
    public function calculateInsuranceCoverage($user_id) {
        try {
            $conn = $this->db->getConnection();
            
            // Get total user funds
            $total_funds = $this->getTotalUserFunds($user_id);
            $insurance_coverage = $total_funds * $this->insurance_percentage;
            
            return [
                'total_funds' => $total_funds,
                'insurance_coverage' => $insurance_coverage,
                'insurance_percentage' => $this->insurance_percentage * 100,
                'coverage_status' => $insurance_coverage > 0 ? 'active' : 'none'
            ];
            
        } catch (Exception $e) {
            error_log("Insurance calculation failed: " . $e->getMessage());
            return ['error' => 'Failed to calculate insurance coverage'];
        }
    }
    
    /**
     * Get recovery fund status
     */
    public function getRecoveryFundStatus() {
        try {
            $conn = $this->db->getConnection();
            
            // Calculate total system funds
            $stmt = $conn->prepare("
                SELECT SUM(balance * cp.buy_price) as total_value
                FROM crypto_wallets cw
                JOIN crypto_prices cp ON cw.currency = cp.crypto
                WHERE cw.user_id = 0 AND cp.is_active = 1
            ");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $total_system_funds = $result['total_value'] ?? 0;
            $recovery_fund = $total_system_funds * 0.1; // 10% of system funds as recovery
            
            return [
                'total_system_funds' => $total_system_funds,
                'recovery_fund' => $recovery_fund,
                'minimum_required' => $this->recovery_fund_minimum,
                'status' => $recovery_fund >= $this->recovery_fund_minimum ? 'adequate' : 'insufficient',
                'coverage_ratio' => $recovery_fund / $this->recovery_fund_minimum
            ];
            
        } catch (Exception $e) {
            error_log("Recovery fund calculation failed: " . $e->getMessage());
            return ['error' => 'Failed to calculate recovery fund status'];
        }
    }
    
    /**
     * Initiate fund recovery process
     */
    public function initiateFundRecovery($user_id, $loss_amount, $reason, $evidence) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();
            
            // Create recovery request
            $stmt = $conn->prepare("
                INSERT INTO fund_recovery_requests (
                    user_id, loss_amount, reason, evidence, 
                    status, created_at
                ) VALUES (?, ?, ?, ?, 'pending', CURRENT_TIMESTAMP)
            ");
            $stmt->execute([
                $user_id, $loss_amount, $reason, json_encode($evidence)
            ]);
            
            $recovery_id = $conn->lastInsertId();
            
            // Check if auto-approval is possible
            if ($this->canAutoApproveRecovery($user_id, $loss_amount)) {
                $this->approveRecovery($recovery_id, 'auto_approved', 'Automatically approved based on criteria');
            }
            
            $conn->commit();
            
            return [
                'success' => true,
                'recovery_id' => $recovery_id,
                'status' => 'pending',
                'message' => 'Fund recovery request submitted'
            ];
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Fund recovery initiation failed: " . $e->getMessage());
            return ['error' => 'Failed to initiate fund recovery'];
        }
    }
    
    /**
     * Approve fund recovery
     */
    public function approveRecovery($recovery_id, $approved_by, $notes) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();
            
            // Get recovery request
            $stmt = $conn->prepare("
                SELECT * FROM fund_recovery_requests WHERE id = ?
            ");
            $stmt->execute([$recovery_id]);
            $request = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$request) {
                throw new Exception("Recovery request not found");
            }
            
            // Update status
            $stmt = $conn->prepare("
                UPDATE fund_recovery_requests 
                SET status = 'approved', approved_by = ?, approved_at = CURRENT_TIMESTAMP, 
                    approval_notes = ?
                WHERE id = ?
            ");
            $stmt->execute([$approved_by, $notes, $recovery_id]);
            
            // Process the recovery payment
            $this->processRecoveryPayment($request['user_id'], $request['loss_amount']);
            
            $conn->commit();
            
            return [
                'success' => true,
                'message' => 'Fund recovery approved and processed'
            ];
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Recovery approval failed: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get fund protection statistics
     */
    public function getProtectionStatistics() {
        try {
            $conn = $this->db->getConnection();
            
            // Total users with insurance
            $stmt = $conn->prepare("
                SELECT COUNT(DISTINCT user_id) as insured_users
                FROM user_fiat_balances ufb
                JOIN crypto_wallets cw ON ufb.user_id = cw.user_id
                WHERE ufb.balance > 0 OR cw.balance > 0
            ");
            $stmt->execute();
            $insured_users = $stmt->fetch(PDO::FETCH_ASSOC)['insured_users'];
            
            // Total insurance coverage
            $stmt = $conn->prepare("
                SELECT 
                    (SELECT SUM(balance) FROM user_fiat_balances) as fiat_coverage,
                    (SELECT SUM(cw.balance * cp.buy_price) 
                     FROM crypto_wallets cw 
                     JOIN crypto_prices cp ON cw.currency = cp.crypto 
                     WHERE cp.is_active = 1) as crypto_coverage
            ");
            $stmt->execute();
            $coverage = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $total_coverage = ($coverage['fiat_coverage'] ?? 0) + ($coverage['crypto_coverage'] ?? 0);
            $insurance_amount = $total_coverage * $this->insurance_percentage;
            
            // Recovery requests stats
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_requests,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'approved' THEN loss_amount ELSE 0 END) as total_paid_out
                FROM fund_recovery_requests
            ");
            $stmt->execute();
            $recovery_stats = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return [
                'insured_users' => $insured_users,
                'total_fund_coverage' => $total_coverage,
                'insurance_amount' => $insurance_amount,
                'recovery_fund_status' => $this->getRecoveryFundStatus(),
                'recovery_statistics' => $recovery_stats,
                'protection_level' => $this->calculateProtectionLevel()
            ];
            
        } catch (Exception $e) {
            error_log("Protection statistics failed: " . $e->getMessage());
            return ['error' => 'Failed to get protection statistics'];
        }
    }
    
    /**
     * Monitor system security
     */
    public function monitorSystemSecurity() {
        $alerts = [];
        
        // Check recovery fund adequacy
        $recovery_status = $this->getRecoveryFundStatus();
        if ($recovery_status['status'] === 'insufficient') {
            $alerts[] = [
                'type' => 'critical',
                'message' => 'Recovery fund is below minimum required amount',
                'details' => $recovery_status
            ];
        }
        
        // Check for unusual transaction patterns
        $unusual_activity = $this->detectUnusualActivity();
        if ($unusual_activity) {
            $alerts[] = [
                'type' => 'warning',
                'message' => 'Unusual transaction activity detected',
                'details' => $unusual_activity
            ];
        }
        
        // Check system wallet balances
        $wallet_alerts = $this->checkWalletBalances();
        $alerts = array_merge($alerts, $wallet_alerts);
        
        return [
            'status' => empty($alerts) ? 'secure' : 'alerts_present',
            'alerts' => $alerts,
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Helper methods
     */
    private function getTotalUserFunds($user_id) {
        try {
            $conn = $this->db->getConnection();
            
            // Fiat balance
            $stmt = $conn->prepare("
                SELECT SUM(balance) as fiat_total
                FROM user_fiat_balances 
                WHERE user_id = ?
            ");
            $stmt->execute([$user_id]);
            $fiat_total = $stmt->fetch(PDO::FETCH_ASSOC)['fiat_total'] ?? 0;
            
            // Crypto balance (in USD)
            $stmt = $conn->prepare("
                SELECT SUM(cw.balance * cp.buy_price) as crypto_total
                FROM crypto_wallets cw
                JOIN crypto_prices cp ON cw.currency = cp.crypto
                WHERE cw.user_id = ? AND cp.is_active = 1
            ");
            $stmt->execute([$user_id]);
            $crypto_total = $stmt->fetch(PDO::FETCH_ASSOC)['crypto_total'] ?? 0;
            
            return $fiat_total + $crypto_total;
            
        } catch (Exception $e) {
            return 0;
        }
    }
    
    private function canAutoApproveRecovery($user_id, $loss_amount) {
        // Auto-approve if loss is small and user has good history
        return $loss_amount <= 100; // Auto-approve losses under $100
    }
    
    private function processRecoveryPayment($user_id, $amount) {
        try {
            $conn = $this->db->getConnection();
            
            // Add to user's USD balance
            $stmt = $conn->prepare("
                INSERT OR REPLACE INTO user_fiat_balances (user_id, currency_code, balance, updated_at)
                VALUES (?, 'USD', COALESCE((SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency_code = 'USD'), 0) + ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$user_id, $user_id, $amount]);
            
            // Log the recovery payment
            $stmt = $conn->prepare("
                INSERT INTO recovery_payments (user_id, amount, type, created_at)
                VALUES (?, ?, 'fund_recovery', CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$user_id, $amount]);
            
        } catch (Exception $e) {
            error_log("Recovery payment processing failed: " . $e->getMessage());
            throw $e;
        }
    }
    
    private function calculateProtectionLevel() {
        $recovery_status = $this->getRecoveryFundStatus();
        
        if ($recovery_status['coverage_ratio'] >= 2.0) {
            return 'excellent';
        } elseif ($recovery_status['coverage_ratio'] >= 1.5) {
            return 'good';
        } elseif ($recovery_status['coverage_ratio'] >= 1.0) {
            return 'adequate';
        } else {
            return 'insufficient';
        }
    }
    
    private function detectUnusualActivity() {
        // Check for unusual transaction patterns
        try {
            $conn = $this->db->getConnection();
            
            // Check for high-volume transactions in last hour
            $stmt = $conn->prepare("
                SELECT COUNT(*) as transaction_count, SUM(amount) as total_amount
                FROM crypto_transactions 
                WHERE created_at >= datetime('now', '-1 hour')
                AND type IN ('buy', 'sell')
            ");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result['transaction_count'] > 100 || $result['total_amount'] > 100000) {
                return [
                    'type' => 'high_volume',
                    'count' => $result['transaction_count'],
                    'amount' => $result['total_amount']
                ];
            }
            
            return null;
            
        } catch (Exception $e) {
            return null;
        }
    }
    
    private function checkWalletBalances() {
        $alerts = [];
        
        try {
            $conn = $this->db->getConnection();
            
            // Check system wallet balances
            $stmt = $conn->prepare("
                SELECT currency, balance, address
                FROM crypto_wallets 
                WHERE user_id = 0 AND balance < 10
            ");
            $stmt->execute();
            $low_balance_wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($low_balance_wallets as $wallet) {
                $alerts[] = [
                    'type' => 'warning',
                    'message' => "Low balance in system wallet: {$wallet['currency']} ({$wallet['balance']})",
                    'wallet' => $wallet
                ];
            }
            
        } catch (Exception $e) {
            // Silent fail for monitoring
        }
        
        return $alerts;
    }
}
?>

