<?php
/**
 * Secure Private Key Management System
 * Production-ready private key encryption and management
 */

class KeyManagement {
    private $conn;
    private $encryptionKey;
    private $kmsConfig;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->encryptionKey = $this->getEncryptionKey();
        $this->kmsConfig = $this->getKMSConfig();
    }
    
    /**
     * Get encryption key from environment or generate one
     */
    private function getEncryptionKey() {
        $key = getenv('ENCRYPTION_KEY') ?: ENCRYPTION_KEY;
        
        if (strlen($key) < 32) {
            error_log("WARNING: Encryption key is too short. Please set ENCRYPTION_KEY environment variable with at least 32 characters.");
            // Generate a temporary key for development
            $key = hash('sha256', 'temp-key-' . time() . '-' . uniqid());
        }
        
        return $key;
    }
    
    /**
     * Get KMS configuration
     */
    private function getKMSConfig() {
        return [
            'aws_kms' => [
                'enabled' => getenv('AWS_KMS_ENABLED') === 'true',
                'region' => getenv('AWS_KMS_REGION') ?: 'us-east-1',
                'key_id' => getenv('AWS_KMS_KEY_ID'),
                'access_key' => getenv('AWS_KMS_ACCESS_KEY'),
                'secret_key' => getenv('AWS_KMS_SECRET_KEY')
            ],
            'azure_kms' => [
                'enabled' => getenv('AZURE_KMS_ENABLED') === 'true',
                'vault_url' => getenv('AZURE_VAULT_URL'),
                'client_id' => getenv('AZURE_CLIENT_ID'),
                'client_secret' => getenv('AZURE_CLIENT_SECRET'),
                'tenant_id' => getenv('AZURE_TENANT_ID')
            ],
            'google_kms' => [
                'enabled' => getenv('GOOGLE_KMS_ENABLED') === 'true',
                'project_id' => getenv('GOOGLE_PROJECT_ID'),
                'location' => getenv('GOOGLE_KMS_LOCATION') ?: 'global',
                'key_ring' => getenv('GOOGLE_KEY_RING'),
                'key_name' => getenv('GOOGLE_KEY_NAME'),
                'credentials' => getenv('GOOGLE_APPLICATION_CREDENTIALS')
            ]
        ];
    }
    
    /**
     * Generate a new private key for a specific network
     */
    public function generatePrivateKey($network, $userId) {
        try {
            switch ($network) {
                case 'ethereum':
                case 'bsc':
                case 'polygon':
                case 'base':
                    return $this->generateEthereumPrivateKey($userId);
                    
                case 'tron':
                    return $this->generateTronPrivateKey($userId);
                    
                case 'solana':
                    return $this->generateSolanaPrivateKey($userId);
                    
                default:
                    return ['success' => false, 'error' => 'Unsupported network: ' . $network];
            }
        } catch (Exception $e) {
            error_log("Private key generation error for $network: " . $e->getMessage());
            return ['success' => false, 'error' => 'Generation failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Generate Ethereum-compatible private key
     */
    private function generateEthereumPrivateKey($userId) {
        // Generate 32 random bytes
        $privateKeyBytes = random_bytes(32);
        $privateKey = bin2hex($privateKeyBytes);
        
        // Generate public key and address
        $publicKey = $this->privateKeyToPublicKey($privateKey, 'ethereum');
        $address = $this->publicKeyToAddress($publicKey, 'ethereum');
        
        // Encrypt the private key
        $encryptedPrivateKey = $this->encryptPrivateKey($privateKey, $userId);
        
        return [
            'success' => true,
            'private_key' => $encryptedPrivateKey,
            'public_key' => $publicKey,
            'address' => $address,
            'network' => 'ethereum'
        ];
    }
    
    /**
     * Generate Tron private key
     */
    private function generateTronPrivateKey($userId) {
        // Generate 32 random bytes
        $privateKeyBytes = random_bytes(32);
        $privateKey = bin2hex($privateKeyBytes);
        
        // Generate Tron address
        $address = $this->privateKeyToTronAddress($privateKey);
        
        // Encrypt the private key
        $encryptedPrivateKey = $this->encryptPrivateKey($privateKey, $userId);
        
        return [
            'success' => true,
            'private_key' => $encryptedPrivateKey,
            'address' => $address,
            'network' => 'tron'
        ];
    }
    
    /**
     * Generate Solana private key
     */
    private function generateSolanaPrivateKey($userId) {
        // Generate 64 random bytes for Solana
        $privateKeyBytes = random_bytes(64);
        $privateKey = base64_encode($privateKeyBytes);
        
        // Generate Solana address
        $address = $this->privateKeyToSolanaAddress($privateKeyBytes);
        
        // Encrypt the private key
        $encryptedPrivateKey = $this->encryptPrivateKey($privateKey, $userId);
        
        return [
            'success' => true,
            'private_key' => $encryptedPrivateKey,
            'address' => $address,
            'network' => 'solana'
        ];
    }
    
    /**
     * Encrypt private key with multiple layers of security
     */
    private function encryptPrivateKey($privateKey, $userId) {
        try {
            // Layer 1: AES-256-GCM encryption
            $iv = random_bytes(16);
            $tag = '';
            
            $encrypted = openssl_encrypt(
                $privateKey,
                'aes-256-gcm',
                $this->encryptionKey,
                OPENSSL_RAW_DATA,
                $iv,
                $tag
            );
            
            if ($encrypted === false) {
                throw new Exception('AES encryption failed');
            }
            
            // Layer 2: User-specific encryption
            $userKey = hash('sha256', $this->encryptionKey . $userId . 'user-specific-key', true);
            $userIv = random_bytes(16);
            $userTag = '';
            
            $userEncrypted = openssl_encrypt(
                $encrypted,
                'aes-256-gcm',
                $userKey,
                OPENSSL_RAW_DATA,
                $userIv,
                $userTag
            );
            
            if ($userEncrypted === false) {
                throw new Exception('User-specific encryption failed');
            }
            
            // Combine all components
            $encryptedData = [
                'version' => '2.0',
                'algorithm' => 'aes-256-gcm-double',
                'data' => base64_encode($userEncrypted),
                'iv' => base64_encode($userIv),
                'tag' => base64_encode($userTag),
                'inner_iv' => base64_encode($iv),
                'inner_tag' => base64_encode($tag),
                'timestamp' => time(),
                'user_id' => $userId
            ];
            
            // Layer 3: KMS encryption (if available)
            if ($this->shouldUseKMS()) {
                $kmsResult = $this->encryptWithKMS(json_encode($encryptedData));
                if ($kmsResult['success']) {
                    $encryptedData['kms_encrypted'] = true;
                    $encryptedData['kms_data'] = $kmsResult['data'];
                }
            }
            
            return base64_encode(json_encode($encryptedData));
            
        } catch (Exception $e) {
            error_log("Private key encryption error: " . $e->getMessage());
            throw new Exception('Encryption failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Decrypt private key
     */
    public function decryptPrivateKey($encryptedPrivateKey, $userId) {
        try {
            $encryptedData = json_decode(base64_decode($encryptedPrivateKey), true);
            
            if (!$encryptedData || !isset($encryptedData['version'])) {
                throw new Exception('Invalid encrypted data format');
            }
            
            // Check if KMS encrypted
            if (isset($encryptedData['kms_encrypted']) && $encryptedData['kms_encrypted']) {
                $kmsResult = $this->decryptWithKMS($encryptedData['kms_data']);
                if (!$kmsResult['success']) {
                    throw new Exception('KMS decryption failed: ' . $kmsResult['error']);
                }
                $encryptedData = json_decode($kmsResult['data'], true);
            }
            
            // Verify user ID
            if ($encryptedData['user_id'] !== $userId) {
                throw new Exception('User ID mismatch');
            }
            
            // Layer 2: User-specific decryption
            $userKey = hash('sha256', $this->encryptionKey . $userId . 'user-specific-key', true);
            $userDecrypted = openssl_decrypt(
                base64_decode($encryptedData['data']),
                'aes-256-gcm',
                $userKey,
                OPENSSL_RAW_DATA,
                base64_decode($encryptedData['iv']),
                base64_decode($encryptedData['tag'])
            );
            
            if ($userDecrypted === false) {
                throw new Exception('User-specific decryption failed');
            }
            
            // Layer 1: AES decryption
            $privateKey = openssl_decrypt(
                $userDecrypted,
                'aes-256-gcm',
                $this->encryptionKey,
                OPENSSL_RAW_DATA,
                base64_decode($encryptedData['inner_iv']),
                base64_decode($encryptedData['inner_tag'])
            );
            
            if ($privateKey === false) {
                throw new Exception('AES decryption failed');
            }
            
            return [
                'success' => true,
                'private_key' => $privateKey
            ];
            
        } catch (Exception $e) {
            error_log("Private key decryption error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Decryption failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Check if KMS should be used
     */
    private function shouldUseKMS() {
        return $this->kmsConfig['aws_kms']['enabled'] || 
               $this->kmsConfig['azure_kms']['enabled'] || 
               $this->kmsConfig['google_kms']['enabled'];
    }
    
    /**
     * Encrypt with KMS (placeholder for AWS/Azure/Google KMS integration)
     */
    private function encryptWithKMS($data) {
        // This is a placeholder. In production, integrate with actual KMS services
        // AWS KMS, Azure Key Vault, or Google Cloud KMS
        
        if ($this->kmsConfig['aws_kms']['enabled']) {
            return $this->encryptWithAWSKMS($data);
        } elseif ($this->kmsConfig['azure_kms']['enabled']) {
            return $this->encryptWithAzureKMS($data);
        } elseif ($this->kmsConfig['google_kms']['enabled']) {
            return $this->encryptWithGoogleKMS($data);
        }
        
        return ['success' => false, 'error' => 'No KMS configured'];
    }
    
    /**
     * Decrypt with KMS
     */
    private function decryptWithKMS($encryptedData) {
        if ($this->kmsConfig['aws_kms']['enabled']) {
            return $this->decryptWithAWSKMS($encryptedData);
        } elseif ($this->kmsConfig['azure_kms']['enabled']) {
            return $this->decryptWithAzureKMS($encryptedData);
        } elseif ($this->kmsConfig['google_kms']['enabled']) {
            return $this->decryptWithGoogleKMS($encryptedData);
        }
        
        return ['success' => false, 'error' => 'No KMS configured'];
    }
    
    /**
     * AWS KMS encryption (placeholder)
     */
    private function encryptWithAWSKMS($data) {
        // In production, use AWS SDK for PHP
        // $kms = new Aws\Kms\KmsClient([...]);
        // $result = $kms->encrypt([...]);
        
        error_log("AWS KMS encryption not implemented yet");
        return ['success' => false, 'error' => 'AWS KMS not implemented'];
    }
    
    /**
     * AWS KMS decryption (placeholder)
     */
    private function decryptWithAWSKMS($encryptedData) {
        // In production, use AWS SDK for PHP
        error_log("AWS KMS decryption not implemented yet");
        return ['success' => false, 'error' => 'AWS KMS not implemented'];
    }
    
    /**
     * Azure Key Vault encryption (placeholder)
     */
    private function encryptWithAzureKMS($data) {
        // In production, use Azure Key Vault SDK
        error_log("Azure KMS encryption not implemented yet");
        return ['success' => false, 'error' => 'Azure KMS not implemented'];
    }
    
    /**
     * Azure Key Vault decryption (placeholder)
     */
    private function decryptWithAzureKMS($encryptedData) {
        // In production, use Azure Key Vault SDK
        error_log("Azure KMS decryption not implemented yet");
        return ['success' => false, 'error' => 'Azure KMS not implemented'];
    }
    
    /**
     * Google Cloud KMS encryption (placeholder)
     */
    private function encryptWithGoogleKMS($data) {
        // In production, use Google Cloud KMS client
        error_log("Google KMS encryption not implemented yet");
        return ['success' => false, 'error' => 'Google KMS not implemented'];
    }
    
    /**
     * Google Cloud KMS decryption (placeholder)
     */
    private function decryptWithGoogleKMS($encryptedData) {
        // In production, use Google Cloud KMS client
        error_log("Google KMS decryption not implemented yet");
        return ['success' => false, 'error' => 'Google KMS not implemented'];
    }
    
    /**
     * Convert private key to public key (simplified - use proper cryptographic libraries in production)
     */
    private function privateKeyToPublicKey($privateKey, $network) {
        // This is a simplified implementation
        // In production, use proper cryptographic libraries like phpseclib
        
        if ($network === 'ethereum') {
            // Use secp256k1 curve for Ethereum
            // This is a placeholder - implement proper ECDSA key derivation
            return '0x' . hash('sha256', $privateKey . 'ethereum-public-key');
        }
        
        return hash('sha256', $privateKey . $network . 'public-key');
    }
    
    /**
     * Convert public key to address
     */
    private function publicKeyToAddress($publicKey, $network) {
        if ($network === 'ethereum') {
            // Remove 0x prefix and take last 40 characters
            $publicKey = str_replace('0x', '', $publicKey);
            return '0x' . substr($publicKey, -40);
        }
        
        return hash('sha256', $publicKey . $network . 'address');
    }
    
    /**
     * Convert private key to Tron address
     */
    private function privateKeyToTronAddress($privateKey) {
        // This is a simplified implementation
        // In production, use proper Tron address generation
        return 'T' . substr(hash('sha256', $privateKey . 'tron-address'), 0, 33);
    }
    
    /**
     * Convert private key to Solana address
     */
    private function privateKeyToSolanaAddress($privateKeyBytes) {
        // This is a simplified implementation
        // In production, use proper Solana key derivation
        return base58_encode(hash('sha256', $privateKeyBytes . 'solana-address', true));
    }
    
    /**
     * Store encrypted private key in database
     */
    public function storePrivateKey($userId, $network, $currency, $address, $encryptedPrivateKey, $publicKey = null) {
        try {
            $this->conn->beginTransaction();
            
            // Check if wallet already exists
            $stmt = $this->conn->prepare("SELECT id FROM crypto_wallets WHERE user_id = ? AND network = ? AND currency = ?");
            $stmt->execute([$userId, $network, $currency]);
            
            if ($stmt->fetch()) {
                $this->conn->rollBack();
                return ['success' => false, 'error' => 'Wallet already exists for this user, network, and currency'];
            }
            
            // Insert new wallet
            $stmt = $this->conn->prepare("
                INSERT INTO crypto_wallets (user_id, network, currency, address, private_key, qr_code, balance, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
            ");
            
            $qrCode = $this->generateQRCode($address);
            
            $stmt->execute([
                $userId,
                $network,
                $currency,
                $address,
                $encryptedPrivateKey,
                $qrCode
            ]);
            
            $walletId = $this->conn->lastInsertId();
            
            // Log the wallet creation
            $this->logWalletCreation($userId, $walletId, $network, $currency, $address);
            
            $this->conn->commit();
            
            return [
                'success' => true,
                'wallet_id' => $walletId,
                'address' => $address,
                'qr_code' => $qrCode
            ];
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Wallet storage error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Storage failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Retrieve and decrypt private key from database
     */
    public function getPrivateKey($walletId, $userId) {
        try {
            $stmt = $this->conn->prepare("
                SELECT private_key FROM crypto_wallets 
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([$walletId, $userId]);
            
            $result = $stmt->fetch();
            
            if (!$result) {
                return ['success' => false, 'error' => 'Wallet not found or access denied'];
            }
            
            $decryptionResult = $this->decryptPrivateKey($result['private_key'], $userId);
            
            if (!$decryptionResult['success']) {
                return ['success' => false, 'error' => 'Failed to decrypt private key: ' . $decryptionResult['error']];
            }
            
            // Log the private key access
            $this->logPrivateKeyAccess($userId, $walletId);
            
            return [
                'success' => true,
                'private_key' => $decryptionResult['private_key']
            ];
            
        } catch (Exception $e) {
            error_log("Private key retrieval error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Retrieval failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Generate QR code for address
     */
    private function generateQRCode($address) {
        return "https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=" . urlencode($address);
    }
    
    /**
     * Log wallet creation
     */
    private function logWalletCreation($userId, $walletId, $network, $currency, $address) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, new_values, created_at)
                VALUES (?, 'wallet_created', 'crypto_wallets', ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $newValues = json_encode([
                'wallet_id' => $walletId,
                'network' => $network,
                'currency' => $currency,
                'address' => $address,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            $stmt->execute([$userId, $walletId, $newValues]);
        } catch (Exception $e) {
            error_log("Wallet creation logging error: " . $e->getMessage());
        }
    }
    
    /**
     * Log private key access
     */
    private function logPrivateKeyAccess($userId, $walletId) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, created_at)
                VALUES (?, 'private_key_accessed', 'crypto_wallets', ?, CURRENT_TIMESTAMP)
            ");
            
            $stmt->execute([$userId, $walletId]);
        } catch (Exception $e) {
            error_log("Private key access logging error: " . $e->getMessage());
        }
    }
    
    /**
     * Validate private key format
     */
    public function validatePrivateKey($privateKey, $network) {
        switch ($network) {
            case 'ethereum':
            case 'bsc':
            case 'polygon':
            case 'base':
                // Ethereum private keys are 64 hex characters
                return [
                    'success' => preg_match('/^[a-fA-F0-9]{64}$/', $privateKey),
                    'message' => 'Ethereum private key must be 64 hex characters'
                ];
                
            case 'tron':
                // Tron private keys are 64 hex characters
                return [
                    'success' => preg_match('/^[a-fA-F0-9]{64}$/', $privateKey),
                    'message' => 'Tron private key must be 64 hex characters'
                ];
                
            case 'solana':
                // Solana private keys are base64 encoded 64 bytes
                return [
                    'success' => strlen(base64_decode($privateKey)) === 64,
                    'message' => 'Solana private key must be base64 encoded 64 bytes'
                ];
                
            default:
                return ['success' => false, 'error' => 'Unsupported network: ' . $network];
        }
    }
}

/**
 * Base58 encoding function (needed for Solana)
 */
function base58_encode($data) {
    $alphabet = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';
    $base = strlen($alphabet);
    
    // Convert to base58
    $encoded = '';
    $num = gmp_init('0x' . bin2hex($data));
    
    while (gmp_cmp($num, 0) > 0) {
        $remainder = gmp_mod($num, $base);
        $encoded = $alphabet[gmp_intval($remainder)] . $encoded;
        $num = gmp_div($num, $base);
    }
    
    // Add leading zeros
    for ($i = 0; $i < strlen($data) && $data[$i] === "\0"; $i++) {
        $encoded = '1' . $encoded;
    }
    
    return $encoded;
}
?>

