<?php

require_once 'config/database.php';

class LiquidityManager {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    /**
     * Monitor system wallet balances and trigger alerts
     */
    public function monitorBalances() {
        $conn = $this->db->getConnection();
        
        // Get all system wallets (user_id = 0)
        $stmt = $conn->prepare("
            SELECT currency, network, balance, address 
            FROM crypto_wallets 
            WHERE user_id = 0
        ");
        $stmt->execute();
        $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $alerts = [];
        
        foreach ($wallets as $wallet) {
            $balance = floatval($wallet['balance']);
            $currency = $wallet['currency'];
            
            // Define minimum thresholds
            $thresholds = [
                'USDT' => 1000,    // Minimum 1000 USDT
                'ETH' => 1,        // Minimum 1 ETH
                'BTC' => 0.01,     // Minimum 0.01 BTC
                'USDC' => 1000     // Minimum 1000 USDC
            ];
            
            if (isset($thresholds[$currency]) && $balance < $thresholds[$currency]) {
                $alerts[] = [
                    'currency' => $currency,
                    'network' => $wallet['network'],
                    'current_balance' => $balance,
                    'minimum_threshold' => $thresholds[$currency],
                    'address' => $wallet['address'],
                    'severity' => $balance < ($thresholds[$currency] * 0.5) ? 'critical' : 'warning'
                ];
            }
        }
        
        return $alerts;
    }
    
    /**
     * Get funding recommendations
     */
    public function getFundingRecommendations() {
        $alerts = $this->monitorBalances();
        $recommendations = [];
        
        foreach ($alerts as $alert) {
            $recommendations[] = [
                'action' => 'fund_wallet',
                'currency' => $alert['currency'],
                'network' => $alert['network'],
                'address' => $alert['address'],
                'recommended_amount' => $alert['minimum_threshold'] * 2, // Fund 2x minimum
                'priority' => $alert['severity'] === 'critical' ? 'high' : 'medium',
                'reason' => "Balance ({$alert['current_balance']}) below threshold ({$alert['minimum_threshold']})"
            ];
        }
        
        return $recommendations;
    }
    
    /**
     * Calculate optimal wallet distribution
     */
    public function calculateOptimalDistribution($totalUsdValue) {
        // Recommended distribution based on typical usage
        $distribution = [
            'USDT' => 0.4,  // 40% USDT
            'USDC' => 0.3,  // 30% USDC  
            'ETH' => 0.2,   // 20% ETH
            'BTC' => 0.1    // 10% BTC
        ];
        
        $allocations = [];
        foreach ($distribution as $currency => $percentage) {
            $allocations[$currency] = $totalUsdValue * $percentage;
        }
        
        return $allocations;
    }
    
    /**
     * Get system wallet summary
     */
    public function getSystemWalletSummary() {
        $conn = $this->db->getConnection();
        
        $stmt = $conn->prepare("
            SELECT 
                cw.currency,
                cw.network,
                cw.balance,
                cw.address,
                cp.buy_price,
                cp.sell_price
            FROM crypto_wallets cw
            LEFT JOIN crypto_prices cp ON cw.currency = cp.currency
            WHERE cw.user_id = 0
            ORDER BY cw.currency, cw.network
        ");
        $stmt->execute();
        $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $summary = [
            'total_wallets' => count($wallets),
            'wallets' => [],
            'total_usd_value' => 0,
            'low_balance_alerts' => 0
        ];
        
        foreach ($wallets as $wallet) {
            $usdValue = floatval($wallet['balance']) * floatval($wallet['buy_price']);
            $summary['total_usd_value'] += $usdValue;
            
            $walletData = [
                'currency' => $wallet['currency'],
                'network' => $wallet['network'],
                'balance' => floatval($wallet['balance']),
                'usd_value' => $usdValue,
                'address' => $wallet['address'],
                'buy_price' => floatval($wallet['buy_price']),
                'sell_price' => floatval($wallet['sell_price'])
            ];
            
            $summary['wallets'][] = $walletData;
            
            // Check for low balance
            $thresholds = [
                'USDT' => 1000,
                'ETH' => 1,
                'BTC' => 0.01,
                'USDC' => 1000
            ];
            
            if (isset($thresholds[$wallet['currency']]) && 
                floatval($wallet['balance']) < $thresholds[$wallet['currency']]) {
                $summary['low_balance_alerts']++;
            }
        }
        
        return $summary;
    }
    
    /**
     * Send funding alerts
     */
    public function sendFundingAlerts() {
        $alerts = $this->monitorBalances();
        
        if (empty($alerts)) {
            return ['status' => 'ok', 'message' => 'All wallets have sufficient balance'];
        }
        
        $criticalAlerts = array_filter($alerts, function($alert) {
            return $alert['severity'] === 'critical';
        });
        
        if (!empty($criticalAlerts)) {
            error_log("CRITICAL: System wallets need immediate funding: " . json_encode($criticalAlerts));
            // Here you could send email/SMS alerts
        }
        
        return [
            'status' => 'alert',
            'alerts' => $alerts,
            'recommendations' => $this->getFundingRecommendations()
        ];
    }
}

// API endpoint handler
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action'])) {
    $liquidityManager = new LiquidityManager();
    
    switch ($_GET['action']) {
        case 'monitor':
            echo json_encode($liquidityManager->monitorBalances());
            break;
            
        case 'recommendations':
            echo json_encode($liquidityManager->getFundingRecommendations());
            break;
            
        case 'summary':
            echo json_encode($liquidityManager->getSystemWalletSummary());
            break;
            
        case 'alerts':
            echo json_encode($liquidityManager->sendFundingAlerts());
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action']);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

