<?php
/**
 * Market Data API
 * Handles market prices, trading pairs, and coin information
 */

require_once __DIR__ . '/../config/database.php';

class MarketAPI {
    private $db;

    public function __construct() {
        $this->db = new Database();
    }

    public function getTradingPairs() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT tp.*, bc.symbol as base_symbol, bc.name as base_name, bc.logo_url as base_logo,
                       qc.symbol as quote_symbol, qc.name as quote_name, qc.logo_url as quote_logo,
                       mp.price, mp.volume_24h, mp.change_24h, mp.high_24h, mp.low_24h
                FROM trading_pairs tp
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                LEFT JOIN market_prices mp ON tp.id = mp.pair_id
                WHERE tp.is_active = 1
                ORDER BY mp.volume_24h DESC, bc.symbol
            ");
            $stmt->execute();
            $pairs = $stmt->fetchAll();

            return [
                'success' => true,
                'pairs' => $pairs
            ];
        } catch (Exception $e) {
            error_log("Get trading pairs error: " . $e->getMessage());
            return ['error' => 'Failed to get trading pairs'];
        }
    }

    public function getMarketPrices() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT mp.*, tp.symbol, bc.symbol as base_symbol, qc.symbol as quote_symbol
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                WHERE tp.is_active = 1
                ORDER BY mp.volume_24h DESC
            ");
            $stmt->execute();
            $prices = $stmt->fetchAll();

            return [
                'success' => true,
                'prices' => $prices
            ];
        } catch (Exception $e) {
            error_log("Get market prices error: " . $e->getMessage());
            return ['error' => 'Failed to get market prices'];
        }
    }

    public function getMarketOverview() {
        try {
            $conn = $this->db->getConnection();
            
            // Get top gainers
            $stmt = $conn->prepare("
                SELECT mp.*, tp.symbol, bc.symbol as base_symbol, qc.symbol as quote_symbol
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                WHERE tp.is_active = 1 AND mp.change_24h > 0
                ORDER BY mp.change_24h DESC
                LIMIT 10
            ");
            $stmt->execute();
            $top_gainers = $stmt->fetchAll();

            // Get top losers
            $stmt = $conn->prepare("
                SELECT mp.*, tp.symbol, bc.symbol as base_symbol, qc.symbol as quote_symbol
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                WHERE tp.is_active = 1 AND mp.change_24h < 0
                ORDER BY mp.change_24h ASC
                LIMIT 10
            ");
            $stmt->execute();
            $top_losers = $stmt->fetchAll();

            // Get most traded
            $stmt = $conn->prepare("
                SELECT mp.*, tp.symbol, bc.symbol as base_symbol, qc.symbol as quote_symbol
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                WHERE tp.is_active = 1
                ORDER BY mp.volume_24h DESC
                LIMIT 10
            ");
            $stmt->execute();
            $most_traded = $stmt->fetchAll();

            // Get market stats
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_pairs,
                    SUM(mp.volume_24h) as total_volume_24h,
                    AVG(mp.change_24h) as avg_change_24h
                FROM market_prices mp
                JOIN trading_pairs tp ON mp.pair_id = tp.id
                WHERE tp.is_active = 1
            ");
            $stmt->execute();
            $market_stats = $stmt->fetch();

            return [
                'success' => true,
                'top_gainers' => $top_gainers,
                'top_losers' => $top_losers,
                'most_traded' => $most_traded,
                'market_stats' => $market_stats
            ];
        } catch (Exception $e) {
            error_log("Get market overview error: " . $e->getMessage());
            return ['error' => 'Failed to get market overview'];
        }
    }

    public function getCoinDetails($pair_symbol) {
        try {
            $conn = $this->db->getConnection();
            
            // Get pair details
            $stmt = $conn->prepare("
                SELECT tp.*, bc.symbol as base_symbol, bc.name as base_name, bc.logo_url as base_logo,
                       qc.symbol as quote_symbol, qc.name as quote_name, qc.logo_url as quote_logo,
                       mp.price, mp.volume_24h, mp.change_24h, mp.high_24h, mp.low_24h, mp.updated_at
                FROM trading_pairs tp
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                LEFT JOIN market_prices mp ON tp.id = mp.pair_id
                WHERE tp.symbol = ? AND tp.is_active = 1
            ");
            $stmt->execute([$pair_symbol]);
            $pair = $stmt->fetch();

            if (!$pair) {
                return ['error' => 'Trading pair not found'];
            }

            // Get price history (last 24 hours)
            $stmt = $conn->prepare("
                SELECT price, created_at
                FROM price_history
                WHERE pair_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY created_at ASC
            ");
            $stmt->execute([$pair['id']]);
            $price_history = $stmt->fetchAll();

            // Get recent trades
            $stmt = $conn->prepare("
                SELECT o.*, u.first_name, u.last_name
                FROM orders o
                JOIN users u ON o.user_id = u.id
                WHERE o.pair_id = ? AND o.status = 'filled'
                ORDER BY o.updated_at DESC
                LIMIT 20
            ");
            $stmt->execute([$pair['id']]);
            $recent_trades = $stmt->fetchAll();

            // Get order book (simplified)
            $stmt = $conn->prepare("
                SELECT type, price, SUM(amount - filled_amount) as total_amount
                FROM orders
                WHERE pair_id = ? AND status = 'pending' AND side = 'limit'
                GROUP BY type, price
                ORDER BY price DESC
                LIMIT 20
            ");
            $stmt->execute([$pair['id']]);
            $order_book = $stmt->fetchAll();

            return [
                'success' => true,
                'pair' => $pair,
                'price_history' => $price_history,
                'recent_trades' => $recent_trades,
                'order_book' => $order_book
            ];
        } catch (Exception $e) {
            error_log("Get coin details error: " . $e->getMessage());
            return ['error' => 'Failed to get coin details'];
        }
    }

    public function updateMarketPrices() {
        try {
            $conn = $this->db->getConnection();
            
            // Get all active trading pairs
            $stmt = $conn->prepare("
                SELECT tp.id, tp.symbol, bc.symbol as base_symbol, qc.symbol as quote_symbol
                FROM trading_pairs tp
                JOIN cryptocurrencies bc ON tp.base_crypto_id = bc.id
                JOIN cryptocurrencies qc ON tp.quote_crypto_id = qc.id
                WHERE tp.is_active = 1
            ");
            $stmt->execute();
            $pairs = $stmt->fetchAll();

            $updated_count = 0;
            foreach ($pairs as $pair) {
                // Fetch price from external API (CoinMarketCap, CoinGecko, etc.)
                $price_data = $this->fetchExternalPrice($pair['base_symbol'], $pair['quote_symbol']);
                
                if ($price_data) {
                    $stmt = $conn->prepare("
                        INSERT INTO market_prices (pair_id, price, volume_24h, change_24h, high_24h, low_24h)
                        VALUES (?, ?, ?, ?, ?, ?)
                        ON DUPLICATE KEY UPDATE
                        price = VALUES(price),
                        volume_24h = VALUES(volume_24h),
                        change_24h = VALUES(change_24h),
                        high_24h = VALUES(high_24h),
                        low_24h = VALUES(low_24h),
                        updated_at = CURRENT_TIMESTAMP
                    ");
                    $stmt->execute([
                        $pair['id'],
                        $price_data['price'],
                        $price_data['volume_24h'],
                        $price_data['change_24h'],
                        $price_data['high_24h'],
                        $price_data['low_24h']
                    ]);
                    $updated_count++;
                }
            }

            return [
                'success' => true,
                'message' => "Updated $updated_count trading pairs",
                'updated_count' => $updated_count
            ];
        } catch (Exception $e) {
            error_log("Update market prices error: " . $e->getMessage());
            return ['error' => 'Failed to update market prices'];
        }
    }

    public function getSupportedCurrencies() {
        try {
            $conn = $this->db->getConnection();
            
            // Get fiat currencies
            $stmt = $conn->prepare("
                SELECT * FROM fiat_currencies 
                WHERE is_active = 1 
                ORDER BY code
            ");
            $stmt->execute();
            $fiat_currencies = $stmt->fetchAll();

            // Get cryptocurrencies
            $stmt = $conn->prepare("
                SELECT * FROM cryptocurrencies 
                WHERE is_active = 1 
                ORDER BY symbol
            ");
            $stmt->execute();
            $crypto_currencies = $stmt->fetchAll();

            return [
                'success' => true,
                'fiat_currencies' => $fiat_currencies,
                'crypto_currencies' => $crypto_currencies
            ];
        } catch (Exception $e) {
            error_log("Get supported currencies error: " . $e->getMessage());
            return ['error' => 'Failed to get supported currencies'];
        }
    }

    private function fetchExternalPrice($base_symbol, $quote_symbol) {
        try {
            // Use real market data from CoinGecko API
            require_once __DIR__ . '/market_prices.php';
            $marketPrices = new MarketPrices();
            
            $priceData = $marketPrices->getMarketPrice($base_symbol, 'USD');
            if ($priceData && isset($priceData['price'])) {
                $basePriceUsd = floatval($priceData['price']);
            } else {
                // Fallback prices if API fails
                $fallbackPrices = [
                    'USDT' => 1.00,
                    'USDC' => 1.00,
                    'BTC' => 45000.00,
                    'ETH' => 3000.00
                ];
                $basePriceUsd = $fallbackPrices[$base_symbol] ?? 1.00;
            }
            
            // Get exchange rate for quote currency
            $exchangeRates = [
                'UGX' => 3700,
                'KES' => 150,
                'TZS' => 2500,
                'RWF' => 1200,
                'CDF' => 2500,
                'NGN' => 800,
                'USD' => 1
            ];
            
            $quoteRate = $exchangeRates[$quote_symbol] ?? 1.00;
            $price = $basePriceUsd * $quoteRate;
            
            return [
                'price' => $price,
                'volume_24h' => rand(1000000, 10000000), // This would come from real API
                'change_24h' => rand(-10, 10) + (rand(0, 99) / 100), // This would come from real API
                'high_24h' => $price * (1 + rand(1, 5) / 100), // This would come from real API
                'low_24h' => $price * (1 - rand(1, 5) / 100) // This would come from real API
            ];
            
        } catch (Exception $e) {
            error_log("Error fetching external price: " . $e->getMessage());
            // Return fallback price if API fails
            $base_price = [
                'USDT' => 1.00,
                'USDC' => 1.00,
                'BTC' => 45000.00,
                'ETH' => 3000.00
            ];

            $quote_rate = [
                'UGX' => 3700,
                'KES' => 150,
                'TZS' => 2500,
                'RWF' => 1200,
                'CDF' => 2500,
                'NGN' => 800,
                'USD' => 1
            ];

            if (isset($base_price[$base_symbol]) && isset($quote_rate[$quote_symbol])) {
                $price = $base_price[$base_symbol] * $quote_rate[$quote_symbol];
                
                return [
                    'price' => $price,
                    'volume_24h' => rand(1000000, 10000000),
                    'change_24h' => rand(-10, 10) + (rand(0, 99) / 100),
                    'high_24h' => $price * (1 + rand(1, 5) / 100),
                    'low_24h' => $price * (1 - rand(1, 5) / 100)
                ];
            }

            return null;
        }
    }
}
?>
