<?php
/**
 * Market Prices API
 * Handles real-time cryptocurrency price fetching and caching
 * Similar to Yellow Card's approach - fetches market prices and applies spreads
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/env.php';

class MarketPricesAPI {
    private $conn;
    private $apiKeys;
    
    // CoinGecko API endpoints
    private $coingeckoBaseUrl = 'https://api.coingecko.com/api/v3';
    
    // Crypto mapping for CoinGecko
    private $cryptoMapping = [
        'USDT' => 'tether',
        'USDC' => 'usd-coin', 
        'BTC' => 'bitcoin',
        'ETH' => 'ethereum',
        'BNB' => 'binancecoin',
        'ADA' => 'cardano',
        'SOL' => 'solana',
        'MATIC' => 'matic-network',
        'TRX' => 'tron'
    ];
    
    // Currency mapping for CoinGecko
    private $currencyMapping = [
        'USD' => 'usd',
        'UGX' => 'usd', // CoinGecko doesn't support UGX directly, we'll convert from USD
        'KES' => 'usd',
        'TZS' => 'usd', 
        'RWF' => 'usd',
        'CDF' => 'usd',
        'NGN' => 'usd'
    ];

    public function __construct($connection) {
        $this->conn = $connection;
        $this->initializeAPIKeys();
    }

    /**
     * Initialize API keys from environment
     */
    private function initializeAPIKeys() {
        EnvConfig::load();
        $this->apiKeys = [
            'exchangerate' => EnvConfig::get('EXCHANGERATE_API_KEY', ''),
        ];
    }

    /**
     * Get current market price for a crypto/currency pair
     * Uses caching to avoid excessive API calls
     */
    public function getMarketPrice($crypto, $currency) {
        try {
            // Check cache first
            $cachedPrice = $this->getCachedPrice($crypto, $currency);
            if ($cachedPrice && $this->isCacheValid($cachedPrice)) {
                return [
                    'success' => true,
                    'price' => $cachedPrice['price_local'],
                    'price_usd' => $cachedPrice['price_usd'],
                    'exchange_rate' => $cachedPrice['exchange_rate'],
                    'source' => 'cache',
                    'last_updated' => $cachedPrice['last_updated']
                ];
            }

            // Fetch fresh price
            $freshPrice = $this->fetchFreshPrice($crypto, $currency);
            if ($freshPrice['success']) {
                // Cache the result
                $this->cachePrice($crypto, $currency, $freshPrice);
                return $freshPrice;
            }

            // Fallback to cached price even if expired
            if ($cachedPrice) {
                return [
                    'success' => true,
                    'price' => $cachedPrice['price_local'],
                    'price_usd' => $cachedPrice['price_usd'],
                    'exchange_rate' => $cachedPrice['exchange_rate'],
                    'source' => 'cache_expired',
                    'last_updated' => $cachedPrice['last_updated']
                ];
            }

            return ['success' => false, 'error' => 'Unable to fetch market price'];

        } catch (Exception $e) {
            error_log("Market price fetch error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Market price fetch failed'];
        }
    }

    /**
     * Fetch fresh price from CoinGecko
     */
    private function fetchFreshPrice($crypto, $currency) {
        try {
            $coingeckoId = $this->cryptoMapping[$crypto] ?? null;
            if (!$coingeckoId) {
                return ['success' => false, 'error' => 'Unsupported cryptocurrency'];
            }

            // Get USD price from CoinGecko
            $url = "{$this->coingeckoBaseUrl}/simple/price?ids={$coingeckoId}&vs_currencies=usd";
            $response = $this->makeHTTPRequest($url);
            
            if (!$response['success']) {
                return ['success' => false, 'error' => 'Failed to fetch from CoinGecko'];
            }

            $data = json_decode($response['data'], true);
            if (!isset($data[$coingeckoId]['usd'])) {
                return ['success' => false, 'error' => 'Invalid response from CoinGecko'];
            }

            $priceUsd = $data[$coingeckoId]['usd'];
            
            // Get exchange rate for local currency
            $exchangeRate = $this->getExchangeRate($currency);
            if (!$exchangeRate['success']) {
                return ['success' => false, 'error' => 'Failed to get exchange rate'];
            }

            $priceLocal = $priceUsd * $exchangeRate['rate'];

            return [
                'success' => true,
                'price' => $priceLocal,
                'price_usd' => $priceUsd,
                'exchange_rate' => $exchangeRate['rate'],
                'source' => 'coingecko'
            ];

        } catch (Exception $e) {
            error_log("Fresh price fetch error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to fetch fresh price'];
        }
    }

    /**
     * Get exchange rate for local currency
     */
    private function getExchangeRate($currency) {
        if ($currency === 'USD') {
            return ['success' => true, 'rate' => 1.0];
        }

        try {
            // Try exchangerate-api.com first if API key is available
            $apiKey = $this->apiKeys['exchangerate'] ?? '';
            if (!empty($apiKey) && $apiKey !== 'your_exchangerate_api_key') {
                $url = "https://v6.exchangerate-api.com/v6/{$apiKey}/latest/USD";
                
                $response = $this->makeHTTPRequest($url);
                if ($response['success']) {
                    $data = json_decode($response['data'], true);
                    if (isset($data['conversion_rates'][$currency])) {
                        return [
                            'success' => true,
                            'rate' => $data['conversion_rates'][$currency]
                        ];
                    }
                }
            }
            
            // Fallback: try free exchangerate-api without key
            $url = "https://api.exchangerate-api.com/v4/latest/USD";
            $response = $this->makeHTTPRequest($url);
            if ($response['success']) {
                $data = json_decode($response['data'], true);
                if (isset($data['rates'][$currency])) {
                    return [
                        'success' => true,
                        'rate' => $data['rates'][$currency]
                    ];
                }
            }

        } catch (Exception $e) {
            error_log("Exchange rate fetch error: " . $e->getMessage());
            
            // Fallback to hardcoded rates (should be updated regularly)
            $fallbackRates = [
                'UGX' => 3700,
                'KES' => 130,
                'TZS' => 2400,
                'RWF' => 1200,
                'CDF' => 2800,
                'NGN' => 1500
            ];

            if (isset($fallbackRates[$currency])) {
                return ['success' => true, 'rate' => $fallbackRates[$currency]];
            }

            return ['success' => false, 'error' => 'Exchange rate not available'];
        }
    }

    /**
     * Get cached price from database
     */
    private function getCachedPrice($crypto, $currency) {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM market_prices 
                WHERE crypto = ? AND currency = ?
            ");
            $stmt->execute([$crypto, $currency]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Cache retrieval error: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Check if cached price is still valid
     */
    private function isCacheValid($cachedPrice) {
        $expiresAt = strtotime($cachedPrice['expires_at']);
        return time() < $expiresAt;
    }

    /**
     * Cache price in database
     */
    private function cachePrice($crypto, $currency, $priceData) {
        try {
            $stmt = $this->conn->prepare("
                INSERT OR REPLACE INTO market_prices 
                (crypto, currency, price_usd, price_local, exchange_rate, source, last_updated, expires_at)
                VALUES (?, ?, ?, ?, ?, ?, datetime('now'), datetime('now', '+5 minutes'))
            ");
            $stmt->execute([
                $crypto,
                $currency,
                $priceData['price_usd'],
                $priceData['price'],
                $priceData['exchange_rate'],
                $priceData['source']
            ]);
        } catch (Exception $e) {
            error_log("Price caching error: " . $e->getMessage());
        }
    }

    /**
     * Make HTTP request with error handling
     */
    private function makeHTTPRequest($url, $timeout = 5) {
        try {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => $timeout,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_SSL_VERIFYPEER => true,
                CURLOPT_USERAGENT => 'YELLOW-Crypto-Exchange/1.0'
            ]);

            $data = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            if ($error) {
                return ['success' => false, 'error' => $error];
            }

            if ($httpCode !== 200) {
                return ['success' => false, 'error' => "HTTP {$httpCode}"];
            }

            return ['success' => true, 'data' => $data];

        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Get multiple market prices at once
     */
    public function getMultiplePrices($cryptoCurrencyPairs) {
        $results = [];
        
        foreach ($cryptoCurrencyPairs as $pair) {
            $results[] = [
                'crypto' => $pair['crypto'],
                'currency' => $pair['currency'],
                'data' => $this->getMarketPrice($pair['crypto'], $pair['currency'])
            ];
        }
        
        return $results;
    }

    /**
     * Update all cached prices (for cron jobs)
     */
    public function updateAllPrices() {
        try {
            $stmt = $this->conn->prepare("
                SELECT DISTINCT crypto, currency FROM pricing_spreads 
                WHERE is_active = 1
            ");
            $stmt->execute();
            $pairs = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $updated = 0;
            foreach ($pairs as $pair) {
                $result = $this->getMarketPrice($pair['crypto'], $pair['currency']);
                if ($result['success']) {
                    $updated++;
                }
            }

            return [
                'success' => true,
                'updated' => $updated,
                'total' => count($pairs)
            ];

        } catch (Exception $e) {
            error_log("Bulk price update error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Bulk update failed'];
        }
    }
}
?>
