<?php
/**
 * YELLOW Crypto Exchange - Monetization System
 * Comprehensive revenue management and fee calculation
 */

class MonetizationSystem {
    private $conn;
    
    // Optimized fee structure: Beat Yellow Card, More Profitable than Binance
    private $tradingFees = [
        'maker_fee' => 0.02,   // 2% (Yellow Card charges 2-3%, we're competitive)
        'taker_fee' => 0.02,   // 2% (still better than Yellow Card's 3%)
        'conversion_fee' => 0.015, // 1.5% for conversions (Yellow Card: 2%)
        'spread_markup' => 0.025   // 2.5% spread built into prices (Yellow Card: 3-5%)
    ];
    
    // Network-based withdrawal fees (Binance-exact fees)
    private $withdrawalFees = [
        'BTC' => 0.0005,      // Binance: 0.0005 BTC (~$31)
        'ETH' => 0.005,       // Binance: 0.005 ETH (~$12)
        'USDT' => 1.0,        // Binance: 1 USDT (TRC20), varies by network
        'USDC' => 1.0,        // Binance: ~1 USDC
        'BNB' => 0.0005,      // Binance: 0.0005 BNB (~$0.30)
        'TRX' => 1.0,         // Binance: 1 TRX
        'SOL' => 0.01,        // Binance: 0.01 SOL (~$1.80)
        'MATIC' => 0.1,       // Binance: 0.1 MATIC (~$0.10)
        'CELO' => 0.01        // Binance: 0.01 CELO
    ];
    
    // Network-specific withdrawal fees (Profitable based on real costs)
    private $networkWithdrawalFees = [
        'trc20' => 2.0,       // Real cost: ~$1.35 (6.7735 TRX), Profit: $0.65 ✅
        'erc20' => 12.0,      // Real cost: ~$8-10, Profit: $2-4 ✅
        'bep20' => 1.0,       // Real cost: ~$0.50, Profit: $0.50 ✅
        'polygon' => 0.2,     // Real cost: ~$0.05, Profit: $0.15 ✅
        'solana' => 0.1,      // Real cost: ~$0.001, Profit: $0.099 ✅
        'base' => 0.2,        // Real cost: ~$0.05, Profit: $0.15 ✅
        'arbitrum' => 1.0,    // Real cost: ~$0.50, Profit: $0.50 ✅
        'optimism' => 1.0     // Real cost: ~$0.50, Profit: $0.50 ✅
    ];
    
    // Removed premium subscriptions - not suitable for target market
    
    // Removed staking - not suitable for target market
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->ensureTablesExist();
    }
    
    /**
     * Calculate trading fees (Binance-style 0.1% flat rate)
     */
    public function calculateTradingFees($userId, $amount, $isMaker = false) {
        $feeRate = $isMaker ? $this->tradingFees['maker_fee'] : $this->tradingFees['taker_fee'];
        $fee = $amount * $feeRate;
        
        return [
            'fee' => $fee,
            'fee_usd' => $fee, // Assuming amount is in USD
            'net_amount' => $amount - $fee,
            'fee_rate' => $feeRate,
            'fee_percentage' => ($feeRate * 100) . '%'
        ];
    }
    
    /**
     * Calculate withdrawal fees (Binance-style network-based)
     */
    public function calculateWithdrawalFees($userId, $amount, $currency, $network = null) {
        // Get network-specific fee if network is provided
        if ($network && isset($this->networkWithdrawalFees[$network])) {
            $fee = $this->networkWithdrawalFees[$network];
        } else {
            // Fallback to currency-based fee
            $fee = $this->withdrawalFees[$currency] ?? 0.001;
        }
        
        return [
            'fee' => $fee,
            'net_amount' => $amount - $fee,
            'currency' => $currency,
            'network' => $network
        ];
    }
    
    /**
     * Calculate conversion fees (Binance-style 0.1%)
     */
    public function calculateConversionFees($userId, $amount, $fromCurrency, $toCurrency) {
        $feeRate = $this->tradingFees['conversion_fee'];
        $fee = $amount * $feeRate;
        
        return [
            'fee_amount' => $fee,
            'fee_rate' => $feeRate,
            'fee_percentage' => ($feeRate * 100) . '%',
            'net_amount' => $amount - $fee,
            'from_currency' => $fromCurrency,
            'to_currency' => $toCurrency
        ];
    }
    
    /**
     * Record fee collection and transfer to dedicated fee wallet
     */
    public function recordFee($userId, $type, $amount, $currency, $transactionId = null, $network = null) {
        try {
            // Record in fee_collections table for tracking
            $stmt = $this->conn->prepare("
                INSERT INTO fee_collections (
                    user_id, type, amount, currency, transaction_id, collected_at
                ) VALUES (?, ?, ?, ?, ?, datetime('now', 'localtime'))
            ");
            
            $stmt->execute([$userId, $type, $amount, $currency, $transactionId]);
            
            // Auto-transfer fee to dedicated fee wallet (user_id = -1)
            if ($network && $amount > 0) {
                $this->transferFeeToFeeWallet($amount, $currency, $network, $type);
            }
            
            return [
                'success' => true,
                'fee_id' => $this->conn->lastInsertId()
            ];
        } catch (Exception $e) {
            error_log("Fee recording error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Transfer collected fee to dedicated fee wallet (user_id = -1)
     */
    private function transferFeeToFeeWallet($amount, $currency, $network, $type) {
        try {
            // Get fee wallet for this network (user_id = -1)
            $stmt = $this->conn->prepare("
                SELECT id FROM crypto_wallets 
                WHERE user_id = -1 AND currency = ? AND network = ?
                LIMIT 1
            ");
            $stmt->execute([$currency, strtolower($network)]);
            $feeWallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($feeWallet) {
                // Credit the fee wallet
                $stmt = $this->conn->prepare("
                    UPDATE crypto_wallets 
                    SET balance = balance + ?, updated_at = datetime('now', 'localtime')
                    WHERE id = ?
                ");
                $stmt->execute([$amount, $feeWallet['id']]);
                
                error_log("✅ Fee transferred to fee wallet: {$amount} {$currency} ({$network}) - Type: {$type}");
            } else {
                error_log("⚠️ No fee wallet found for {$currency} on {$network}");
            }
        } catch (Exception $e) {
            error_log("Fee transfer error (non-critical): " . $e->getMessage());
            // Don't fail the main transaction if fee transfer fails
        }
    }
    
    /**
     * Get user's trading tier based on 30-day volume
     */
    public function getUserTier($userId) {
        $volume = $this->getUserTradingVolume($userId, 30);
        
        foreach ($this->tradingTiers as $tierKey => $tier) {
            if ($volume >= $tier['min_volume'] && $volume < $tier['max_volume']) {
                return $tier;
            }
        }
        
        return $this->tradingTiers['diamond']; // Default to highest tier
    }
    
    // Removed staking rewards calculation - not suitable for target market
    
    /**
     * Process referral commission based on transaction fees
     */
    public function processReferralCommission($referrerId, $refereeId, $transactionFee) {
        $commissionRate = 0.05; // 5% commission on transaction fees
        $commission = $transactionFee * $commissionRate;
        
        // Credit referrer's account
        $this->creditCommission($referrerId, $commission, 'referral_commission');
        
        return [
            'referrer_id' => $referrerId,
            'referee_id' => $refereeId,
            'commission' => $commission,
            'fee_used' => $transactionFee,
            'rate' => $commissionRate
        ];
    }
    
    /**
     * Get revenue analytics
     */
    public function getRevenueAnalytics($period = 30) {
        $startDate = date('Y-m-d H:i:s', strtotime("-{$period} days"));
        
        $revenue = [
            'trading_fees' => $this->getTradingFeesRevenue($startDate),
            'withdrawal_fees' => $this->getWithdrawalFeesRevenue($startDate),
            'deposit_fees' => $this->getDepositFeesRevenue($startDate),
            'conversion_fees' => $this->getConversionFeesRevenue($startDate),
            'referral_commissions_paid' => $this->getReferralCommissionsPaid($startDate),
            'total_revenue' => 0
        ];
        
        $revenue['total_revenue'] = array_sum($revenue);
        
        return $revenue;
    }
    
    /**
     * Private helper methods
     */
    private function getUserTradingVolume($userId, $days) {
        $startDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(amount), 0) as volume 
            FROM transactions 
            WHERE user_id = ? AND type IN ('buy', 'sell') AND created_at >= ?
        ");
        $stmt->execute([$userId, $startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['volume']);
    }
    
    private function getNetworkFee($currency) {
        $networkFees = [
            'BTC' => 0.0001,
            'ETH' => 0.005,
            'USDT' => 0.001,
            'USDC' => 0.001,
            'BNB' => 0.0005,
            'ADA' => 1.0,
            'SOL' => 0.001,
            'MATIC' => 0.1
        ];
        
        return $networkFees[$currency] ?? 0.001;
    }
    
    private function creditCommission($userId, $amount, $type) {
        $stmt = $this->conn->prepare("
            INSERT INTO commission_transactions (user_id, amount, type, created_at) 
            VALUES (?, ?, ?, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$userId, $amount, $type]);
        
        // Update user balance
        $stmt = $this->conn->prepare("
            UPDATE user_fiat_balances 
            SET balance = balance + ? 
            WHERE user_id = ? AND currency_code = 'USD'
        ");
        $stmt->execute([$amount, $userId]);
    }
    
    private function getTradingFeesRevenue($startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(fee), 0) as revenue 
            FROM transactions 
            WHERE type IN ('buy', 'sell') AND created_at >= ?
        ");
        $stmt->execute([$startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['revenue']);
    }
    
    private function getWithdrawalFeesRevenue($startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(fee), 0) as revenue 
            FROM transactions 
            WHERE type = 'withdrawal' AND created_at >= ?
        ");
        $stmt->execute([$startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['revenue']);
    }
    
    private function getDepositFeesRevenue($startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(fee), 0) as revenue 
            FROM transactions 
            WHERE type = 'deposit' AND created_at >= ?
        ");
        $stmt->execute([$startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['revenue']);
    }
    
    private function getConversionFeesRevenue($startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(fee), 0) as revenue 
            FROM transactions 
            WHERE type = 'convert' AND created_at >= ?
        ");
        $stmt->execute([$startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['revenue']);
    }
    
    private function getReferralCommissionsPaid($startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(amount), 0) as paid 
            FROM commission_transactions 
            WHERE type = 'referral_commission' AND created_at >= ?
        ");
        $stmt->execute([$startDate]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['paid']);
    }
    
    /**
     * Get total fee revenue (Binance-style dashboard)
     */
    public function getTotalFeeRevenue($startDate = null, $endDate = null) {
        try {
            $query = "
                SELECT 
                    SUM(amount) as total_fees,
                    currency,
                    COUNT(*) as transaction_count
                FROM fee_collections
            ";
            
            $params = [];
            if ($startDate) {
                $query .= " WHERE collected_at >= ?";
                $params[] = $startDate;
                
                if ($endDate) {
                    $query .= " AND collected_at <= ?";
                    $params[] = $endDate;
                }
            }
            
            $query .= " GROUP BY currency";
            
            $stmt = $this->conn->prepare($query);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Fee revenue query error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get fee revenue by type
     */
    public function getFeeRevenueByType($startDate = null) {
        try {
            $query = "
                SELECT 
                    type,
                    SUM(amount) as total,
                    COUNT(*) as count,
                    currency
                FROM fee_collections
            ";
            
            if ($startDate) {
                $query .= " WHERE collected_at >= ?";
                $stmt = $this->conn->prepare($query . " GROUP BY type, currency");
                $stmt->execute([$startDate]);
            } else {
                $stmt = $this->conn->prepare($query . " GROUP BY type, currency");
                $stmt->execute();
            }
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Fee revenue by type error: " . $e->getMessage());
            return [];
        }
    }
    
    private function ensureTablesExist() {
        // Fee collections table (Binance-style)
        $this->conn->exec("
            CREATE TABLE IF NOT EXISTS fee_collections (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                type TEXT NOT NULL,
                amount REAL NOT NULL,
                currency TEXT NOT NULL,
                transaction_id TEXT,
                collected_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // Commission transactions table
        $this->conn->exec("
            CREATE TABLE IF NOT EXISTS commission_transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                amount REAL NOT NULL,
                type TEXT NOT NULL,
                description TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // Removed staking and subscription tables - not suitable for target market
    }
}
?>
