<?php
/**
 * Multi-Signature Wallet System
 * Enhanced security for system funds
 */

class MultiSigWallet {
    private $db;
    private $required_signatures = 3; // Require 3 signatures for large transactions
    private $threshold_amount = 1000; // USD - transactions above this require multi-sig
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * Check if transaction requires multi-signature approval
     */
    public function requiresMultiSig($amount, $currency) {
        $usd_amount = $this->convertToUSD($amount, $currency);
        return $usd_amount >= $this->threshold_amount;
    }
    
    /**
     * Create multi-signature transaction request
     */
    public function createMultiSigRequest($user_id, $transaction_type, $amount, $currency, $details) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();
            
            // Create the multi-sig request
            $stmt = $conn->prepare("
                INSERT INTO multisig_requests (
                    user_id, transaction_type, amount, currency, 
                    details, status, required_signatures, created_at
                ) VALUES (?, ?, ?, ?, ?, 'pending', ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([
                $user_id, $transaction_type, $amount, $currency,
                json_encode($details), $this->required_signatures
            ]);
            
            $request_id = $conn->lastInsertId();
            
            // Notify administrators
            $this->notifyAdministrators($request_id, $transaction_type, $amount, $currency);
            
            $conn->commit();
            
            return [
                'success' => true,
                'request_id' => $request_id,
                'message' => 'Multi-signature request created. Awaiting admin approval.',
                'required_signatures' => $this->required_signatures
            ];
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Multi-sig request creation failed: " . $e->getMessage());
            return ['error' => 'Failed to create multi-signature request'];
        }
    }
    
    /**
     * Approve multi-signature transaction
     */
    public function approveTransaction($request_id, $admin_id, $signature) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();
            
            // Verify admin has permission to sign
            if (!$this->canAdminSign($admin_id)) {
                throw new Exception("Admin not authorized to sign transactions");
            }
            
            // Check if already signed by this admin
            $stmt = $conn->prepare("
                SELECT id FROM multisig_signatures 
                WHERE request_id = ? AND admin_id = ?
            ");
            $stmt->execute([$request_id, $admin_id]);
            
            if ($stmt->fetch()) {
                throw new Exception("Transaction already signed by this admin");
            }
            
            // Add signature
            $stmt = $conn->prepare("
                INSERT INTO multisig_signatures (
                    request_id, admin_id, signature, created_at
                ) VALUES (?, ?, ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$request_id, $admin_id, $signature]);
            
            // Check if we have enough signatures
            $stmt = $conn->prepare("
                SELECT COUNT(*) as signature_count
                FROM multisig_signatures 
                WHERE request_id = ?
            ");
            $stmt->execute([$request_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result['signature_count'] >= $this->required_signatures) {
                // Execute the transaction
                $this->executeMultiSigTransaction($request_id);
            }
            
            $conn->commit();
            
            return [
                'success' => true,
                'message' => 'Signature added successfully',
                'signatures_count' => $result['signature_count'],
                'required_signatures' => $this->required_signatures
            ];
            
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Multi-sig approval failed: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Reject multi-signature transaction
     */
    public function rejectTransaction($request_id, $admin_id, $reason) {
        try {
            $conn = $this->db->getConnection();
            
            // Verify admin has permission
            if (!$this->canAdminSign($admin_id)) {
                throw new Exception("Admin not authorized to reject transactions");
            }
            
            // Update request status
            $stmt = $conn->prepare("
                UPDATE multisig_requests 
                SET status = 'rejected', rejected_by = ?, rejection_reason = ?, 
                    rejected_at = CURRENT_TIMESTAMP
                WHERE id = ? AND status = 'pending'
            ");
            $stmt->execute([$admin_id, $reason, $request_id]);
            
            if ($stmt->rowCount() === 0) {
                throw new Exception("Transaction not found or already processed");
            }
            
            // Notify user of rejection
            $this->notifyUserRejection($request_id, $reason);
            
            return [
                'success' => true,
                'message' => 'Transaction rejected successfully'
            ];
            
        } catch (Exception $e) {
            error_log("Multi-sig rejection failed: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Execute approved multi-signature transaction
     */
    private function executeMultiSigTransaction($request_id) {
        try {
            $conn = $this->db->getConnection();
            
            // Get transaction details
            $stmt = $conn->prepare("
                SELECT * FROM multisig_requests WHERE id = ?
            ");
            $stmt->execute([$request_id]);
            $request = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$request) {
                throw new Exception("Multi-sig request not found");
            }
            
            // Update status to executing
            $stmt = $conn->prepare("
                UPDATE multisig_requests 
                SET status = 'executing', executed_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$request_id]);
            
            // Execute based on transaction type
            $details = json_decode($request['details'], true);
            
            switch ($request['transaction_type']) {
                case 'crypto_buy':
                    $result = $this->executeCryptoBuy($request['user_id'], $request['amount'], $request['currency'], $details);
                    break;
                case 'crypto_sell':
                    $result = $this->executeCryptoSell($request['user_id'], $request['amount'], $request['currency'], $details);
                    break;
                default:
                    throw new Exception("Unknown transaction type: " . $request['transaction_type']);
            }
            
            if ($result['success']) {
                // Mark as completed
                $stmt = $conn->prepare("
                    UPDATE multisig_requests 
                    SET status = 'completed', completed_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                ");
                $stmt->execute([$request_id]);
                
                // Notify user of completion
                $this->notifyUserCompletion($request_id, $result);
            } else {
                // Mark as failed
                $stmt = $conn->prepare("
                    UPDATE multisig_requests 
                    SET status = 'failed', failure_reason = ?, failed_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                ");
                $stmt->execute([$result['error'] ?? 'Unknown error', $request_id]);
            }
            
        } catch (Exception $e) {
            error_log("Multi-sig execution failed: " . $e->getMessage());
            
            // Mark as failed
            $stmt = $conn->prepare("
                UPDATE multisig_requests 
                SET status = 'failed', failure_reason = ?, failed_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$e->getMessage(), $request_id]);
        }
    }
    
    /**
     * Get pending multi-signature requests
     */
    public function getPendingRequests($admin_id = null) {
        try {
            $conn = $this->db->getConnection();
            
            $sql = "
                SELECT mr.*, 
                       COUNT(ms.id) as signature_count,
                       GROUP_CONCAT(a.username) as signers
                FROM multisig_requests mr
                LEFT JOIN multisig_signatures ms ON mr.id = ms.request_id
                LEFT JOIN admins a ON ms.admin_id = a.id
                WHERE mr.status = 'pending'
                GROUP BY mr.id
                ORDER BY mr.created_at DESC
            ";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute();
            $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Add user details
            foreach ($requests as &$request) {
                $stmt = $conn->prepare("
                    SELECT first_name, last_name, email FROM users WHERE id = ?
                ");
                $stmt->execute([$request['user_id']]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                $request['user'] = $user;
                
                // Check if current admin has signed
                if ($admin_id) {
                    $stmt = $conn->prepare("
                        SELECT id FROM multisig_signatures 
                        WHERE request_id = ? AND admin_id = ?
                    ");
                    $stmt->execute([$request['id'], $admin_id]);
                    $request['current_admin_signed'] = $stmt->fetch() !== false;
                }
            }
            
            return $requests;
            
        } catch (Exception $e) {
            error_log("Failed to get pending requests: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Helper methods
     */
    private function canAdminSign($admin_id) {
        // Check if admin has multi-sig signing permissions
        // This would integrate with your admin permission system
        return true; // Simplified for demo
    }
    
    private function convertToUSD($amount, $currency) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT buy_price FROM crypto_prices 
                WHERE crypto = ? AND is_active = 1 
                ORDER BY created_at DESC LIMIT 1
            ");
            $stmt->execute([$currency]);
            $price = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $price ? $amount * $price['buy_price'] : 0;
        } catch (Exception $e) {
            return 0;
        }
    }
    
    private function executeCryptoBuy($user_id, $amount, $currency, $details) {
        // This would call the actual crypto buy logic
        // For now, return success
        return ['success' => true, 'transaction_id' => 'multisig_' . time()];
    }
    
    private function executeCryptoSell($user_id, $amount, $currency, $details) {
        // This would call the actual crypto sell logic
        // For now, return success
        return ['success' => true, 'transaction_id' => 'multisig_' . time()];
    }
    
    private function notifyAdministrators($request_id, $transaction_type, $amount, $currency) {
        // Send notifications to administrators
        error_log("Multi-sig notification: Request $request_id for $transaction_type of $amount $currency");
    }
    
    private function notifyUserRejection($request_id, $reason) {
        // Send notification to user about rejection
        error_log("Multi-sig rejection notification: Request $request_id rejected - $reason");
    }
    
    private function notifyUserCompletion($request_id, $result) {
        // Send notification to user about completion
        error_log("Multi-sig completion notification: Request $request_id completed");
    }
}
?>

