<?php
require_once __DIR__ . '/../config/database.php';

class PaymentAPI {
    private $db;
    
    // Bitmuk API Configuration
    private $bitmukBaseUrl;
    private $bitmukApiUsername;
    private $bitmukApiKey;
    private $bitmukCallbackUrl;
    
    public function __construct() {
        $this->db = new Database();
        $this->initializeBitmukConfig();
    }
    
    /**
     * Initialize Bitmuk configuration from environment
     */
    private function initializeBitmukConfig() {
        require_once __DIR__ . '/../config/env.php';
        EnvConfig::load();
        
        $this->bitmukBaseUrl = EnvConfig::get('BITMUK_API_URL', 'https://bitmuk.host/openapi');
        $this->bitmukApiUsername = EnvConfig::get('BITMUK_API_USERNAME', '1321076004');
        $this->bitmukApiKey = EnvConfig::get('BITMUK_API_KEY', '571D0F5E-61653462-3035-4761-b664-633032353366');
        $this->bitmukCallbackUrl = EnvConfig::get('BITMUK_CALLBACK_URL', 'https://proshimefx.com/callback');
    }
    
    /**
     * Initiate a mobile money payment through Bitmuk
     */
    public function initiatePayment($data) {
        try {
            // Validate required fields
            if (empty($data['phone']) || empty($data['amount']) || empty($data['reference'])) {
                return [
                    'success' => false,
                    'message' => 'Missing required fields: phone, amount, reference'
                ];
            }
            
            // Prepare payment data for Bitmuk
            $paymentData = [
                'paymentOption' => 'mobile_money',
                'phone' => $data['phone'],
                'name' => $data['name'] ?? 'Crypto Exchange User',
                'amount' => intval($data['amount']),
                'reference' => $data['reference'],
                'message' => $data['message'] ?? 'Crypto Exchange Deposit'
            ];
            
            // Make request to Bitmuk API
            $response = $this->makeBitmukRequest('/payment', $paymentData);
            
            if ($response['success']) {
                try {
                    // Store payment record in database
                    $this->storePaymentRecord($data, $response['data']);
                    
                    return [
                        'success' => true,
                        'message' => 'Payment initiated successfully',
                        'data' => [
                            'transactionId' => $data['reference'],
                            'status' => 'pending',
                            'bitmukResponse' => $response['data']
                        ]
                    ];
                } catch (Exception $dbError) {
                    error_log("Database error in payment initiation: " . $dbError->getMessage());
                    // Still return success since Bitmuk payment was initiated
                    return [
                        'success' => true,
                        'message' => 'Payment initiated successfully (database error logged)',
                        'data' => [
                            'transactionId' => $data['reference'],
                            'status' => 'pending',
                            'bitmukResponse' => $response['data']
                        ]
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to initiate payment: ' . $response['message']
                ];
            }
            
        } catch (Exception $e) {
            error_log("Payment initiation error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Payment initiation failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Check payment status from Bitmuk
     */
    public function checkPaymentStatus($transactionId) {
        try {
            // Set execution time limit to prevent timeout
            set_time_limit(30);
            
            $conn = $this->db->getConnection();
            
            // Set timeout for database operations
            $conn->setAttribute(PDO::ATTR_TIMEOUT, 5);
            
            // Get payment record
            $stmt = $conn->prepare("SELECT * FROM payments WHERE transaction_id = ?");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$payment) {
                return [
                    'success' => false,
                    'message' => 'Payment not found'
                ];
            }
            
            // If already completed or failed, return cached status
            if (in_array($payment['status'], ['completed', 'failed'])) {
                return [
                    'success' => true,
                    'data' => [
                        'status' => $payment['status'],
                        'message' => $payment['status'] === 'completed' ? 'Payment completed successfully' : 'Payment failed'
                    ]
                ];
            }
            
            // Check with Bitmuk API - use the correct endpoint
            $statusData = [
                'transactionId' => $payment['bitmuk_reference']
            ];
            
            $response = $this->makeBitmukRequest('/paymentStatus', $statusData);
            
            // Log the response for debugging
            error_log("Bitmuk status check response for {$payment['bitmuk_reference']}: " . json_encode($response));
            
            if ($response['success']) {
                $bitmukData = $response['data'];
                $bitmukStatus = $bitmukData['data']['transactionStatus'] ?? 'pending';
                
                // Log the parsed status
                error_log("Parsed Bitmuk status: {$bitmukStatus} for transaction: {$payment['bitmuk_reference']}");
                
                // Map Bitmuk status to our internal status
                $internalStatus = $bitmukStatus;
                if ($bitmukStatus === 'successful') {
                    $internalStatus = 'completed';
                }
                
                // Update local status if it's different
                if ($payment['status'] !== $internalStatus) {
                    $this->updatePaymentStatusInternal($transactionId, [
                        'status' => $internalStatus
                    ]);
                    
                    // If completed, credit user account
                    if ($internalStatus === 'completed') {
                        $this->creditUserAccount($transactionId);
                    }
                }
                
                return [
                    'success' => true,
                    'data' => [
                        'status' => $internalStatus,
                        'message' => $bitmukData['data']['transactionMessage'] ?? 'Payment status from Bitmuk'
                    ]
                ];
            } else {
                // If Bitmuk API fails, return current database status
                return [
                    'success' => true,
                    'data' => [
                        'status' => $payment['status'],
                        'message' => 'Payment status from database (Bitmuk API unavailable)'
                    ]
                ];
            }
            
        } catch (Exception $e) {
            error_log("Payment status check error: " . $e->getMessage());
            return [
                'success' => true,
                'data' => [
                    'status' => 'pending',
                    'message' => 'Payment is still being processed'
                ]
            ];
        }
    }
    

    /**
     * Make HTTP request to Bitmuk API
     */
    private function makeBitmukRequest($endpoint, $data) {
        $url = $this->bitmukBaseUrl . '/' . $this->bitmukApiUsername . $endpoint;
        
        $headers = [
            'api-key: ' . $this->bitmukApiKey,
            'Content-Type: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return [
                'success' => false,
                'message' => 'CURL Error: ' . $error
            ];
        }
        
        // Handle empty response
        if (empty($response)) {
            return [
                'success' => false,
                'message' => 'Empty response from Bitmuk API',
                'httpCode' => $httpCode
            ];
        }
        
        $decodedResponse = json_decode($response, true);
        
        // Check for JSON decode errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'message' => 'JSON decode error: ' . json_last_error_msg(),
                'httpCode' => $httpCode,
                'response' => $response
            ];
        }
        
        if ($httpCode === 200 && $decodedResponse) {
            // Check if Bitmuk returned success status
            if (isset($decodedResponse['status']) && $decodedResponse['status'] === 200) {
                return [
                    'success' => true,
                    'data' => $decodedResponse
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Bitmuk API returned error: ' . ($decodedResponse['statusMessage'] ?? 'Unknown error'),
                    'httpCode' => $httpCode,
                    'response' => $response,
                    'decodedResponse' => $decodedResponse
                ];
            }
        } else {
            return [
                'success' => false,
                'message' => 'API Error: ' . ($decodedResponse['message'] ?? 'Unknown error'),
                'httpCode' => $httpCode,
                'response' => $response,
                'decodedResponse' => $decodedResponse
            ];
        }
    }
    
    /**
     * Store payment record in database
     */
    private function storePaymentRecord($data, $bitmukResponse) {
        try {
            $conn = $this->db->getConnection();
            
            // First ensure the payments table exists
            $this->ensurePaymentsTableExists($conn);
            
            $stmt = $conn->prepare("
                INSERT INTO payments (
                    user_id, transaction_id, phone, amount, currency, 
                    payment_method, status, bitmuk_reference, 
                    created_at, updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, datetime('now'), datetime('now'))
            ");
            
            $stmt->execute([
                $data['user_id'] ?? 1, // Default user for testing
                $data['reference'],
                $data['phone'],
                $data['amount'],
                'UGX',
                'mobile_money',
                'pending',
                $bitmukResponse['data']['transactionId'] ?? $data['reference']
            ]);
            
        } catch (Exception $e) {
            error_log("Database error storing payment: " . $e->getMessage());
        }
    }
    
    /**
     * Ensure payments table exists
     */
    private function ensurePaymentsTableExists($conn) {
        try {
            $createTableSQL = "
            CREATE TABLE IF NOT EXISTS payments (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                transaction_id TEXT NOT NULL UNIQUE,
                phone TEXT NOT NULL,
                amount REAL NOT NULL,
                currency TEXT DEFAULT 'UGX',
                payment_method TEXT DEFAULT 'mobile_money',
                status TEXT DEFAULT 'pending',
                bitmuk_reference TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )";
            
            $conn->exec($createTableSQL);
        } catch (Exception $e) {
            error_log("Error creating payments table: " . $e->getMessage());
        }
    }
    
    /**
     * Update payment status (public method for manual updates)
     */
    public function updatePaymentStatus($transactionId, $status) {
        try {
            $conn = $this->db->getConnection();
            
            // Validate status
            $validStatuses = ['pending', 'completed', 'failed'];
            if (!in_array($status, $validStatuses)) {
                return [
                    'success' => false,
                    'message' => 'Invalid status. Must be one of: ' . implode(', ', $validStatuses)
                ];
            }
            
            // Update payment status
            $stmt = $conn->prepare("
                UPDATE payments 
                SET status = ?, updated_at = datetime('now') 
                WHERE transaction_id = ?
            ");
            
            $stmt->execute([$status, $transactionId]);
            
            // If marking as completed, credit the user's account
            if ($status === 'completed') {
                $this->creditUserAccount($transactionId);
            }
            
            return [
                'success' => true,
                'message' => "Payment status updated to {$status}"
            ];
            
        } catch (Exception $e) {
            error_log("Manual payment status update error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to update payment status: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update payment status in database (private method for internal use)
     */
    private function updatePaymentStatusInternal($transactionId, $statusData) {
        $maxRetries = 2; // Reduced to prevent long timeouts
        $retryCount = 0;
        
        while ($retryCount < $maxRetries) {
            try {
                $conn = $this->db->getConnection();
                
                $status = 'pending';
                if (isset($statusData['status'])) {
                    $status = $statusData['status'] === 'success' ? 'completed' : 'failed';
                }
                
                // Use a simple, fast update without transactions to avoid locks
                $stmt = $conn->prepare("
                    UPDATE payments 
                    SET status = ?, updated_at = datetime('now') 
                    WHERE transaction_id = ?
                ");
                
                $stmt->execute([$status, $transactionId]);
                
                error_log("Payment status updated: {$transactionId} -> {$status}");
                return; // Success, exit retry loop
                
            } catch (Exception $e) {
                $retryCount++;
                error_log("Database error updating payment status (attempt {$retryCount}): " . $e->getMessage());
                
                if ($retryCount < $maxRetries) {
                    // Wait before retrying (exponential backoff)
                    usleep(100000 * $retryCount); // 100ms, 200ms, 300ms
                } else {
                    error_log("Failed to update payment status after {$maxRetries} attempts");
                    // Don't throw the exception to prevent blocking the main flow
                }
            }
        }
    }
    
    /**
     * Handle payment callback from Bitmuk
     */
    public function handleCallback($data) {
        try {
            // Verify callback data
            if (empty($data['transactionId']) || empty($data['status'])) {
                return ['success' => false, 'message' => 'Invalid callback data'];
            }
            
            $transactionId = $data['transactionId'];
            $status = $data['status'] === 'success' ? 'completed' : 'failed';
            
            // Update payment status
            $this->updatePaymentStatusInternal($transactionId, $data);
            
            // If payment successful, credit user's account
            if ($status === 'completed') {
                $this->creditUserAccount($transactionId);
            }
            
            return ['success' => true, 'message' => 'Callback processed'];
            
        } catch (Exception $e) {
            error_log("Callback handling error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Callback processing failed'];
        }
    }
    
    /**
     * Credit user's account after successful payment
     */
    private function creditUserAccount($transactionId) {
        $maxRetries = 2; // Reduced to prevent long timeouts
        $retryCount = 0;
        
        while ($retryCount < $maxRetries) {
            try {
                $conn = $this->db->getConnection();
            
            // Get payment details
            $stmt = $conn->prepare("SELECT * FROM payments WHERE transaction_id = ?");
            $stmt->execute([$transactionId]);
            $payment = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$payment) {
                error_log("Payment not found for transaction: " . $transactionId);
                return;
            }
            
            // Check if payment has already been credited to prevent duplicate crediting
            if ($payment['status'] === 'completed') {
                error_log("Payment {$transactionId} already credited, skipping duplicate credit");
                return;
            }
            
            // Detect country and currency from phone number
            $countryInfo = $this->detectCountryFromPhone($payment['phone']);
            $currencyCode = $countryInfo['currency'];
            $countryName = $countryInfo['country'];
            
            // Use the original deposit amount in local currency
            $localAmount = $payment['amount'];
            
            error_log("Starting wallet credit process for transaction {$transactionId}: {$localAmount} {$currencyCode} to user {$payment['user_id']}");
            
            // Use a simpler approach without transactions to prevent locks
            // $conn->beginTransaction(); // Commented out to prevent locks
            
            // Ensure user_fiat_balances table exists
            $conn->exec("
                CREATE TABLE IF NOT EXISTS user_fiat_balances (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    currency_code TEXT NOT NULL,
                    balance REAL DEFAULT 0.0,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    UNIQUE(user_id, currency_code)
                )
            ");
            
            // Get current balance for logging
            $stmt = $conn->prepare("SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency_code = ?");
            $stmt->execute([$payment['user_id'], $currencyCode]);
            $currentBalance = $stmt->fetchColumn() ?: 0;
            
            // Update user's local currency balance (simplified approach)
            // First, try to insert if doesn't exist
            $stmt = $conn->prepare("
                INSERT OR IGNORE INTO user_fiat_balances (user_id, currency_code, balance, updated_at)
                VALUES (?, ?, 0, datetime('now'))
            ");
            $stmt->execute([$payment['user_id'], $currencyCode]);
            
            // Then update the balance
            $stmt = $conn->prepare("
                UPDATE user_fiat_balances 
                SET balance = balance + ?, updated_at = datetime('now')
                WHERE user_id = ? AND currency_code = ?
            ");
            $stmt->execute([$localAmount, $payment['user_id'], $currencyCode]);
            
            // Get new balance for logging
            $stmt = $conn->prepare("SELECT balance FROM user_fiat_balances WHERE user_id = ? AND currency_code = ?");
            $stmt->execute([$payment['user_id'], $currencyCode]);
            $newBalance = $stmt->fetchColumn() ?: 0;
            
            error_log("Wallet balance updated: User {$payment['user_id']} {$currencyCode} wallet: {$currentBalance} → {$newBalance} (+{$localAmount})");
            
            // Ensure transactions table exists
            $conn->exec("
                CREATE TABLE IF NOT EXISTS transactions (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    type TEXT NOT NULL,
                    amount REAL NOT NULL,
                    currency TEXT NOT NULL,
                    description TEXT,
                    reference TEXT,
                    status TEXT DEFAULT 'pending',
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Record transaction in local currency
            $stmt = $conn->prepare("
                INSERT INTO transactions (
                    user_id, type, amount, currency, description, 
                    reference, status, created_at
                ) VALUES (?, 'deposit', ?, ?, ?, ?, 'completed', datetime('now'))
            ");
            
            $stmt->execute([
                $payment['user_id'],
                $localAmount,
                $currencyCode,
                "Mobile Money Deposit from {$countryName} via Bitmuk",
                $transactionId
            ]);
            
            // Mark payment as credited in the payments table
            $stmt = $conn->prepare("UPDATE payments SET status = 'completed', updated_at = datetime('now') WHERE transaction_id = ?");
            $stmt->execute([$transactionId]);
            
            // Transaction commit removed to prevent locks
            // $conn->commit();
            
                error_log("✅ WALLET CREDITED SUCCESSFULLY: User {$payment['user_id']}, Amount: {$localAmount} {$currencyCode} from {$countryName}, Transaction: {$transactionId}");
                return; // Success, exit retry loop
                
            } catch (Exception $e) {
                $retryCount++;
                error_log("❌ Account crediting error (attempt {$retryCount}): " . $e->getMessage());
                
                if ($retryCount < $maxRetries) {
                    // Wait before retrying (exponential backoff)
                    usleep(200000 * $retryCount); // 200ms, 400ms, 600ms
                } else {
                    error_log("Failed to credit account after {$maxRetries} attempts");
                    // Don't throw the exception to prevent blocking the main flow
                }
            }
        }
    }
    
    /**
     * Detect country and currency from phone number
     */
    private function detectCountryFromPhone($phone) {
        // Remove any non-digit characters
        $phone = preg_replace('/\D/', '', $phone);
        
        // Country detection based on phone number prefixes
        if (strpos($phone, '256') === 0) {
            return ['country' => 'Uganda', 'currency' => 'UGX'];
        } elseif (strpos($phone, '254') === 0) {
            return ['country' => 'Kenya', 'currency' => 'KES'];
        } elseif (strpos($phone, '255') === 0) {
            return ['country' => 'Tanzania', 'currency' => 'TZS'];
        } elseif (strpos($phone, '250') === 0) {
            return ['country' => 'Rwanda', 'currency' => 'RWF'];
        } elseif (strpos($phone, '243') === 0) {
            return ['country' => 'DR Congo', 'currency' => 'CDF'];
        } elseif (strpos($phone, '234') === 0) {
            return ['country' => 'Nigeria', 'currency' => 'NGN'];
        } elseif (strpos($phone, '233') === 0) {
            return ['country' => 'Ghana', 'currency' => 'GHS'];
        } elseif (strpos($phone, '260') === 0) {
            return ['country' => 'Zambia', 'currency' => 'ZMW'];
        } elseif (strpos($phone, '265') === 0) {
            return ['country' => 'Malawi', 'currency' => 'MWK'];
        } else {
            // Default to Uganda if country cannot be detected
            return ['country' => 'Uganda', 'currency' => 'UGX'];
        }
    }
}

?>
