<?php
/**
 * Automated Payment Status Checker
 * 
 * This script handles automated payment status checking when Bitmuk's API is unreliable.
 * It provides multiple fallback methods to determine payment status and automatically
 * credit user accounts when payments are confirmed.
 */

require_once __DIR__ . '/../config/database.php';

class AutomatedPaymentChecker {
    private $conn;
    private $bitmukApiKey;
    private $bitmukApiSecret;
    private $bitmukBaseUrl;
    
    public function __construct() {
        $db = new Database();
        $this->conn = $db->getConnection();
        $this->bitmukApiKey = 'test_key_123456789';
        $this->bitmukApiSecret = 'test_secret_abcdefghijklmnopqrstuvwxyz123456789';
        $this->bitmukBaseUrl = 'https://api.bitmuk.com/v1';
    }
    
    /**
     * Main method to check all pending payments
     * DISABLED: System now completely depends on Bitmuk's status
     */
    public function checkPendingPayments() {
        error_log("Automated payment checker disabled - system now completely depends on Bitmuk's status");
        
        // Return early - no automated checking
        return;
        
        // OLD CODE (DISABLED):
        // Get all pending payments older than 5 minutes
        // $stmt = $this->conn->prepare("
        //     SELECT * FROM payments 
        //     WHERE status = 'pending' 
        //     AND created_at < datetime('now', '-5 minutes')
        //     ORDER BY created_at ASC
        // ");
        // $stmt->execute();
        // $pendingPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        // 
        // error_log("Found " . count($pendingPayments) . " pending payments to check");
        // 
        // foreach ($pendingPayments as $payment) {
        //     $this->checkPaymentStatus($payment);
        // }
        // 
        // error_log("Automated payment status check completed");
    }
    
    /**
     * Check status of a specific payment using multiple methods
     */
    private function checkPaymentStatus($payment) {
        $transactionId = $payment['transaction_id'];
        $bitmukReference = $payment['bitmuk_reference'];
        
        error_log("Checking payment status for: $transactionId (Bitmuk Ref: $bitmukReference)");
        
        // Method 1: Try Bitmuk API (if it's working)
        $status = $this->checkBitmukStatus($bitmukReference);
        
        // Method 2: If Bitmuk API fails, use fallback logic
        if (!$status) {
            $status = $this->checkFallbackStatus($payment);
        }
        
        // Method 3: If still no status, use time-based logic for old payments
        if (!$status) {
            $status = $this->checkTimeBasedStatus($payment);
        }
        
        // Update payment status if we found a definitive status
        if ($status && $status !== 'pending') {
            $this->updatePaymentStatus($payment, $status);
        }
    }
    
    /**
     * Method 1: Check Bitmuk API status
     */
    private function checkBitmukStatus($bitmukReference) {
        try {
            $url = $this->bitmukBaseUrl . '/collections/status';
            $data = [
                'reference' => $bitmukReference,
                'api_key' => $this->bitmukApiKey,
                'api_secret' => $this->bitmukApiSecret
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Accept: application/json'
            ]);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200 && $response) {
                $decodedResponse = json_decode($response, true);
                if (isset($decodedResponse['status']) && $decodedResponse['status'] === 200) {
                    $data = $decodedResponse['data'] ?? [];
                    if (isset($data['status'])) {
                        $status = strtolower($data['status']);
                        error_log("Bitmuk API returned status: $status for reference: $bitmukReference");
                        return $this->mapBitmukStatus($status);
                    }
                }
            }
            
            error_log("Bitmuk API failed for reference: $bitmukReference (HTTP: $httpCode)");
            return false;
            
        } catch (Exception $e) {
            error_log("Bitmuk API error for reference: $bitmukReference - " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Method 2: Fallback status checking using payment patterns
     * DISABLED: This was causing failed payments to be marked as completed
     */
    private function checkFallbackStatus($payment) {
        // Disabled fallback logic to prevent incorrect status updates
        // Only rely on Bitmuk API or manual verification
        return false;
    }
    
    /**
     * Method 3: Time-based status checking for very old payments
     */
    private function checkTimeBasedStatus($payment) {
        $transactionId = $payment['transaction_id'];
        $createdAt = new DateTime($payment['created_at']);
        $now = new DateTime();
        $hoursSinceCreated = $now->diff($createdAt)->h;
        
        // Intelligent fallback logic for Bitmuk API failures
        $minutesSinceCreated = $now->diff($createdAt)->i;
        
        // If payment is older than 15 minutes and still pending, likely failed
        // This handles cases where Bitmuk API is unreliable
        if (($minutesSinceCreated >= 15 || $hoursSinceCreated >= 1) && $payment['status'] === 'pending') {
            error_log("Intelligent fallback: Payment $transactionId is older than 15 minutes, marking as failed");
            return 'failed';
        }
        
        return false;
    }
    
    /**
     * Update payment status and credit user account if completed
     */
    private function updatePaymentStatus($payment, $status) {
        $transactionId = $payment['transaction_id'];
        $userId = $payment['user_id'];
        $amount = $payment['amount'];
        $currency = $payment['currency'];
        
        error_log("Updating payment $transactionId to status: $status");
        
        // Update payment status
        $stmt = $this->conn->prepare("
            UPDATE payments 
            SET status = ?, updated_at = datetime('now')
            WHERE transaction_id = ?
        ");
        $stmt->execute([$status, $transactionId]);
        
        // If completed, credit user account
        if ($status === 'completed') {
            $this->creditUserAccount($payment);
        }
    }
    
    /**
     * Credit user account when payment is completed
     */
    private function creditUserAccount($payment) {
        $userId = $payment['user_id'];
        $amount = $payment['amount'];
        $currency = $payment['currency'];
        $transactionId = $payment['transaction_id'];
        
        error_log("Crediting user $userId with $amount $currency for transaction $transactionId");
        
        // Check if user already has this currency balance
        $stmt = $this->conn->prepare("
            SELECT balance FROM user_fiat_balances 
            WHERE user_id = ? AND currency_code = ?
        ");
        $stmt->execute([$userId, $currency]);
        $existingBalance = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingBalance) {
            // Update existing balance
            $stmt = $this->conn->prepare("
                UPDATE user_fiat_balances 
                SET balance = balance + ?, updated_at = datetime('now')
                WHERE user_id = ? AND currency_code = ?
            ");
            $stmt->execute([$amount, $userId, $currency]);
        } else {
            // Create new balance record
            $stmt = $this->conn->prepare("
                INSERT INTO user_fiat_balances (user_id, currency_code, balance, created_at, updated_at)
                VALUES (?, ?, ?, datetime('now'), datetime('now'))
            ");
            $stmt->execute([$userId, $currency, $amount]);
        }
        
        // Add transaction record
        $stmt = $this->conn->prepare("
            INSERT INTO transactions (user_id, type, currency, amount, status, created_at)
            VALUES (?, 'deposit', ?, ?, 'completed', datetime('now'))
        ");
        $stmt->execute([$userId, $currency, $amount]);
        
        error_log("Successfully credited user $userId with $amount $currency");
    }
    
    /**
     * Map Bitmuk status to our internal status
     */
    private function mapBitmukStatus($bitmukStatus) {
        $statusMap = [
            'success' => 'completed',
            'successful' => 'completed',
            'completed' => 'completed',
            'failed' => 'failed',
            'cancelled' => 'failed',
            'expired' => 'failed'
        ];
        
        return $statusMap[$bitmukStatus] ?? 'pending';
    }
}

// Run the automated checker if called directly
if (php_sapi_name() === 'cli' || (isset($_GET['run']) && $_GET['run'] === 'checker')) {
    $checker = new AutomatedPaymentChecker();
    $checker->checkPendingPayments();
    
    if (php_sapi_name() !== 'cli') {
        echo json_encode(['success' => true, 'message' => 'Payment status check completed']);
    }
}
?>
