<?php
/**
 * Performance Optimization and Scalability System
 * Production-ready performance optimization and scalability features
 */

class PerformanceOptimizer {
    private $conn;
    private $cacheConfig;
    private $performanceConfig;
    private $connectionPool;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->cacheConfig = $this->getCacheConfig();
        $this->performanceConfig = $this->getPerformanceConfig();
        $this->connectionPool = [];
        $this->initializeOptimizations();
    }
    
    /**
     * Get cache configuration
     */
    private function getCacheConfig() {
        return [
            'redis' => [
                'enabled' => getenv('REDIS_ENABLED') === 'true',
                'host' => getenv('REDIS_HOST') ?: 'localhost',
                'port' => getenv('REDIS_PORT') ?: 6379,
                'password' => getenv('REDIS_PASSWORD'),
                'database' => getenv('REDIS_DATABASE') ?: 0,
                'timeout' => 5,
                'persistent' => true
            ],
            'memcached' => [
                'enabled' => getenv('MEMCACHED_ENABLED') === 'true',
                'host' => getenv('MEMCACHED_HOST') ?: 'localhost',
                'port' => getenv('MEMCACHED_PORT') ?: 11211,
                'timeout' => 5
            ],
            'file_cache' => [
                'enabled' => true,
                'path' => __DIR__ . '/../cache/',
                'ttl' => 3600, // 1 hour default
                'max_size' => 100 * 1024 * 1024 // 100MB
            ],
            'default_ttl' => [
                'user_data' => 300, // 5 minutes
                'market_data' => 60, // 1 minute
                'transaction_data' => 1800, // 30 minutes
                'blockchain_data' => 300, // 5 minutes
                'api_responses' => 600 // 10 minutes
            ]
        ];
    }
    
    /**
     * Get performance configuration
     */
    private function getPerformanceConfig() {
        return [
            'database' => [
                'connection_pool_size' => 10,
                'max_connections' => 100,
                'query_timeout' => 30,
                'slow_query_threshold' => 1.0, // seconds
                'enable_query_cache' => true,
                'enable_result_cache' => true
            ],
            'api' => [
                'rate_limiting' => [
                    'enabled' => true,
                    'requests_per_minute' => 1000,
                    'burst_limit' => 100,
                    'per_user_limit' => 100
                ],
                'response_compression' => true,
                'cors_optimization' => true,
                'request_batching' => true
            ],
            'blockchain' => [
                'parallel_requests' => 10,
                'request_timeout' => 30,
                'retry_attempts' => 3,
                'circuit_breaker' => [
                    'enabled' => true,
                    'failure_threshold' => 5,
                    'recovery_timeout' => 60
                ]
            ],
            'memory' => [
                'max_memory_usage' => '512M',
                'gc_probability' => 1,
                'gc_divisor' => 100
            ]
        ];
    }
    
    /**
     * Initialize performance optimizations
     */
    private function initializeOptimizations() {
        // Set memory limit
        ini_set('memory_limit', $this->performanceConfig['memory']['max_memory_usage']);
        
        // Enable garbage collection
        ini_set('gc_probability', $this->performanceConfig['memory']['gc_probability']);
        ini_set('gc_divisor', $this->performanceConfig['memory']['gc_divisor']);
        
        // Enable output compression
        if ($this->performanceConfig['api']['response_compression'] && !ob_get_level()) {
            ob_start('ob_gzhandler');
        }
        
        // Initialize cache directory
        if ($this->cacheConfig['file_cache']['enabled']) {
            $this->ensureCacheDirectory();
        }
        
        // Initialize connection pool
        $this->initializeConnectionPool();
    }
    
    /**
     * Ensure cache directory exists
     */
    private function ensureCacheDirectory() {
        $cachePath = $this->cacheConfig['file_cache']['path'];
        if (!is_dir($cachePath)) {
            mkdir($cachePath, 0755, true);
        }
    }
    
    /**
     * Initialize database connection pool
     */
    private function initializeConnectionPool() {
        $poolSize = $this->performanceConfig['database']['connection_pool_size'];
        
        for ($i = 0; $i < $poolSize; $i++) {
            try {
                require_once __DIR__ . '/../config/database.php';
                $db = new Database();
                $this->connectionPool[] = $db->getConnection();
            } catch (Exception $e) {
                error_log("Connection pool initialization error: " . $e->getMessage());
            }
        }
    }
    
    /**
     * Get connection from pool
     */
    public function getConnection() {
        if (!empty($this->connectionPool)) {
            return array_pop($this->connectionPool);
        }
        
        // Create new connection if pool is empty
        try {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            return $db->getConnection();
        } catch (Exception $e) {
            error_log("Failed to create new database connection: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Return connection to pool
     */
    public function returnConnection($connection) {
        if (count($this->connectionPool) < $this->performanceConfig['database']['connection_pool_size']) {
            $this->connectionPool[] = $connection;
        } else {
            // Close excess connection
            $connection = null;
        }
    }
    
    /**
     * Cache data with multiple backends
     */
    public function cache($key, $data, $ttl = null, $tags = []) {
        $results = [];
        
        // Determine TTL
        if ($ttl === null) {
            $ttl = $this->getDefaultTTL($key);
        }
        
        // Try Redis first
        if ($this->cacheConfig['redis']['enabled']) {
            $results['redis'] = $this->cacheRedis($key, $data, $ttl, $tags);
        }
        
        // Try Memcached
        if ($this->cacheConfig['memcached']['enabled']) {
            $results['memcached'] = $this->cacheMemcached($key, $data, $ttl);
        }
        
        // Fallback to file cache
        if ($this->cacheConfig['file_cache']['enabled']) {
            $results['file'] = $this->cacheFile($key, $data, $ttl);
        }
        
        return $results;
    }
    
    /**
     * Retrieve cached data
     */
    public function getCache($key) {
        // Try Redis first
        if ($this->cacheConfig['redis']['enabled']) {
            $data = $this->getRedisCache($key);
            if ($data !== false) {
                return $data;
            }
        }
        
        // Try Memcached
        if ($this->cacheConfig['memcached']['enabled']) {
            $data = $this->getMemcachedCache($key);
            if ($data !== false) {
                return $data;
            }
        }
        
        // Try file cache
        if ($this->cacheConfig['file_cache']['enabled']) {
            $data = $this->getFileCache($key);
            if ($data !== false) {
                return $data;
            }
        }
        
        return false;
    }
    
    /**
     * Cache in Redis
     */
    private function cacheRedis($key, $data, $ttl, $tags) {
        try {
            $redis = new Redis();
            $redis->connect(
                $this->cacheConfig['redis']['host'],
                $this->cacheConfig['redis']['port'],
                $this->cacheConfig['redis']['timeout']
            );
            
            if ($this->cacheConfig['redis']['password']) {
                $redis->auth($this->cacheConfig['redis']['password']);
            }
            
            $redis->select($this->cacheConfig['redis']['database']);
            
            $serializedData = serialize($data);
            $result = $redis->setex($key, $ttl, $serializedData);
            
            // Store tags for cache invalidation
            if (!empty($tags)) {
                foreach ($tags as $tag) {
                    $redis->sAdd("tag:$tag", $key);
                    $redis->expire("tag:$tag", $ttl);
                }
            }
            
            $redis->close();
            return $result;
            
        } catch (Exception $e) {
            error_log("Redis cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get from Redis cache
     */
    private function getRedisCache($key) {
        try {
            $redis = new Redis();
            $redis->connect(
                $this->cacheConfig['redis']['host'],
                $this->cacheConfig['redis']['port'],
                $this->cacheConfig['redis']['timeout']
            );
            
            if ($this->cacheConfig['redis']['password']) {
                $redis->auth($this->cacheConfig['redis']['password']);
            }
            
            $redis->select($this->cacheConfig['redis']['database']);
            
            $data = $redis->get($key);
            $redis->close();
            
            return $data !== false ? unserialize($data) : false;
            
        } catch (Exception $e) {
            error_log("Redis get cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Cache in Memcached
     */
    private function cacheMemcached($key, $data, $ttl) {
        try {
            $memcached = new Memcached();
            $memcached->addServer(
                $this->cacheConfig['memcached']['host'],
                $this->cacheConfig['memcached']['port']
            );
            
            return $memcached->set($key, $data, $ttl);
            
        } catch (Exception $e) {
            error_log("Memcached cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get from Memcached cache
     */
    private function getMemcachedCache($key) {
        try {
            $memcached = new Memcached();
            $memcached->addServer(
                $this->cacheConfig['memcached']['host'],
                $this->cacheConfig['memcached']['port']
            );
            
            return $memcached->get($key);
            
        } catch (Exception $e) {
            error_log("Memcached get cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Cache in file system
     */
    private function cacheFile($key, $data, $ttl) {
        try {
            $cachePath = $this->cacheConfig['file_cache']['path'];
            $filename = $cachePath . md5($key) . '.cache';
            
            $cacheData = [
                'data' => $data,
                'expires' => time() + $ttl,
                'created' => time()
            ];
            
            return file_put_contents($filename, serialize($cacheData), LOCK_EX) !== false;
            
        } catch (Exception $e) {
            error_log("File cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get from file cache
     */
    private function getFileCache($key) {
        try {
            $cachePath = $this->cacheConfig['file_cache']['path'];
            $filename = $cachePath . md5($key) . '.cache';
            
            if (!file_exists($filename)) {
                return false;
            }
            
            $cacheData = unserialize(file_get_contents($filename));
            
            if ($cacheData['expires'] < time()) {
                unlink($filename);
                return false;
            }
            
            return $cacheData['data'];
            
        } catch (Exception $e) {
            error_log("File get cache error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get default TTL for cache key
     */
    private function getDefaultTTL($key) {
        $prefix = explode(':', $key)[0];
        return $this->cacheConfig['default_ttl'][$prefix] ?? 300; // 5 minutes default
    }
    
    /**
     * Invalidate cache by tags
     */
    public function invalidateCacheByTags($tags) {
        $results = [];
        
        if ($this->cacheConfig['redis']['enabled']) {
            $results['redis'] = $this->invalidateRedisByTags($tags);
        }
        
        return $results;
    }
    
    /**
     * Invalidate Redis cache by tags
     */
    private function invalidateRedisByTags($tags) {
        try {
            $redis = new Redis();
            $redis->connect(
                $this->cacheConfig['redis']['host'],
                $this->cacheConfig['redis']['port']
            );
            
            if ($this->cacheConfig['redis']['password']) {
                $redis->auth($this->cacheConfig['redis']['password']);
            }
            
            $redis->select($this->cacheConfig['redis']['database']);
            
            $deleted = 0;
            foreach ($tags as $tag) {
                $keys = $redis->sMembers("tag:$tag");
                if (!empty($keys)) {
                    $deleted += $redis->del($keys);
                    $redis->del("tag:$tag");
                }
            }
            
            $redis->close();
            return $deleted;
            
        } catch (Exception $e) {
            error_log("Redis cache invalidation error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Optimize database queries
     */
    public function optimizeQuery($query, $params = []) {
        $startTime = microtime(true);
        
        try {
            $connection = $this->getConnection();
            $stmt = $connection->prepare($query);
            $stmt->execute($params);
            $result = $stmt->fetchAll();
            
            $executionTime = microtime(true) - $startTime;
            
            // Log slow queries
            if ($executionTime > $this->performanceConfig['database']['slow_query_threshold']) {
                $this->logSlowQuery($query, $params, $executionTime);
            }
            
            $this->returnConnection($connection);
            
            return [
                'success' => true,
                'data' => $result,
                'execution_time' => $executionTime
            ];
            
        } catch (Exception $e) {
            error_log("Query optimization error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Log slow queries
     */
    private function logSlowQuery($query, $params, $executionTime) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO slow_queries (query, params, execution_time, created_at)
                VALUES (?, ?, ?, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$query, json_encode($params), $executionTime]);
        } catch (Exception $e) {
            error_log("Slow query logging error: " . $e->getMessage());
        }
    }
    
    /**
     * Batch API requests
     */
    public function batchRequests($requests, $maxConcurrency = null) {
        if ($maxConcurrency === null) {
            $maxConcurrency = $this->performanceConfig['blockchain']['parallel_requests'];
        }
        
        $results = [];
        $chunks = array_chunk($requests, $maxConcurrency);
        
        foreach ($chunks as $chunk) {
            $chunkResults = $this->processBatchChunk($chunk);
            $results = array_merge($results, $chunkResults);
        }
        
        return $results;
    }
    
    /**
     * Process batch chunk with parallel execution
     */
    private function processBatchChunk($requests) {
        $multiHandle = curl_multi_init();
        $curlHandles = [];
        
        // Initialize curl handles
        foreach ($requests as $index => $request) {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $request['url'],
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => $this->performanceConfig['blockchain']['request_timeout'],
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_SSL_VERIFYPEER => true
            ]);
            
            if (isset($request['headers'])) {
                curl_setopt($ch, CURLOPT_HTTPHEADER, $request['headers']);
            }
            
            if (isset($request['post_data'])) {
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $request['post_data']);
            }
            
            curl_multi_add_handle($multiHandle, $ch);
            $curlHandles[$index] = $ch;
        }
        
        // Execute requests
        $running = null;
        do {
            curl_multi_exec($multiHandle, $running);
            curl_multi_select($multiHandle);
        } while ($running > 0);
        
        // Collect results
        $results = [];
        foreach ($curlHandles as $index => $ch) {
            $response = curl_multi_getcontent($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            
            $results[$index] = [
                'success' => $httpCode >= 200 && $httpCode < 300 && !$error,
                'data' => $response,
                'http_code' => $httpCode,
                'error' => $error
            ];
            
            curl_multi_remove_handle($multiHandle, $ch);
            curl_close($ch);
        }
        
        curl_multi_close($multiHandle);
        
        return $results;
    }
    
    /**
     * Implement circuit breaker pattern
     */
    public function circuitBreaker($service, $operation, $fallback = null) {
        $circuitKey = "circuit_breaker:$service";
        $circuitState = $this->getCache($circuitKey);
        
        if (!$circuitState) {
            $circuitState = [
                'state' => 'closed', // closed, open, half-open
                'failure_count' => 0,
                'last_failure' => 0,
                'success_count' => 0
            ];
        }
        
        $config = $this->performanceConfig['blockchain']['circuit_breaker'];
        
        // Check if circuit is open
        if ($circuitState['state'] === 'open') {
            if (time() - $circuitState['last_failure'] > $config['recovery_timeout']) {
                $circuitState['state'] = 'half-open';
                $circuitState['success_count'] = 0;
            } else {
                // Circuit is still open, use fallback
                return $fallback ? $fallback() : ['success' => false, 'error' => 'Circuit breaker open'];
            }
        }
        
        try {
            $result = $operation();
            
            if ($result['success']) {
                // Operation successful
                $circuitState['failure_count'] = 0;
                $circuitState['success_count']++;
                
                if ($circuitState['state'] === 'half-open' && $circuitState['success_count'] >= 2) {
                    $circuitState['state'] = 'closed';
                }
                
                $this->cache($circuitKey, $circuitState, 3600); // Cache for 1 hour
                return $result;
                
            } else {
                // Operation failed
                $circuitState['failure_count']++;
                $circuitState['last_failure'] = time();
                
                if ($circuitState['failure_count'] >= $config['failure_threshold']) {
                    $circuitState['state'] = 'open';
                }
                
                $this->cache($circuitKey, $circuitState, 3600);
                return $fallback ? $fallback() : $result;
            }
            
        } catch (Exception $e) {
            // Exception occurred
            $circuitState['failure_count']++;
            $circuitState['last_failure'] = time();
            
            if ($circuitState['failure_count'] >= $config['failure_threshold']) {
                $circuitState['state'] = 'open';
            }
            
            $this->cache($circuitKey, $circuitState, 3600);
            
            return $fallback ? $fallback() : ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Get performance metrics
     */
    public function getPerformanceMetrics() {
        try {
            $metrics = [
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true),
                'memory_limit' => ini_get('memory_limit'),
                'connection_pool_size' => count($this->connectionPool),
                'cache_stats' => $this->getCacheStats(),
                'database_stats' => $this->getDatabaseStats()
            ];
            
            return [
                'success' => true,
                'metrics' => $metrics,
                'timestamp' => time()
            ];
            
        } catch (Exception $e) {
            error_log("Performance metrics error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Get cache statistics
     */
    private function getCacheStats() {
        $stats = [];
        
        if ($this->cacheConfig['file_cache']['enabled']) {
            $cachePath = $this->cacheConfig['file_cache']['path'];
            $files = glob($cachePath . '*.cache');
            $stats['file_cache'] = [
                'files_count' => count($files),
                'total_size' => array_sum(array_map('filesize', $files))
            ];
        }
        
        return $stats;
    }
    
    /**
     * Get database statistics
     */
    private function getDatabaseStats() {
        try {
            $stmt = $this->conn->query("
                SELECT 
                    COUNT(*) as total_queries,
                    AVG(execution_time) as avg_execution_time,
                    MAX(execution_time) as max_execution_time
                FROM slow_queries 
                WHERE created_at >= datetime('now', '-1 hour')
            ");
            
            $result = $stmt->fetch();
            
            return [
                'slow_queries_count' => $result['total_queries'] ?? 0,
                'avg_execution_time' => $result['avg_execution_time'] ?? 0,
                'max_execution_time' => $result['max_execution_time'] ?? 0
            ];
            
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Clean up old cache files
     */
    public function cleanupCache() {
        $cleaned = 0;
        
        if ($this->cacheConfig['file_cache']['enabled']) {
            $cachePath = $this->cacheConfig['file_cache']['path'];
            $files = glob($cachePath . '*.cache');
            
            foreach ($files as $file) {
                if (filemtime($file) < time() - $this->cacheConfig['file_cache']['ttl']) {
                    if (unlink($file)) {
                        $cleaned++;
                    }
                }
            }
        }
        
        return $cleaned;
    }
    
    /**
     * Ensure performance tables exist
     */
    public function ensurePerformanceTablesExist() {
        try {
            // Create slow_queries table
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS slow_queries (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    query TEXT NOT NULL,
                    params TEXT,
                    execution_time REAL NOT NULL,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Create performance_metrics table
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS performance_metrics (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    metric_name TEXT NOT NULL,
                    metric_value REAL NOT NULL,
                    metric_unit TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Create indexes
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_slow_queries_created ON slow_queries(created_at)");
            $this->conn->exec("CREATE INDEX IF NOT EXISTS idx_performance_metrics_name ON performance_metrics(metric_name)");
            
        } catch (Exception $e) {
            error_log("Performance tables creation error: " . $e->getMessage());
        }
    }
}
?>

