/**
 * Real-time Price Feed Service
 * Fetches real-time cryptocurrency prices from external APIs
 */

class PriceFeedService {
    private static $api_keys = [];
    private static $last_update = null;
    private static $price_cache = [];
    private static $cache_duration = 60; // 60 seconds
    
    public static function initialize() {
        require_once __DIR__ . '/../config/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        self::$api_keys = [
            'coinmarketcap' => EnvConfig::get('COINMARKETCAP_API_KEY'),
            'coingecko' => EnvConfig::get('COINGECKO_API_KEY'),
            'binance' => EnvConfig::get('BINANCE_API_KEY')
        ];
    }
    
    public static function getPrices($symbols = ['BTC', 'ETH', 'USDT', 'USDC', 'BNB', 'ADA', 'SOL', 'MATIC']) {
        try {
            // Check cache first
            if (self::isCacheValid()) {
                return self::getCachedPrices($symbols);
            }
            
            // Fetch fresh prices
            $prices = self::fetchFromAPI($symbols);
            
            if ($prices) {
                self::$price_cache = $prices;
                self::$last_update = time();
                return $prices;
            }
            
            // Fallback to cached data if API fails
            return self::getCachedPrices($symbols);
            
        } catch (Exception $e) {
            error_log("Price feed error: " . $e->getMessage());
            return self::getFallbackPrices($symbols);
        }
    }
    
    private static function isCacheValid() {
        return self::$last_update && (time() - self::$last_update) < self::$cache_duration;
    }
    
    private static function getCachedPrices($symbols) {
        $result = [];
        foreach ($symbols as $symbol) {
            if (isset(self::$price_cache[$symbol])) {
                $result[$symbol] = self::$price_cache[$symbol];
            }
        }
        return $result;
    }
    
    private static function fetchFromAPI($symbols) {
        // Try CoinGecko first (free tier)
        $prices = self::fetchFromCoinGecko($symbols);
        if ($prices) return $prices;
        
        // Try Binance API
        $prices = self::fetchFromBinance($symbols);
        if ($prices) return $prices;
        
        // Try CoinMarketCap if API key available
        if (self::$api_keys['coinmarketcap']) {
            $prices = self::fetchFromCoinMarketCap($symbols);
            if ($prices) return $prices;
        }
        
        return null;
    }
    
    private static function fetchFromCoinGecko($symbols) {
        try {
            $symbols_str = implode(',', array_map('strtolower', $symbols));
            $url = "https://api.coingecko.com/api/v3/simple/price?ids={$symbols_str}&vs_currencies=usd&include_24hr_change=true&include_24hr_vol=true";
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_USERAGENT, 'YELLOW-Crypto-Exchange/1.0');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $data = json_decode($response, true);
                $prices = [];
                
                foreach ($symbols as $symbol) {
                    $id = strtolower($symbol);
                    if (isset($data[$id])) {
                        $prices[$symbol] = [
                            'price' => $data[$id]['usd'],
                            'change_24h' => $data[$id]['usd_24h_change'] ?? 0,
                            'volume_24h' => $data[$id]['usd_24h_vol'] ?? 0
                        ];
                    }
                }
                
                return $prices;
            }
            
        } catch (Exception $e) {
            error_log("CoinGecko API error: " . $e->getMessage());
        }
        
        return null;
    }
    
    private static function fetchFromBinance($symbols) {
        try {
            $prices = [];
            
            foreach ($symbols as $symbol) {
                $symbol_pair = $symbol . 'USDT';
                $url = "https://api.binance.com/api/v3/ticker/24hr?symbol={$symbol_pair}";
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($httpCode === 200) {
                    $data = json_decode($response, true);
                    if (isset($data['lastPrice'])) {
                        $prices[$symbol] = [
                            'price' => floatval($data['lastPrice']),
                            'change_24h' => floatval($data['priceChangePercent']),
                            'volume_24h' => floatval($data['volume'])
                        ];
                    }
                }
            }
            
            return !empty($prices) ? $prices : null;
            
        } catch (Exception $e) {
            error_log("Binance API error: " . $e->getMessage());
        }
        
        return null;
    }
    
    private static function fetchFromCoinMarketCap($symbols) {
        try {
            $symbols_str = implode(',', $symbols);
            $url = "https://pro-api.coinmarketcap.com/v1/cryptocurrency/quotes/latest?symbol={$symbols_str}";
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'X-CMC_PRO_API_KEY: ' . self::$api_keys['coinmarketcap'],
                'Accept: application/json'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $data = json_decode($response, true);
                $prices = [];
                
                if (isset($data['data'])) {
                    foreach ($data['data'] as $symbol => $coin) {
                        $prices[$symbol] = [
                            'price' => $coin['quote']['USD']['price'],
                            'change_24h' => $coin['quote']['USD']['percent_change_24h'],
                            'volume_24h' => $coin['quote']['USD']['volume_24h']
                        ];
                    }
                }
                
                return $prices;
            }
            
        } catch (Exception $e) {
            error_log("CoinMarketCap API error: " . $e->getMessage());
        }
        
        return null;
    }
    
    private static function getFallbackPrices($symbols) {
        // Fallback prices for development
        $fallback_prices = [
            'BTC' => ['price' => 45000, 'change_24h' => 2.5, 'volume_24h' => 25000000000],
            'ETH' => ['price' => 3000, 'change_24h' => 1.8, 'volume_24h' => 15000000000],
            'USDT' => ['price' => 1.00, 'change_24h' => 0.1, 'volume_24h' => 50000000000],
            'USDC' => ['price' => 1.00, 'change_24h' => 0.05, 'volume_24h' => 20000000000],
            'BNB' => ['price' => 300, 'change_24h' => 3.2, 'volume_24h' => 2000000000],
            'ADA' => ['price' => 0.45, 'change_24h' => -1.2, 'volume_24h' => 800000000],
            'SOL' => ['price' => 100, 'change_24h' => 5.8, 'volume_24h' => 3000000000],
            'MATIC' => ['price' => 0.85, 'change_24h' => 2.1, 'volume_24h' => 1200000000]
        ];
        
        $result = [];
        foreach ($symbols as $symbol) {
            if (isset($fallback_prices[$symbol])) {
                $result[$symbol] = $fallback_prices[$symbol];
            }
        }
        
        return $result;
    }
    
    public static function updateDatabasePrices($prices) {
        try {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            foreach ($prices as $symbol => $data) {
                $stmt = $conn->prepare("
                    UPDATE market_prices mp
                    JOIN trading_pairs tp ON mp.pair_id = tp.id
                    JOIN cryptocurrencies c1 ON tp.base_crypto_id = c1.id
                    JOIN cryptocurrencies c2 ON tp.quote_crypto_id = c2.id
                    SET mp.price = ?, mp.change_24h = ?, mp.volume_24h = ?, mp.updated_at = NOW()
                    WHERE c1.symbol = ? AND c2.symbol = 'USD'
                ");
                
                $stmt->execute([
                    $data['price'],
                    $data['change_24h'],
                    $data['volume_24h'],
                    $symbol
                ]);
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log("Database price update failed: " . $e->getMessage());
            return false;
        }
    }
}

// Initialize the service
PriceFeedService::initialize();
