<?php
/**
 * Price Settings API
 * Handles cryptocurrency price management for buy/sell operations
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/env.php';

if (!class_exists('PriceSettingsAPI')) {
class PriceSettingsAPI {
    private $conn;
    
    public function __construct() {
        $this->conn = (new Database())->getConnection();
        $this->initializeTables();
    }
    
    /**
     * Initialize required tables
     */
    private function initializeTables() {
        $sql = "
        CREATE TABLE IF NOT EXISTS crypto_prices (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            crypto VARCHAR(10) NOT NULL,
            currency VARCHAR(10) NOT NULL,
            buy_price DECIMAL(20,8) NOT NULL,
            sell_price DECIMAL(20,8) NOT NULL,
            spread DECIMAL(5,2) NOT NULL DEFAULT 0,
            effective_date DATE NOT NULL,
            is_active BOOLEAN DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(crypto, currency, effective_date)
        )";
        
        $this->conn->exec($sql);
        
        // Create indexes for better performance
        $indexes = [
            "CREATE INDEX IF NOT EXISTS idx_crypto_prices_crypto_currency ON crypto_prices(crypto, currency)",
            "CREATE INDEX IF NOT EXISTS idx_crypto_prices_effective_date ON crypto_prices(effective_date)",
            "CREATE INDEX IF NOT EXISTS idx_crypto_prices_active ON crypto_prices(is_active)"
        ];
        
        foreach ($indexes as $indexSql) {
            $this->conn->exec($indexSql);
        }
    }
    
    /**
     * Get all price settings
     */
    public function getAllPrices() {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM crypto_prices 
                WHERE is_active = 1 
                ORDER BY crypto, currency, effective_date DESC
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching all prices: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get price for specific crypto and currency
     */
    public function getPrice($crypto, $currency) {
        try {
            $stmt = $this->conn->prepare("
                SELECT * FROM crypto_prices 
                WHERE crypto = ? AND currency = ? AND is_active = 1 
                AND effective_date <= DATE('now')
                ORDER BY effective_date DESC 
                LIMIT 1
            ");
            $stmt->execute([$crypto, $currency]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching price for {$crypto}/{$currency}: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Save price settings
     */
    public function savePrice($data) {
        try {
            $this->conn->beginTransaction();
            
            // Validate required fields
            $required = ['crypto', 'currency', 'buy_price', 'sell_price'];
            foreach ($required as $field) {
                if (!isset($data[$field]) || empty($data[$field])) {
                    throw new Exception("Missing required field: {$field}");
                }
            }
            
            $crypto = strtoupper($data['crypto']);
            $currency = strtoupper($data['currency']);
            $buyPrice = floatval($data['buy_price']);
            $sellPrice = floatval($data['sell_price']);
            $spread = isset($data['spread']) ? floatval($data['spread']) : 0;
            $effectiveDate = isset($data['effective_date']) ? $data['effective_date'] : date('Y-m-d');
            
            // Calculate spread if not provided
            if ($spread == 0 && $sellPrice > 0) {
                $spread = (($buyPrice - $sellPrice) / $sellPrice) * 100;
            }
            
            // Check if price already exists for this date
            $stmt = $this->conn->prepare("
                SELECT id FROM crypto_prices 
                WHERE crypto = ? AND currency = ? AND effective_date = ?
            ");
            $stmt->execute([$crypto, $currency, $effectiveDate]);
            
            if ($stmt->fetch()) {
                // Update existing record
                $stmt = $this->conn->prepare("
                    UPDATE crypto_prices 
                    SET buy_price = ?, sell_price = ?, spread = ?, updated_at = CURRENT_TIMESTAMP
                    WHERE crypto = ? AND currency = ? AND effective_date = ?
                ");
                $stmt->execute([$buyPrice, $sellPrice, $spread, $crypto, $currency, $effectiveDate]);
            } else {
                // Insert new record
                $stmt = $this->conn->prepare("
                    INSERT INTO crypto_prices (crypto, currency, buy_price, sell_price, spread, effective_date)
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$crypto, $currency, $buyPrice, $sellPrice, $spread, $effectiveDate]);
            }
            
            $this->conn->commit();
            return [
                'success' => true,
                'message' => 'Price settings saved successfully',
                'price_id' => $this->conn->lastInsertId()
            ];
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Error saving price settings: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete price setting
     */
    public function deletePrice($priceId) {
        try {
            $stmt = $this->conn->prepare("
                UPDATE crypto_prices 
                SET is_active = 0, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$priceId]);
            
            if ($stmt->rowCount() > 0) {
                return [
                    'success' => true,
                    'message' => 'Price setting deleted successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'Price setting not found'
                ];
            }
            
        } catch (Exception $e) {
            error_log("Error deleting price setting: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Export prices to CSV
     */
    public function exportPrices() {
        try {
            $prices = $this->getAllPrices();
            
            if (empty($prices)) {
                return [
                    'success' => false,
                    'error' => 'No prices to export'
                ];
            }
            
            $csv = "Crypto,Currency,Buy Price,Sell Price,Spread (%),Effective Date,Created At\n";
            
            foreach ($prices as $price) {
                $csv .= sprintf(
                    "%s,%s,%.8f,%.8f,%.2f,%s,%s\n",
                    $price['crypto'],
                    $price['currency'],
                    $price['buy_price'],
                    $price['sell_price'],
                    $price['spread'],
                    $price['effective_date'],
                    $price['created_at']
                );
            }
            
            return [
                'success' => true,
                'csv' => $csv
            ];
            
        } catch (Exception $e) {
            error_log("Error exporting prices: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get price statistics
     */
    public function getPriceStats() {
        try {
            $stats = [];
            
            // Total active price rules
            $stmt = $this->conn->prepare("SELECT COUNT(*) as count FROM crypto_prices WHERE is_active = 1");
            $stmt->execute();
            $stats['total_rules'] = $stmt->fetch()['count'];
            
            // Unique cryptocurrencies
            $stmt = $this->conn->prepare("SELECT COUNT(DISTINCT crypto) as count FROM crypto_prices WHERE is_active = 1");
            $stmt->execute();
            $stats['unique_cryptos'] = $stmt->fetch()['count'];
            
            // Unique currencies
            $stmt = $this->conn->prepare("SELECT COUNT(DISTINCT currency) as count FROM crypto_prices WHERE is_active = 1");
            $stmt->execute();
            $stats['unique_currencies'] = $stmt->fetch()['count'];
            
            // Average spread
            $stmt = $this->conn->prepare("SELECT AVG(spread) as avg_spread FROM crypto_prices WHERE is_active = 1 AND spread > 0");
            $stmt->execute();
            $result = $stmt->fetch();
            $stats['avg_spread'] = $result['avg_spread'] ? round($result['avg_spread'], 2) : 0;
            
            return $stats;
            
        } catch (Exception $e) {
            error_log("Error getting price stats: " . $e->getMessage());
            return [
                'total_rules' => 0,
                'unique_cryptos' => 0,
                'unique_currencies' => 0,
                'avg_spread' => 0
            ];
        }
    }
}
} // End of class_exists check

// Handle API requests only when accessed directly
if (basename($_SERVER['PHP_SELF']) === 'price_settings.php') {
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $api = new PriceSettingsAPI();
    
    if (isset($_GET['action'])) {
        switch ($_GET['action']) {
            case 'all':
                $prices = $api->getAllPrices();
                echo json_encode(['success' => true, 'prices' => $prices]);
                break;
                
            case 'stats':
                $stats = $api->getPriceStats();
                echo json_encode(['success' => true, 'stats' => $stats]);
                break;
                
            case 'export':
                $result = $api->exportPrices();
                echo json_encode($result);
                break;
                
            default:
                // Get specific price
                if (isset($_GET['crypto']) && isset($_GET['currency'])) {
                    $price = $api->getPrice($_GET['crypto'], $_GET['currency']);
                    echo json_encode(['success' => true, 'price' => $price]);
                } else {
                    // Get all prices
                    $prices = $api->getAllPrices();
                    echo json_encode(['success' => true, 'prices' => $prices]);
                }
                break;
        }
    } else {
        // Default: get all prices
        $prices = $api->getAllPrices();
        echo json_encode(['success' => true, 'prices' => $prices]);
    }
    
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $api = new PriceSettingsAPI();
    
    $input = json_decode(file_get_contents('php://input'), true);
    $result = $api->savePrice($input);
    echo json_encode($result);
    
} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    $api = new PriceSettingsAPI();
    
    // Extract price ID from URL path
    $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $pathParts = explode('/', $path);
    $priceId = end($pathParts);
    
    if (is_numeric($priceId)) {
        $result = $api->deletePrice($priceId);
        echo json_encode($result);
    } else {
        echo json_encode(['success' => false, 'error' => 'Invalid price ID']);
    }
} else {
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
}
} // End of direct access check
?>
