<?php
/**
 * Real Wallet Setup Script
 * Initializes system with real blockchain wallets and crypto
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/blockchain_manager.php';

class RealWalletSetup {
    private $db;
    private $blockchain;
    
    public function __construct() {
        $this->db = new Database();
        $this->blockchain = new BlockchainManager($this->db);
    }
    
    /**
     * Setup real system wallets
     */
    public function setupRealSystemWallets() {
        echo "🚀 Setting up real blockchain system wallets...\n";
        
        try {
            // Initialize system wallets
            $result = $this->blockchain->initializeSystemWallets();
            
            if ($result['success']) {
                echo "✅ System wallets initialized successfully\n";
                
                // Display system wallets
                $this->displaySystemWallets();
                
                // Sync with blockchain
                $this->syncBlockchainBalances();
                
            } else {
                echo "❌ Failed to initialize system wallets: " . $result['error'] . "\n";
            }
            
        } catch (Exception $e) {
            echo "❌ Setup failed: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Display current system wallets
     */
    private function displaySystemWallets() {
        echo "\n📊 Current System Wallets:\n";
        echo str_repeat("-", 80) . "\n";
        
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT currency, network, address, balance, is_real_blockchain, created_at
                FROM crypto_wallets 
                WHERE user_id = 0
                ORDER BY currency, network
            ");
            $stmt->execute();
            $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($wallets)) {
                echo "No system wallets found.\n";
                return;
            }
            
            foreach ($wallets as $wallet) {
                $real_status = $wallet['is_real_blockchain'] ? '🌐 REAL' : '🔧 MOCK';
                echo sprintf(
                    "%-8s %-8s %-42s %-15s %s\n",
                    $wallet['currency'],
                    $wallet['network'],
                    substr($wallet['address'], 0, 42),
                    number_format($wallet['balance'], 8),
                    $real_status
                );
            }
            
        } catch (Exception $e) {
            echo "Error displaying wallets: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Sync blockchain balances
     */
    private function syncBlockchainBalances() {
        echo "\n🔄 Syncing with blockchain...\n";
        
        $result = $this->blockchain->syncBlockchainBalances();
        
        if ($result['success']) {
            echo "✅ Synced {$result['updated_wallets']} wallets with blockchain\n";
        } else {
            echo "❌ Sync failed: " . $result['error'] . "\n";
        }
    }
    
    /**
     * Fund system wallets (for development)
     */
    public function fundSystemWallets() {
        echo "\n💰 Funding system wallets for development...\n";
        echo "⚠️  WARNING: This is for development only!\n";
        
        $funding_amounts = [
            'USDT' => 100000,
            'USDC' => 100000,
            'ETH' => 100,
            'BTC' => 10
        ];
        
        try {
            $conn = $this->db->getConnection();
            
            foreach ($funding_amounts as $currency => $amount) {
                $stmt = $conn->prepare("
                    UPDATE crypto_wallets 
                    SET balance = balance + ?
                    WHERE user_id = 0 AND currency = ? AND is_real_blockchain = 1
                ");
                $stmt->execute([$amount, $currency]);
                
                echo "✅ Added $amount $currency to system wallet\n";
            }
            
        } catch (Exception $e) {
            echo "❌ Funding failed: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Verify wallet functionality
     */
    public function verifyWalletFunctionality() {
        echo "\n🔍 Verifying wallet functionality...\n";
        
        try {
            $conn = $this->db->getConnection();
            
            // Check system wallets
            $stmt = $conn->prepare("
                SELECT COUNT(*) as count FROM crypto_wallets 
                WHERE user_id = 0 AND is_real_blockchain = 1
            ");
            $stmt->execute();
            $system_wallets = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
            
            // Check total system balance
            $stmt = $conn->prepare("
                SELECT SUM(balance) as total_balance FROM crypto_wallets 
                WHERE user_id = 0 AND is_real_blockchain = 1
            ");
            $stmt->execute();
            $total_balance = $stmt->fetch(PDO::FETCH_ASSOC)['total_balance'];
            
            echo "✅ System wallets: $system_wallets\n";
            echo "✅ Total system balance: $total_balance\n";
            
            // Test transaction capability
            $this->testTransactionCapability();
            
        } catch (Exception $e) {
            echo "❌ Verification failed: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Test transaction capability
     */
    private function testTransactionCapability() {
        echo "\n🧪 Testing transaction capability...\n";
        
        try {
            // Get a system wallet
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT * FROM crypto_wallets 
                WHERE user_id = 0 AND is_real_blockchain = 1 AND currency = 'USDT'
                LIMIT 1
            ");
            $stmt->execute();
            $system_wallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$system_wallet) {
                echo "❌ No USDT system wallet found for testing\n";
                return;
            }
            
            echo "✅ System USDT wallet found: {$system_wallet['address']}\n";
            echo "✅ Balance: {$system_wallet['balance']} USDT\n";
            
            // Test balance check
            $real_balance = $this->blockchain->getRealBalance(
                $system_wallet['address'], 
                'USDT', 
                'ERC20'
            );
            
            echo "✅ Real blockchain balance: $real_balance USDT\n";
            
        } catch (Exception $e) {
            echo "❌ Transaction test failed: " . $e->getMessage() . "\n";
        }
    }
}

// Run setup if called directly
if (php_sapi_name() === 'cli') {
    $setup = new RealWalletSetup();
    
    echo "🚀 YELLOW Crypto Exchange - Real Wallet Setup\n";
    echo str_repeat("=", 50) . "\n\n";
    
    // Setup real system wallets
    $setup->setupRealSystemWallets();
    
    // Fund for development (optional)
    echo "\n" . str_repeat("-", 50) . "\n";
    echo "Would you like to fund system wallets for development? (y/n): ";
    $handle = fopen("php://stdin", "r");
    $line = fgets($handle);
    
    if (trim($line) === 'y') {
        $setup->fundSystemWallets();
    }
    
    // Verify functionality
    echo "\n" . str_repeat("-", 50) . "\n";
    $setup->verifyWalletFunctionality();
    
    echo "\n🎉 Setup complete!\n";
    echo "Your crypto exchange now has real blockchain integration!\n";
}
?>

