<?php
/**
 * Security Validator for Crypto Transactions
 * Comprehensive validation and fraud detection system
 */

class SecurityValidator {
    private $db;
    private $max_daily_transaction_limit = 10000; // USD equivalent
    private $max_single_transaction_limit = 1000; // USD equivalent
    private $suspicious_patterns = [];
    
    public function __construct($database) {
        $this->db = $database;
        $this->loadSuspiciousPatterns();
    }
    
    /**
     * Validate crypto buy transaction
     */
    public function validateCryptoBuy($user_id, $currency, $amount, $payment_method) {
        $validations = [
            'user_verification' => $this->validateUserVerification($user_id),
            'transaction_limits' => $this->validateTransactionLimits($user_id, $amount, $currency),
            'fraud_detection' => $this->detectFraud($user_id, $amount, $currency, 'buy'),
            'payment_method' => $this->validatePaymentMethod($payment_method),
            'amount_reasonableness' => $this->validateAmountReasonableness($amount, $currency)
        ];
        
        return $this->processValidations($validations, 'crypto_buy');
    }
    
    /**
     * Validate crypto sell transaction
     */
    public function validateCryptoSell($user_id, $currency, $amount) {
        $validations = [
            'user_verification' => $this->validateUserVerification($user_id),
            'transaction_limits' => $this->validateTransactionLimits($user_id, $amount, $currency),
            'fraud_detection' => $this->detectFraud($user_id, $amount, $currency, 'sell'),
            'wallet_ownership' => $this->validateWalletOwnership($user_id, $currency, $amount),
            'amount_reasonableness' => $this->validateAmountReasonableness($amount, $currency)
        ];
        
        return $this->processValidations($validations, 'crypto_sell');
    }
    
    /**
     * Validate user verification status
     */
    private function validateUserVerification($user_id) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT is_verified, kyc_status, account_frozen 
                FROM users 
                WHERE id = ?
            ");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                return ['valid' => false, 'error' => 'User not found', 'severity' => 'critical'];
            }
            
            if ($user['account_frozen']) {
                return ['valid' => false, 'error' => 'Account is frozen', 'severity' => 'critical'];
            }
            
            if (!$user['is_verified']) {
                return ['valid' => false, 'error' => 'Account not verified', 'severity' => 'high'];
            }
            
            if ($user['kyc_status'] !== 'verified') {
                return ['valid' => false, 'error' => 'KYC verification required', 'severity' => 'high'];
            }
            
            return ['valid' => true, 'message' => 'User verification passed'];
            
        } catch (Exception $e) {
            return ['valid' => false, 'error' => 'Verification check failed', 'severity' => 'critical'];
        }
    }
    
    /**
     * Validate transaction limits
     */
    private function validateTransactionLimits($user_id, $amount, $currency) {
        try {
            // Get USD equivalent of transaction
            $usd_amount = $this->convertToUSD($amount, $currency);
            
            // Check single transaction limit
            if ($usd_amount > $this->max_single_transaction_limit) {
                return [
                    'valid' => false, 
                    'error' => "Transaction exceeds single transaction limit of $" . $this->max_single_transaction_limit,
                    'severity' => 'high'
                ];
            }
            
            // Check daily transaction limit
            $daily_total = $this->getDailyTransactionTotal($user_id);
            if (($daily_total + $usd_amount) > $this->max_daily_transaction_limit) {
                return [
                    'valid' => false,
                    'error' => "Transaction would exceed daily limit of $" . $this->max_daily_transaction_limit,
                    'severity' => 'high'
                ];
            }
            
            return ['valid' => true, 'message' => 'Transaction limits validated'];
            
        } catch (Exception $e) {
            return ['valid' => false, 'error' => 'Limit validation failed', 'severity' => 'critical'];
        }
    }
    
    /**
     * Detect potential fraud
     */
    private function detectFraud($user_id, $amount, $currency, $type) {
        $risk_factors = [];
        $risk_score = 0;
        
        // Check for unusual transaction patterns
        if ($this->isUnusualTransaction($user_id, $amount, $currency, $type)) {
            $risk_factors[] = 'Unusual transaction pattern';
            $risk_score += 30;
        }
        
        // Check for rapid successive transactions
        if ($this->hasRapidTransactions($user_id)) {
            $risk_factors[] = 'Rapid successive transactions';
            $risk_score += 25;
        }
        
        // Check for suspicious amounts
        if ($this->isSuspiciousAmount($amount, $currency)) {
            $risk_factors[] = 'Suspicious amount';
            $risk_score += 20;
        }
        
        // Check IP and device patterns
        if ($this->hasSuspiciousIP($user_id)) {
            $risk_factors[] = 'Suspicious IP address';
            $risk_score += 35;
        }
        
        if ($risk_score >= 95) { // Very high threshold for blocking
            return [
                'valid' => false,
                'error' => 'High fraud risk detected: ' . implode(', ', $risk_factors),
                'severity' => 'critical',
                'risk_score' => $risk_score
            ];
        } elseif ($risk_score >= 80) { // Higher threshold for warnings
            return [
                'valid' => true,
                'warning' => 'Medium fraud risk: ' . implode(', ', $risk_factors),
                'severity' => 'medium',
                'risk_score' => $risk_score,
                'requires_review' => true
            ];
        }
        
        return ['valid' => true, 'message' => 'Fraud check passed', 'risk_score' => $risk_score];
    }
    
    /**
     * Validate wallet ownership
     */
    private function validateWalletOwnership($user_id, $currency, $amount) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT balance, address, network 
                FROM crypto_wallets 
                WHERE user_id = ? AND currency = ? AND (network = 'erc20' OR network = 'ERC20' OR network = 'bep20' OR network = 'BEP20' OR network = 'trc20' OR network = 'TRC20' OR network = 'polygon' OR network = 'POLYGON' OR network = 'solana' OR network = 'SOLANA' OR network = 'btc' OR network = 'BTC')
                ORDER BY balance DESC
                LIMIT 1
            ");
            $stmt->execute([$user_id, $currency]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$wallet) {
                return ['valid' => false, 'error' => 'Wallet not found', 'severity' => 'critical'];
            }
            
            if ($wallet['balance'] < $amount) {
                return ['valid' => false, 'error' => 'Insufficient wallet balance', 'severity' => 'critical'];
            }
            
            return ['valid' => true, 'message' => 'Wallet ownership validated'];
            
        } catch (Exception $e) {
            return ['valid' => false, 'error' => 'Wallet validation failed', 'severity' => 'critical'];
        }
    }
    
    /**
     * Validate payment method
     */
    private function validatePaymentMethod($payment_method) {
        $allowed_methods = ['fiat', 'mobile_money', 'bank_transfer', 'credit_card'];
        
        if (!in_array($payment_method, $allowed_methods)) {
            return [
                'valid' => false,
                'error' => 'Invalid payment method',
                'severity' => 'high'
            ];
        }
        
        return ['valid' => true, 'message' => 'Payment method validated'];
    }
    
    /**
     * Validate amount reasonableness
     */
    private function validateAmountReasonableness($amount, $currency) {
        // Check for minimum amounts
        $minimums = [
            'BTC' => 0.00001,
            'ETH' => 0.0001, // Minimum ETH amount (lowered for small transactions)
            'USDT' => 0.01,  // Lowered from 0.1 to 0.01 USDT
            'USDC' => 0.01,  // Lowered from 0.1 to 0.01 USDC
            'SOL' => 0.0001, // Minimum SOL amount (lowered for small transactions)
            'BNB' => 0.001,  // Minimum BNB amount
            'MATIC' => 0.001, // Minimum MATIC amount
            'TRX' => 0.001,   // Minimum TRX amount
            'CELO' => 0.01    // Minimum CELO amount (stablecoin)
        ];
        
        if (isset($minimums[$currency]) && $amount < $minimums[$currency]) {
            return [
                'valid' => false,
                'error' => "Amount below minimum for $currency",
                'severity' => 'medium'
            ];
        }
        
        // Check for maximum amounts (anti-money laundering)
        $maximums = [
            'BTC' => 10,
            'ETH' => 100,
            'USDT' => 100000,
            'USDC' => 100000,
            'SOL' => 1000,   // Maximum SOL amount
            'BNB' => 1000,   // Maximum BNB amount
            'MATIC' => 100000, // Maximum MATIC amount
            'TRX' => 100000,   // Maximum TRX amount
            'CELO' => 100000   // Maximum CELO amount
        ];
        
        if (isset($maximums[$currency]) && $amount > $maximums[$currency]) {
            return [
                'valid' => false,
                'error' => "Amount above maximum for $currency - requires manual review",
                'severity' => 'high',
                'requires_review' => true
            ];
        }
        
        return ['valid' => true, 'message' => 'Amount reasonableness validated'];
    }
    
    /**
     * Process validation results
     */
    private function processValidations($validations, $transaction_type) {
        $errors = [];
        $warnings = [];
        $critical_errors = [];
        $requires_review = false;
        
        foreach ($validations as $validation_name => $result) {
            if (!$result['valid']) {
                if ($result['severity'] === 'critical') {
                    $critical_errors[] = $result['error'];
                } else {
                    $errors[] = $result['error'];
                }
            }
            
            if (isset($result['warning'])) {
                $warnings[] = $result['warning'];
            }
            
            if (isset($result['requires_review']) && $result['requires_review']) {
                $requires_review = true;
            }
        }
        
        // Log suspicious activity
        if (!empty($errors) || !empty($warnings) || $requires_review) {
            $this->logSuspiciousActivity($validations, $transaction_type);
        }
        
        // Return validation result
        if (!empty($critical_errors)) {
            return [
                'approved' => false,
                'errors' => $critical_errors,
                'warnings' => $warnings,
                'requires_review' => $requires_review,
                'message' => 'Transaction rejected due to critical security issues'
            ];
        }
        
        if (!empty($errors)) {
            return [
                'approved' => false,
                'errors' => $errors,
                'warnings' => $warnings,
                'requires_review' => $requires_review,
                'message' => 'Transaction rejected due to security concerns'
            ];
        }
        
        return [
            'approved' => true,
            'warnings' => $warnings,
            'requires_review' => $requires_review,
            'message' => 'Transaction approved'
        ];
    }
    
    /**
     * Helper methods
     */
    private function convertToUSD($amount, $currency) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT buy_price FROM crypto_prices 
                WHERE crypto = ? AND is_active = 1 
                ORDER BY created_at DESC LIMIT 1
            ");
            $stmt->execute([$currency]);
            $price = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$price) {
                return 0;
            }
            
            // The price is in UGX, convert to USD (assuming 1 USD = 3750 UGX)
            $ugx_amount = $amount * $price['buy_price'];
            $usd_amount = $ugx_amount / 3750; // Convert UGX to USD
            
            return $usd_amount;
        } catch (Exception $e) {
            return 0;
        }
    }
    
    private function getDailyTransactionTotal($user_id) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT SUM(amount * price) as total
                FROM crypto_transactions ct
                JOIN crypto_prices cp ON ct.currency = cp.crypto
                WHERE ct.user_id = ? 
                AND DATE(ct.created_at) = DATE('now')
                AND ct.type IN ('buy', 'sell')
            ");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result['total'] ?? 0;
        } catch (Exception $e) {
            return 0;
        }
    }
    
    private function isUnusualTransaction($user_id, $amount, $currency, $type) {
        // Check if this amount is significantly different from user's typical transactions
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT AVG(amount) as avg_amount, MAX(amount) as max_amount, COUNT(*) as transaction_count
                FROM crypto_transactions 
                WHERE user_id = ? AND currency = ? AND type = ?
                AND created_at >= datetime('now', '-30 days')
            ");
            $stmt->execute([$user_id, $currency, $type]);
            $stats = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // If user has less than 5 transactions, don't flag as unusual (new user)
            if ($stats['transaction_count'] < 5) {
                return false;
            }
            
            if ($stats['avg_amount']) {
                // If transaction is more than 10x the average, flag as unusual (more lenient)
                return $amount > ($stats['avg_amount'] * 10);
            }
            
            return false;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function hasRapidTransactions($user_id) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT COUNT(*) as count
                FROM crypto_transactions 
                WHERE user_id = ? 
                AND created_at >= datetime('now', '-1 minute')
            ");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // If more than 10 transactions in 1 minute, flag as rapid (more lenient for development)
            return $result['count'] > 10;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function isSuspiciousAmount($amount, $currency) {
        // Check for common suspicious amounts (like exact round numbers that might indicate testing)
        $suspicious_patterns = [
            'exact_thousands' => $amount % 1000 === 0 && $amount > 10000, // Only flag large exact thousands
            'very_small' => $amount < 0.0001, // More lenient for small amounts
            'very_large' => $amount > 10000000 // Much higher threshold for large amounts
        ];
        
        return array_filter($suspicious_patterns);
    }
    
    private function hasSuspiciousIP($user_id) {
        // This would integrate with IP reputation services
        // For now, return false as we don't have IP tracking implemented
        return false;
    }
    
    private function loadSuspiciousPatterns() {
        // Load known suspicious patterns from database or config
        $this->suspicious_patterns = [
            'amount_patterns' => [],
            'ip_patterns' => [],
            'behavior_patterns' => []
        ];
    }
    
    private function logSuspiciousActivity($validations, $transaction_type) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                INSERT INTO security_logs (
                    user_id, transaction_type, validation_results, 
                    risk_score, created_at
                ) VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $risk_score = 0;
            foreach ($validations as $validation) {
                if (isset($validation['risk_score'])) {
                    $risk_score += $validation['risk_score'];
                }
            }
            
            $stmt->execute([
                $_SESSION['user_id'] ?? null,
                $transaction_type,
                json_encode($validations),
                $risk_score
            ]);
        } catch (Exception $e) {
            error_log("Failed to log suspicious activity: " . $e->getMessage());
        }
    }
}
?>
