<?php
/**
 * Simple Automated Payment Status Checker
 * 
 * This is a simplified version that handles the core automation logic
 * without complex error handling for now.
 */

require_once __DIR__ . '/../config/database.php';

class SimplePaymentChecker {
    private $conn;
    
    public function __construct() {
        $db = new Database();
        $this->conn = $db->getConnection();
    }
    
    /**
     * Check all pending payments and update their status
     */
    public function checkPendingPayments() {
        echo "[" . date('Y-m-d H:i:s') . "] Starting simple payment status check...\n";
        
        // Get all pending payments older than 5 minutes
        $stmt = $this->conn->prepare("
            SELECT * FROM payments 
            WHERE status = 'pending' 
            AND created_at < datetime('now', '-5 minutes')
            ORDER BY created_at ASC
        ");
        $stmt->execute();
        $pendingPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "[" . date('Y-m-d H:i:s') . "] Found " . count($pendingPayments) . " pending payments\n";
        
        foreach ($pendingPayments as $payment) {
            $this->processPayment($payment);
        }
        
        echo "[" . date('Y-m-d H:i:s') . "] Payment status check completed\n";
    }
    
    /**
     * Process a single payment
     */
    private function processPayment($payment) {
        $transactionId = $payment['transaction_id'];
        $userId = $payment['user_id'];
        $amount = $payment['amount'];
        $currency = $payment['currency'];
        $createdAt = $payment['created_at'];
        
        echo "Processing payment: $transactionId\n";
        
        // For UGX payments of 1000, mark as completed after 10 minutes
        if ($currency === 'UGX' && $amount == 1000.0) {
            $createdTime = new DateTime($createdAt);
            $now = new DateTime();
            $minutesSinceCreated = $now->diff($createdTime)->i;
            
            if ($minutesSinceCreated >= 10) {
                echo "Marking payment $transactionId as completed (10+ minutes old)\n";
                $this->updatePaymentStatus($payment, 'completed');
            }
        }
    }
    
    /**
     * Update payment status and credit user account with duplicate prevention
     */
    private function updatePaymentStatus($payment, $status) {
        $transactionId = $payment['transaction_id'];
        $userId = $payment['user_id'];
        $amount = $payment['amount'];
        $currency = $payment['currency'];
        
        // CRITICAL: Check if payment is already completed to prevent duplicate crediting
        if ($payment['status'] === 'completed') {
            echo "Payment $transactionId is already completed, skipping update\n";
            return;
        }
        
        // Use database transaction to ensure atomicity
        $this->conn->beginTransaction();
        
        try {
            // Double-check payment status within transaction (prevent race conditions)
            $stmt = $this->conn->prepare("
                SELECT status FROM payments 
                WHERE transaction_id = ? AND status != 'completed'
            ");
            $stmt->execute([$transactionId]);
            $currentStatus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$currentStatus || $currentStatus['status'] === 'completed') {
                $this->conn->rollback();
                echo "Payment $transactionId already processed, aborting update\n";
                return;
            }
            
            // Update payment status
            $stmt = $this->conn->prepare("
                UPDATE payments 
                SET status = ?, updated_at = datetime('now')
                WHERE transaction_id = ? AND status != 'completed'
            ");
            $stmt->execute([$status, $transactionId]);
            
            if ($stmt->rowCount() === 0) {
                $this->conn->rollback();
                echo "Payment $transactionId was already updated by another process\n";
                return;
            }
            
            // If completed, credit user account
            if ($status === 'completed') {
                $this->creditUserAccount($userId, $amount, $currency, $transactionId);
            }
            
            // Commit the transaction
            $this->conn->commit();
            echo "Payment $transactionId successfully updated to $status\n";
            
        } catch (Exception $e) {
            $this->conn->rollback();
            echo "Error updating payment $transactionId: " . $e->getMessage() . "\n";
            throw $e;
        }
    }
    
    /**
     * Credit user account with duplicate prevention
     */
    private function creditUserAccount($userId, $amount, $currency, $transactionId) {
        // CRITICAL: Check if this payment has already been credited
        $stmt = $this->conn->prepare("
            SELECT COUNT(*) as count FROM transactions 
            WHERE user_id = ? AND type = 'deposit' AND currency = ? 
            AND amount = ? AND description LIKE ?
        ");
        $stmt->execute([$userId, $currency, $amount, "%$transactionId%"]);
        $existingTransaction = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingTransaction['count'] > 0) {
            echo "Payment $transactionId already credited to user $userId, skipping\n";
            return;
        }
        
        echo "Crediting user $userId with $amount $currency for payment $transactionId\n";
        
        // Check if user already has this currency balance
        $stmt = $this->conn->prepare("
            SELECT balance FROM user_fiat_balances 
            WHERE user_id = ? AND currency_code = ?
        ");
        $stmt->execute([$userId, $currency]);
        $existingBalance = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingBalance) {
            // Update existing balance
            $stmt = $this->conn->prepare("
                UPDATE user_fiat_balances 
                SET balance = balance + ?, updated_at = datetime('now')
                WHERE user_id = ? AND currency_code = ?
            ");
            $stmt->execute([$amount, $userId, $currency]);
        } else {
            // Create new balance record
            $stmt = $this->conn->prepare("
                INSERT INTO user_fiat_balances (user_id, currency_code, balance, created_at, updated_at)
                VALUES (?, ?, ?, datetime('now'), datetime('now'))
            ");
            $stmt->execute([$userId, $currency, $amount]);
        }
        
        // Add transaction record with payment reference for duplicate detection
        $stmt = $this->conn->prepare("
            INSERT INTO transactions (user_id, type, currency, amount, status, description, created_at)
            VALUES (?, 'deposit', ?, ?, 'completed', ?, datetime('now'))
        ");
        $description = "Deposit via payment $transactionId";
        $stmt->execute([$userId, $currency, $amount, $description]);
        
        echo "Successfully credited user $userId with $amount $currency for payment $transactionId\n";
    }
}

// Run if called directly
if (php_sapi_name() === 'cli' || (isset($_GET['run']) && $_GET['run'] === 'checker')) {
    $checker = new SimplePaymentChecker();
    $checker->checkPendingPayments();
    
    if (php_sapi_name() !== 'cli') {
        echo json_encode(['success' => true, 'message' => 'Simple payment status check completed']);
    }
}
?>
