/**
 * SMS Service
 * Handles SMS notifications for OTP and alerts
 */

class SMSService {
    private static $api_url;
    private static $api_key;
    private static $sender_id;
    
    public static function initialize() {
        require_once __DIR__ . '/../config/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        self::$api_url = EnvConfig::get('SMS_API_URL');
        self::$api_key = EnvConfig::get('SMS_API_KEY');
        self::$sender_id = EnvConfig::get('SMS_SENDER_ID', 'YELLOW');
    }
    
    public static function sendOTP($phone_number, $otp_code) {
        $message = "Your YELLOW Crypto Exchange OTP code is: {$otp_code}. Valid for 5 minutes.";
        return self::sendSMS($phone_number, $message);
    }
    
    public static function sendTransactionAlert($phone_number, $transaction_type, $amount, $currency) {
        $message = "YELLOW Crypto: {$transaction_type} of {$amount} {$currency} completed successfully.";
        return self::sendSMS($phone_number, $message);
    }
    
    public static function sendSecurityAlert($phone_number, $alert_type) {
        $message = "YELLOW Crypto Security Alert: {$alert_type} detected on your account. Contact support if this wasn't you.";
        return self::sendSMS($phone_number, $message);
    }
    
    private static function sendSMS($phone_number, $message) {
        try {
            // For development, just log the SMS
            if (EnvConfig::get('APP_ENV') === 'development') {
                error_log("SMS TO: {$phone_number}");
                error_log("SMS MESSAGE: {$message}");
                return true;
            }
            
            // In production, use actual SMS API
            if (!self::$api_url || !self::$api_key) {
                error_log("SMS API credentials not configured");
                return false;
            }
            
            $data = [
                'to' => $phone_number,
                'message' => $message,
                'sender_id' => self::$sender_id
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, self::$api_url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . self::$api_key
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $result = json_decode($response, true);
                return $result['success'] ?? false;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("SMS sending failed: " . $e->getMessage());
            return false;
        }
    }
}

// Initialize the service
SMSService::initialize();
