<?php
/**
 * System Wallet Balance Synchronization
 * Automatically syncs system wallet balances from blockchain to database
 * Ensures database reflects actual blockchain holdings
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/blockchain_apis.php';

class SystemWalletSync {
    private $db;
    private $blockchainAPIs;
    
    public function __construct() {
        $this->db = new Database();
        $this->blockchainAPIs = new BlockchainAPIs($this->db->getConnection());
    }
    
    /**
     * Sync all system wallets (user_id = 0)
     */
    public function syncAllSystemWallets() {
        $startTime = microtime(true);
        $results = [
            'success' => true,
            'synced' => 0,
            'updated' => 0,
            'errors' => [],
            'wallets' => []
        ];
        
        try {
            error_log("🔄 Starting system wallet sync...");
            
            // Get all system wallets
            $wallets = $this->getSystemWallets();
            $results['synced'] = count($wallets);
            
            error_log("📊 Found {$results['synced']} system wallets to sync");
            
            foreach ($wallets as $wallet) {
                try {
                    $syncResult = $this->syncWallet($wallet);
                    
                    if ($syncResult['updated']) {
                        $results['updated']++;
                        $results['wallets'][] = $syncResult;
                    }
                    
                } catch (Exception $e) {
                    $error = "Error syncing wallet {$wallet['address']}: " . $e->getMessage();
                    error_log("❌ " . $error);
                    $results['errors'][] = $error;
                }
            }
            
            $duration = round(microtime(true) - $startTime, 2);
            error_log("✅ System wallet sync completed in {$duration}s. Updated: {$results['updated']}/{$results['synced']}");
            
        } catch (Exception $e) {
            $results['success'] = false;
            $results['error'] = $e->getMessage();
            error_log("❌ System wallet sync failed: " . $e->getMessage());
        }
        
        return $results;
    }
    
    /**
     * Get all system wallets
     */
    private function getSystemWallets() {
        $conn = $this->db->getConnection();
        
        $stmt = $conn->prepare("
            SELECT 
                id, user_id, currency, network, address, balance, 
                is_real_blockchain, private_key_encrypted
            FROM crypto_wallets 
            WHERE user_id = 0 
            AND address IS NOT NULL 
            AND address != ''
            AND is_real_blockchain = 1
            ORDER BY network, currency
        ");
        
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Sync a specific wallet
     */
    private function syncWallet($wallet) {
        $result = [
            'wallet_id' => $wallet['id'],
            'currency' => $wallet['currency'],
            'network' => $wallet['network'],
            'address' => $wallet['address'],
            'old_balance' => $wallet['balance'],
            'new_balance' => null,
            'updated' => false
        ];
        
        try {
            // Get blockchain balance
            $blockchainBalance = $this->getBlockchainBalance($wallet['address'], $wallet['currency'], $wallet['network']);
            
            if ($blockchainBalance === null) {
                error_log("⚠️ Could not fetch balance for {$wallet['address']} ({$wallet['network']})");
                return $result;
            }
            
            $dbBalance = floatval($wallet['balance']);
            $result['new_balance'] = $blockchainBalance;
            
            // Check if balance differs (with small tolerance for floating point)
            $difference = abs($blockchainBalance - $dbBalance);
            
            if ($difference > 0.000001) {
                error_log("🔄 Syncing wallet {$wallet['address']}");
                error_log("   Currency: {$wallet['currency']} ({$wallet['network']})");
                error_log("   Old DB Balance: {$dbBalance}");
                error_log("   New Blockchain Balance: {$blockchainBalance}");
                error_log("   Difference: " . ($blockchainBalance - $dbBalance));
                
                // Update database balance
                $updated = $this->updateWalletBalance($wallet['id'], $blockchainBalance);
                
                if ($updated) {
                    $result['updated'] = true;
                    error_log("✅ Wallet synced successfully!");
                }
            } else {
                error_log("✓ Wallet {$wallet['address']} already in sync ({$blockchainBalance} {$wallet['currency']})");
            }
            
        } catch (Exception $e) {
            error_log("❌ Error syncing wallet {$wallet['address']}: " . $e->getMessage());
            throw $e;
        }
        
        return $result;
    }
    
    /**
     * Get blockchain balance for an address
     */
    private function getBlockchainBalance($address, $currency, $network) {
        try {
            $network = strtolower($network);
            
            // Map database network names to API network names
            $networkMap = [
                'trc20' => 'tron',
                'erc20' => 'ethereum',
                'bep20' => 'bsc',
                'polygon' => 'polygon',
                'solana' => 'solana',
                'base' => 'base',
                'celo' => 'celo',
                'btc' => 'bitcoin'
            ];
            
            $apiNetwork = $networkMap[$network] ?? $network;
            
            // Use BlockchainAPIs unified method
            $token = ($currency === 'USDT' || $currency === 'USDC') ? $currency : null;
            $result = $this->blockchainAPIs->getAccountBalance($apiNetwork, $address, $token);
            
            // Handle response format
            if (is_array($result)) {
                if (isset($result['success']) && $result['success'] && isset($result['balance'])) {
                    return floatval($result['balance']);
                } elseif (isset($result['error'])) {
                    error_log("⚠️ API error for {$address}: " . $result['error']);
                    return null;
                }
            } elseif (is_numeric($result)) {
                return floatval($result);
            }
            
            return null;
            
        } catch (Exception $e) {
            error_log("❌ Error fetching blockchain balance for {$address} ({$network}): " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Update wallet balance in database
     */
    private function updateWalletBalance($walletId, $newBalance) {
        $conn = $this->db->getConnection();
        
        try {
            $stmt = $conn->prepare("
                UPDATE crypto_wallets 
                SET balance = ?, 
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$newBalance, $walletId]);
            
            return true;
            
        } catch (Exception $e) {
            error_log("❌ Failed to update wallet balance: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Sync a specific system wallet by currency and network
     */
    public function syncSystemWallet($currency, $network) {
        try {
            $conn = $this->db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id, user_id, currency, network, address, balance, 
                    is_real_blockchain, private_key_encrypted
                FROM crypto_wallets 
                WHERE user_id = 0 
                AND currency = ? 
                AND (network = ? OR network = ?)
                AND address IS NOT NULL 
                AND is_real_blockchain = 1
                LIMIT 1
            ");
            
            $stmt->execute([$currency, $network, strtoupper($network)]);
            $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$wallet) {
                return [
                    'success' => false,
                    'error' => "No system wallet found for {$currency} on {$network}"
                ];
            }
            
            $syncResult = $this->syncWallet($wallet);
            $syncResult['success'] = true;
            
            return $syncResult;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

// If run directly from command line
if (php_sapi_name() === 'cli' && realpath($_SERVER['SCRIPT_FILENAME']) === __FILE__) {
    echo "🔄 Starting System Wallet Sync...\n\n";
    
    $sync = new SystemWalletSync();
    $results = $sync->syncAllSystemWallets();
    
    echo "📊 SYNC RESULTS:\n";
    echo "========================\n";
    echo "Wallets Synced: {$results['synced']}\n";
    echo "Wallets Updated: {$results['updated']}\n";
    
    if (count($results['wallets']) > 0) {
        echo "\n💰 UPDATED WALLETS:\n";
        foreach ($results['wallets'] as $wallet) {
            if ($wallet['updated']) {
                $diff = $wallet['new_balance'] - $wallet['old_balance'];
                $symbol = $diff > 0 ? '+' : '';
                echo "  - {$wallet['currency']} ({$wallet['network']})\n";
                echo "    Address: {$wallet['address']}\n";
                echo "    Old: {$wallet['old_balance']}, New: {$wallet['new_balance']} ({$symbol}{$diff})\n";
            }
        }
    }
    
    if (count($results['errors']) > 0) {
        echo "\n❌ ERRORS:\n";
        foreach ($results['errors'] as $error) {
            echo "  - {$error}\n";
        }
    }
    
    echo "\n✅ Sync completed!\n";
}
?>
