<?php
/**
 * Comprehensive Transaction Monitoring System
 * Real-time transaction tracking and status updates
 */

class TransactionMonitor {
    private $conn;
    private $blockchainAPIs;
    private $monitoringConfig;
    
    public function __construct($connection) {
        $this->conn = $connection;
        require_once __DIR__ . '/blockchain_apis.php';
        $this->blockchainAPIs = new BlockchainAPIs($connection);
        $this->monitoringConfig = $this->getMonitoringConfig();
    }
    
    /**
     * Get monitoring configuration
     */
    private function getMonitoringConfig() {
        return [
            'check_interval' => 30, // seconds
            'max_retries' => 10,
            'retry_delay' => 60, // seconds
            'confirmation_thresholds' => [
                'ethereum' => 12,
                'bsc' => 3,
                'polygon' => 30,
                'base' => 2,
                'tron' => 19,
                'solana' => 32
            ],
            'timeout_thresholds' => [
                'ethereum' => 3600, // 1 hour
                'bsc' => 900, // 15 minutes
                'polygon' => 1800, // 30 minutes
                'base' => 600, // 10 minutes
                'tron' => 1800, // 30 minutes
                'solana' => 300 // 5 minutes
            ],
            'batch_size' => 100,
            'parallel_requests' => 10
        ];
    }
    
    /**
     * Monitor all pending transactions
     */
    public function monitorAllTransactions() {
        try {
            $this->conn->beginTransaction();
            
            // Get all pending transactions
            $stmt = $this->conn->prepare("
                SELECT ct.*, cw.network, cw.currency, cw.address as wallet_address
                FROM crypto_transactions ct
                JOIN crypto_wallets cw ON ct.wallet_id = cw.id
                WHERE ct.status = 'pending'
                ORDER BY ct.created_at ASC
                LIMIT ?
            ");
            $stmt->execute([$this->monitoringConfig['batch_size']]);
            $transactions = $stmt->fetchAll();
            
            $processedCount = 0;
            $errorCount = 0;
            $results = [];
            
            foreach ($transactions as $tx) {
                $result = $this->monitorTransaction($tx);
                $results[] = $result;
                
                if ($result['success']) {
                    $processedCount++;
                } else {
                    $errorCount++;
                }
            }
            
            $this->conn->commit();
            
            // Log monitoring results
            $this->logMonitoringResults($processedCount, $errorCount, count($transactions));
            
            return [
                'success' => true,
                'processed' => $processedCount,
                'errors' => $errorCount,
                'total' => count($transactions),
                'results' => $results
            ];
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Transaction monitoring error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Monitoring failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Monitor a specific transaction
     */
    public function monitorTransaction($transaction) {
        try {
            $network = $transaction['network'];
            $txHash = $transaction['tx_hash'];
            
            // Get transaction details from blockchain
            $txDetails = $this->blockchainAPIs->getTransactionDetails($network, $txHash);
            
            if (!$txDetails['success']) {
                return $this->handleTransactionError($transaction, $txDetails['error']);
            }
            
            $txData = $txDetails['transaction'];
            
            // Update transaction status based on blockchain data
            return $this->updateTransactionStatus($transaction, $txData);
            
        } catch (Exception $e) {
            error_log("Transaction monitoring error for {$transaction['tx_hash']}: " . $e->getMessage());
            return $this->handleTransactionError($transaction, $e->getMessage());
        }
    }
    
    /**
     * Update transaction status based on blockchain data
     */
    private function updateTransactionStatus($transaction, $txData) {
        try {
            $network = $transaction['network'];
            $txHash = $transaction['tx_hash'];
            $walletId = $transaction['wallet_id'];
            $userId = $transaction['user_id'];
            
            // Determine new status
            $newStatus = $this->determineTransactionStatus($transaction, $txData, $network);
            
            if ($newStatus === $transaction['status']) {
                // No status change needed
                return [
                    'success' => true,
                    'status' => 'unchanged',
                    'tx_hash' => $txHash,
                    'current_status' => $transaction['status']
                ];
            }
            
            // Update transaction in database
            $stmt = $this->conn->prepare("
                UPDATE crypto_transactions 
                SET status = ?, updated_at = CURRENT_TIMESTAMP
                WHERE tx_hash = ?
            ");
            $stmt->execute([$newStatus, $txHash]);
            
            // If transaction is confirmed, update wallet balance
            if ($newStatus === 'confirmed' && $transaction['amount'] > 0) {
                $this->updateWalletBalance($walletId, $userId, $transaction['amount']);
                $this->recordMainTransaction($userId, $transaction);
            }
            
            // Log the status change
            $this->logStatusChange($userId, $txHash, $transaction['status'], $newStatus);
            
            return [
                'success' => true,
                'status' => 'updated',
                'tx_hash' => $txHash,
                'old_status' => $transaction['status'],
                'new_status' => $newStatus,
                'confirmations' => $txData['confirmations'] ?? 0
            ];
            
        } catch (Exception $e) {
            error_log("Transaction status update error for {$transaction['tx_hash']}: " . $e->getMessage());
            return ['success' => false, 'error' => 'Status update failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Determine transaction status based on blockchain data
     */
    private function determineTransactionStatus($transaction, $txData, $network) {
        // Check if transaction failed
        if (isset($txData['status']) && $txData['status'] === 'failed') {
            return 'failed';
        }
        
        // Check confirmations
        $confirmations = $txData['confirmations'] ?? 0;
        $requiredConfirmations = $this->monitoringConfig['confirmation_thresholds'][$network] ?? 1;
        
        if ($confirmations >= $requiredConfirmations) {
            return 'confirmed';
        }
        
        // Check if transaction is too old
        $createdAt = strtotime($transaction['created_at']);
        $timeoutThreshold = $this->monitoringConfig['timeout_thresholds'][$network] ?? 3600;
        
        if ((time() - $createdAt) > $timeoutThreshold) {
            return 'timeout';
        }
        
        // Check retry count
        $retryCount = $this->getRetryCount($transaction['tx_hash']);
        if ($retryCount >= $this->monitoringConfig['max_retries']) {
            return 'failed';
        }
        
        return 'pending';
    }
    
    /**
     * Handle transaction monitoring error
     */
    private function handleTransactionError($transaction, $error) {
        try {
            $retryCount = $this->getRetryCount($transaction['tx_hash']);
            
            if ($retryCount >= $this->monitoringConfig['max_retries']) {
                // Mark as failed after max retries
                $stmt = $this->conn->prepare("
                    UPDATE crypto_transactions 
                    SET status = 'failed', updated_at = CURRENT_TIMESTAMP
                    WHERE tx_hash = ?
                ");
                $stmt->execute([$transaction['tx_hash']]);
                
                $this->logStatusChange(
                    $transaction['user_id'],
                    $transaction['tx_hash'],
                    $transaction['status'],
                    'failed',
                    $error
                );
                
                return [
                    'success' => false,
                    'status' => 'failed',
                    'tx_hash' => $transaction['tx_hash'],
                    'error' => $error,
                    'retry_count' => $retryCount
                ];
            } else {
                // Increment retry count
                $this->incrementRetryCount($transaction['tx_hash']);
                
                return [
                    'success' => false,
                    'status' => 'retry',
                    'tx_hash' => $transaction['tx_hash'],
                    'error' => $error,
                    'retry_count' => $retryCount + 1
                ];
            }
            
        } catch (Exception $e) {
            error_log("Transaction error handling failed for {$transaction['tx_hash']}: " . $e->getMessage());
            return ['success' => false, 'error' => 'Error handling failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get retry count for a transaction
     */
    private function getRetryCount($txHash) {
        $stmt = $this->conn->prepare("
            SELECT COUNT(*) FROM audit_logs 
            WHERE action = 'transaction_retry' AND record_id = ?
        ");
        $stmt->execute([$txHash]);
        return $stmt->fetchColumn();
    }
    
    /**
     * Increment retry count for a transaction
     */
    private function incrementRetryCount($txHash) {
        $stmt = $this->conn->prepare("
            INSERT INTO audit_logs (action, table_name, record_id, created_at)
            VALUES ('transaction_retry', 'crypto_transactions', ?, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$txHash]);
    }
    
    /**
     * Update wallet balance after confirmed transaction
     */
    private function updateWalletBalance($walletId, $userId, $amount) {
        try {
            $stmt = $this->conn->prepare("
                UPDATE crypto_wallets 
                SET balance = balance + ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([$amount, $walletId, $userId]);
            
            if ($stmt->rowCount() > 0) {
                error_log("✅ Wallet balance updated: Wallet $walletId (+$amount)");
            } else {
                error_log("❌ Wallet balance update failed: Wallet $walletId");
            }
            
        } catch (Exception $e) {
            error_log("Wallet balance update error: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Record transaction in main transactions table
     */
    private function recordMainTransaction($userId, $transaction) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO transactions (user_id, type, currency, amount, status, description, reference, created_at)
                VALUES (?, 'receive', ?, ?, 'completed', ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $description = "Crypto receive from blockchain - {$transaction['network']} network";
            $reference = $transaction['tx_hash'];
            
            $stmt->execute([
                $userId,
                $transaction['currency'],
                $transaction['amount'],
                $description,
                $reference
            ]);
            
        } catch (Exception $e) {
            error_log("Main transaction recording error: " . $e->getMessage());
            // Don't throw - this is not critical
        }
    }
    
    /**
     * Log status change
     */
    private function logStatusChange($userId, $txHash, $oldStatus, $newStatus, $error = null) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, created_at)
                VALUES (?, 'transaction_status_change', 'crypto_transactions', ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $oldValues = json_encode(['status' => $oldStatus]);
            $newValues = json_encode([
                'status' => $newStatus,
                'error' => $error
            ]);
            
            $stmt->execute([$userId, $txHash, $oldValues, $newValues]);
            
        } catch (Exception $e) {
            error_log("Status change logging error: " . $e->getMessage());
        }
    }
    
    /**
     * Log monitoring results
     */
    private function logMonitoringResults($processed, $errors, $total) {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO blockchain_monitor_logs (network, currency, wallets_checked, transactions_found, errors, duration_ms, created_at)
                VALUES (?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $stmt->execute([
                'all',
                'all',
                0, // wallets_checked
                $total,
                $errors,
                0 // duration_ms (could be calculated)
            ]);
            
        } catch (Exception $e) {
            error_log("Monitoring results logging error: " . $e->getMessage());
        }
    }
    
    /**
     * Get transaction statistics
     */
    public function getTransactionStatistics($userId = null, $network = null, $timeframe = '24h') {
        try {
            $timeCondition = $this->getTimeframeCondition($timeframe);
            $userCondition = $userId ? "AND ct.user_id = ?" : "";
            $networkCondition = $network ? "AND ct.network = ?" : "";
            
            $params = [];
            if ($userId) $params[] = $userId;
            if ($network) $params[] = $network;
            
            // Get status counts
            $stmt = $this->conn->prepare("
                SELECT 
                    ct.status,
                    COUNT(*) as count,
                    SUM(ct.amount) as total_amount
                FROM crypto_transactions ct
                WHERE ct.created_at >= $timeCondition
                $userCondition $networkCondition
                GROUP BY ct.status
            ");
            $stmt->execute($params);
            $statusCounts = $stmt->fetchAll();
            
            // Get network distribution
            $stmt = $this->conn->prepare("
                SELECT 
                    ct.network,
                    COUNT(*) as count,
                    SUM(ct.amount) as total_amount
                FROM crypto_transactions ct
                WHERE ct.created_at >= $timeCondition
                $userCondition
                GROUP BY ct.network
                ORDER BY count DESC
            ");
            $stmt->execute($userId ? [$userId] : []);
            $networkDistribution = $stmt->fetchAll();
            
            // Get currency distribution
            $stmt = $this->conn->prepare("
                SELECT 
                    ct.currency,
                    COUNT(*) as count,
                    SUM(ct.amount) as total_amount
                FROM crypto_transactions ct
                WHERE ct.created_at >= $timeCondition
                $userCondition
                GROUP BY ct.currency
                ORDER BY count DESC
            ");
            $stmt->execute($userId ? [$userId] : []);
            $currencyDistribution = $stmt->fetchAll();
            
            return [
                'success' => true,
                'timeframe' => $timeframe,
                'status_counts' => $statusCounts,
                'network_distribution' => $networkDistribution,
                'currency_distribution' => $currencyDistribution
            ];
            
        } catch (Exception $e) {
            error_log("Transaction statistics error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Statistics failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get timeframe condition for SQL
     */
    private function getTimeframeCondition($timeframe) {
        switch ($timeframe) {
            case '1h':
                return "datetime('now', '-1 hour')";
            case '24h':
                return "datetime('now', '-1 day')";
            case '7d':
                return "datetime('now', '-7 days')";
            case '30d':
                return "datetime('now', '-30 days')";
            default:
                return "datetime('now', '-1 day')";
        }
    }
    
    /**
     * Get pending transactions for a user
     */
    public function getPendingTransactions($userId, $limit = 50) {
        try {
            $stmt = $this->conn->prepare("
                SELECT 
                    ct.*,
                    cw.network,
                    cw.currency,
                    cw.address as wallet_address
                FROM crypto_transactions ct
                JOIN crypto_wallets cw ON ct.wallet_id = cw.id
                WHERE ct.user_id = ? AND ct.status = 'pending'
                ORDER BY ct.created_at DESC
                LIMIT ?
            ");
            $stmt->execute([$userId, $limit]);
            
            return [
                'success' => true,
                'transactions' => $stmt->fetchAll()
            ];
            
        } catch (Exception $e) {
            error_log("Pending transactions error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to get pending transactions: ' . $e->getMessage()];
        }
    }
    
    /**
     * Get transaction history for a user
     */
    public function getTransactionHistory($userId, $limit = 100, $offset = 0) {
        try {
            $stmt = $this->conn->prepare("
                SELECT 
                    ct.*,
                    cw.network,
                    cw.currency,
                    cw.address as wallet_address
                FROM crypto_transactions ct
                JOIN crypto_wallets cw ON ct.wallet_id = cw.id
                WHERE ct.user_id = ?
                ORDER BY ct.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute([$userId, $limit, $offset]);
            
            return [
                'success' => true,
                'transactions' => $stmt->fetchAll(),
                'limit' => $limit,
                'offset' => $offset
            ];
            
        } catch (Exception $e) {
            error_log("Transaction history error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to get transaction history: ' . $e->getMessage()];
        }
    }
    
    /**
     * Force update transaction status
     */
    public function forceUpdateTransactionStatus($txHash, $newStatus, $adminUserId = null) {
        try {
            $this->conn->beginTransaction();
            
            // Get current transaction
            $stmt = $this->conn->prepare("
                SELECT * FROM crypto_transactions WHERE tx_hash = ?
            ");
            $stmt->execute([$txHash]);
            $transaction = $stmt->fetch();
            
            if (!$transaction) {
                $this->conn->rollBack();
                return ['success' => false, 'error' => 'Transaction not found'];
            }
            
            // Update status
            $stmt = $this->conn->prepare("
                UPDATE crypto_transactions 
                SET status = ?, updated_at = CURRENT_TIMESTAMP
                WHERE tx_hash = ?
            ");
            $stmt->execute([$newStatus, $txHash]);
            
            // Log the manual update
            $stmt = $this->conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, created_at)
                VALUES (?, 'manual_status_update', 'crypto_transactions', ?, ?, ?, CURRENT_TIMESTAMP)
            ");
            
            $oldValues = json_encode(['status' => $transaction['status']]);
            $newValues = json_encode(['status' => $newStatus]);
            
            $stmt->execute([$adminUserId, $txHash, $oldValues, $newValues]);
            
            $this->conn->commit();
            
            return [
                'success' => true,
                'tx_hash' => $txHash,
                'old_status' => $transaction['status'],
                'new_status' => $newStatus
            ];
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Force update error for $txHash: " . $e->getMessage());
            return ['success' => false, 'error' => 'Force update failed: ' . $e->getMessage()];
        }
    }
}
?>

