<?php
/**
 * User Management API
 * Handles user profile, KYC, and account management
 */

require_once __DIR__ . '/../config/database.php';

class UserAPI {
    private $db;
    private $user_id;

    public function __construct($user_id) {
        $this->db = new Database();
        $this->user_id = $user_id;
    }

    public function getUserProfile() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT u.*, p.*, f.balance as fiat_balance, f.currency_id
                FROM users u
                LEFT JOIN user_profiles p ON u.id = p.user_id
                LEFT JOIN user_fiat_balances f ON u.id = f.user_id AND f.currency_id = (
                    SELECT id FROM fiat_currencies WHERE code = u.currency
                )
                WHERE u.id = ?
            ");
            $stmt->execute([$this->user_id]);
            $user = $stmt->fetch();

            if (!$user) {
                return ['error' => 'User not found'];
            }

            // Remove sensitive data
            unset($user['password_hash']);
            unset($user['two_factor_secret']);
            
            // Add country info based on phone number
            $user['detected_country'] = $this->detectCountryFromPhone($user['phone']);
            $user['country_currency'] = $this->getCurrencyForCountry($user['detected_country']);

            return [
                'success' => true,
                'user' => $user
            ];
        } catch (Exception $e) {
            error_log("Get user profile error: " . $e->getMessage());
            return ['error' => 'Failed to get user profile'];
        }
    }
    
    /**
     * Detect country from phone number
     */
    private function detectCountryFromPhone($phoneNumber) {
        if (!$phoneNumber) return 'ug';
        
        // Remove any non-digit characters
        $cleanPhone = preg_replace('/\D/', '', $phoneNumber);
        
        // Country phone prefixes
        $countryPrefixes = [
            '256' => 'ug', // Uganda
            '254' => 'ke', // Kenya  
            '255' => 'tz', // Tanzania
            '250' => 'rw', // Rwanda
            '243' => 'cd', // Democratic Republic of Congo
            '234' => 'ng', // Nigeria
            '1' => 'us'    // United States
        ];
        
        // Check for country prefixes (longest first)
        $sortedPrefixes = array_keys($countryPrefixes);
        usort($sortedPrefixes, function($a, $b) {
            return strlen($b) - strlen($a);
        });
        
        foreach ($sortedPrefixes as $prefix) {
            if (strpos($cleanPhone, $prefix) === 0) {
                return $countryPrefixes[$prefix];
            }
        }
        
        // Default to Uganda
        return 'ug';
    }
    
    /**
     * Get currency for country
     */
    private function getCurrencyForCountry($countryCode) {
        $currencies = [
            'ug' => 'UGX',
            'ke' => 'KES',
            'tz' => 'TZS', 
            'rw' => 'RWF',
            'cd' => 'CDF',
            'ng' => 'NGN',
            'us' => 'USD'
        ];
        
        return $currencies[$countryCode] ?? 'UGX';
    }

    public function updateProfile($data) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();

            // Update user basic info
            $allowed_fields = ['first_name', 'last_name', 'phone'];
            $update_fields = [];
            $update_values = [];

            foreach ($allowed_fields as $field) {
                if (isset($data[$field])) {
                    $update_fields[] = "$field = ?";
                    $update_values[] = $data[$field];
                }
            }

            if (!empty($update_fields)) {
                $update_values[] = $this->user_id;
                $stmt = $conn->prepare("
                    UPDATE users SET " . implode(', ', $update_fields) . " WHERE id = ?
                ");
                $stmt->execute($update_values);
            }

            // Update or create profile
            $profile_fields = ['date_of_birth', 'address', 'city', 'state', 'postal_code'];
            $profile_data = [];
            foreach ($profile_fields as $field) {
                if (isset($data[$field])) {
                    $profile_data[$field] = $data[$field];
                }
            }

            if (!empty($profile_data)) {
                $stmt = $conn->prepare("
                    INSERT INTO user_profiles (user_id, " . implode(', ', array_keys($profile_data)) . ")
                    VALUES (?, " . str_repeat('?,', count($profile_data) - 1) . "?)
                    ON DUPLICATE KEY UPDATE " . implode(' = ?, ', array_keys($profile_data)) . " = ?"
                );
                $values = array_merge([$this->user_id], array_values($profile_data), array_values($profile_data));
                $stmt->execute($values);
            }

            $conn->commit();
            return ['success' => true, 'message' => 'Profile updated successfully'];

        } catch (Exception $e) {
            $conn->rollback();
            error_log("Update profile error: " . $e->getMessage());
            return ['error' => 'Failed to update profile'];
        }
    }

    public function submitKYC($data) {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();

            // Validate required fields
            $required_fields = ['id_type', 'id_number', 'id_front_image', 'id_back_image', 'selfie_image'];
            foreach ($required_fields as $field) {
                if (empty($data[$field])) {
                    return ['error' => "Field $field is required"];
                }
            }

            // Update profile with KYC data
            $stmt = $conn->prepare("
                INSERT INTO user_profiles (user_id, id_type, id_number, id_front_image, id_back_image, selfie_image)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                id_type = VALUES(id_type),
                id_number = VALUES(id_number),
                id_front_image = VALUES(id_front_image),
                id_back_image = VALUES(id_back_image),
                selfie_image = VALUES(selfie_image)
            ");
            $stmt->execute([
                $this->user_id,
                $data['id_type'],
                $data['id_number'],
                $data['id_front_image'],
                $data['id_back_image'],
                $data['selfie_image']
            ]);

            // Update KYC status
            $stmt = $conn->prepare("UPDATE users SET kyc_status = 'pending' WHERE id = ?");
            $stmt->execute([$this->user_id]);

            $conn->commit();
            return ['success' => true, 'message' => 'KYC documents submitted successfully'];

        } catch (Exception $e) {
            $conn->rollback();
            error_log("KYC submission error: " . $e->getMessage());
            return ['error' => 'Failed to submit KYC documents'];
        }
    }

    public function getReferralInfo() {
        try {
            $conn = $this->db->getConnection();
            
            // Get user's referral code
            $stmt = $conn->prepare("SELECT referral_code FROM users WHERE id = ?");
            $stmt->execute([$this->user_id]);
            $user = $stmt->fetch();

            // Get referrals made by this user
            $stmt = $conn->prepare("
                SELECT r.*, u.first_name, u.last_name, u.email, u.created_at
                FROM referrals r
                JOIN users u ON r.referred_id = u.id
                WHERE r.referrer_id = ?
                ORDER BY r.created_at DESC
            ");
            $stmt->execute([$this->user_id]);
            $referrals = $stmt->fetchAll();

            // Get total commission earned
            $stmt = $conn->prepare("
                SELECT SUM(ct.commission_amount) as total_earned
                FROM commission_transactions ct
                JOIN referrals r ON ct.referral_id = r.id
                WHERE r.referrer_id = ?
            ");
            $stmt->execute([$this->user_id]);
            $commission = $stmt->fetch();

            return [
                'success' => true,
                'referral_code' => $user['referral_code'],
                'referrals' => $referrals,
                'total_earned' => $commission['total_earned'] ?? 0
            ];

        } catch (Exception $e) {
            error_log("Get referral info error: " . $e->getMessage());
            return ['error' => 'Failed to get referral information'];
        }
    }

    public function freezeAccount() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("UPDATE users SET is_active = 0 WHERE id = ?");
            $stmt->execute([$this->user_id]);

            // Log the action
            $this->logAction('account_freeze', 'users', $this->user_id);

            return ['success' => true, 'message' => 'Account frozen successfully'];

        } catch (Exception $e) {
            error_log("Freeze account error: " . $e->getMessage());
            return ['error' => 'Failed to freeze account'];
        }
    }

    public function deleteAccount() {
        try {
            $conn = $this->db->getConnection();
            $conn->beginTransaction();

            // Check if user has any pending transactions
            $stmt = $conn->prepare("
                SELECT COUNT(*) as count FROM transactions 
                WHERE user_id = ? AND status IN ('pending', 'processing')
            ");
            $stmt->execute([$this->user_id]);
            $pending = $stmt->fetch();

            if ($pending['count'] > 0) {
                return ['error' => 'Cannot delete account with pending transactions'];
            }

            // Check if user has any balance
            $stmt = $conn->prepare("
                SELECT SUM(balance) as total_balance FROM user_wallets WHERE user_id = ?
                UNION ALL
                SELECT SUM(balance) as total_balance FROM user_fiat_balances WHERE user_id = ?
            ");
            $stmt->execute([$this->user_id, $this->user_id]);
            $balances = $stmt->fetchAll();

            $total_balance = array_sum(array_column($balances, 'total_balance'));
            if ($total_balance > 0) {
                return ['error' => 'Cannot delete account with remaining balance'];
            }

            // Delete user (cascade will handle related records)
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->execute([$this->user_id]);

            $conn->commit();
            return ['success' => true, 'message' => 'Account deleted successfully'];

        } catch (Exception $e) {
            $conn->rollback();
            error_log("Delete account error: " . $e->getMessage());
            return ['error' => 'Failed to delete account'];
        }
    }

    private function logAction($action, $table_name, $record_id, $old_values = null, $new_values = null) {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $this->user_id,
                $action,
                $table_name,
                $record_id,
                $old_values ? json_encode($old_values) : null,
                $new_values ? json_encode($new_values) : null,
                $_SERVER['REMOTE_ADDR'] ?? '',
                $_SERVER['HTTP_USER_AGENT'] ?? ''
            ]);
        } catch (Exception $e) {
            error_log("Audit log error: " . $e->getMessage());
        }
    }
}
?>
