<?php
/**
 * Database Configuration
 * Cryptocurrency Exchange System
 */

class Database {
    private $host;
    private $db_name;
    private $username;
    private $password;
    private $conn;

    public function __construct() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        $this->host = EnvConfig::get('DB_HOST', 'localhost');
        $this->db_name = EnvConfig::get('DB_NAME', 'crypto_exchange');
        $this->username = EnvConfig::get('DB_USER', 'root');
        $this->password = EnvConfig::get('DB_PASS', '');
        
        $this->ensureDatabaseExists();
    }

    private function ensureDatabaseExists() {
        try {
            // First connect without database to create it
            $conn = new PDO(
                "mysql:host={$this->host};charset=utf8mb4",
                $this->username,
                $this->password,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            // Create database if it doesn't exist
            $conn->exec("CREATE DATABASE IF NOT EXISTS `{$this->db_name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            $conn = null;
            
            // Now connect to the database
            $this->conn = new PDO(
                "mysql:host={$this->host};dbname={$this->db_name};charset=utf8mb4",
                $this->username,
                $this->password,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                    PDO::ATTR_TIMEOUT => 60,
                    PDO::ATTR_PERSISTENT => false,
                ]
            );
            
            // Check if tables exist, if not create them
            $this->createTablesIfNotExist();
            
        } catch(PDOException $exception) {
            error_log("Database connection error: " . $exception->getMessage());
            throw new Exception("Database connection failed: " . $exception->getMessage());
        }
    }

    public function getConnection() {
        if (!$this->conn) {
            $this->ensureDatabaseExists();
        }
        return $this->conn;
    }

    public function closeConnection() {
        if ($this->conn) {
            $this->conn = null;
        }
    }

    public function __destruct() {
        $this->closeConnection();
    }

    private function createTablesIfNotExist() {
        $conn = $this->conn;
        
        // Read and execute the MySQL schema
        $schema_file = __DIR__ . '/../database/schema.sql';
        if (file_exists($schema_file)) {
            $sql = file_get_contents($schema_file);
            
            // Split SQL into individual statements
            $statements = array_filter(array_map('trim', explode(';', $sql)));
            
            foreach ($statements as $statement) {
                if (!empty($statement) && !preg_match('/^(CREATE DATABASE|USE)/i', $statement)) {
                    try {
                        $conn->exec($statement);
                    } catch (PDOException $e) {
                        // Ignore "table already exists" errors
                        if (strpos($e->getMessage(), 'already exists') === false) {
                            error_log("Schema execution error: " . $e->getMessage());
                        }
                    }
                }
            }
        }
    }
}

// Database configuration constants
define('DB_HOST', 'localhost');
define('DB_NAME', 'crypto_exchange');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');

// Application constants
define('APP_NAME', 'YELLOW Crypto Exchange');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'http://localhost:8000');
define('API_URL', 'http://localhost:8000/api');

// Security constants
define('JWT_SECRET', 'your-secret-key-change-this-in-production');
define('ENCRYPTION_KEY', 'your-encryption-key-change-this-in-production');
define('BCRYPT_ROUNDS', 12);

// Rate limiting
define('RATE_LIMIT_REQUESTS', 100);
define('RATE_LIMIT_WINDOW', 60); // seconds

// File upload limits
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_FILE_TYPES', ['jpg', 'jpeg', 'png', 'pdf']);

// Transaction limits
define('MIN_TRADE_AMOUNT', 0.00000001);
define('MAX_TRADE_AMOUNT', 1000000.00000000);
define('MIN_WITHDRAWAL', 0.00000001);
define('MAX_WITHDRAWAL', 1000000.00000000);

// Commission rates
define('DEFAULT_COMMISSION_RATE', 20.00); // Percentage
define('TRADING_FEE_RATE', 0.25); // Percentage

// Timeouts
define('SESSION_TIMEOUT', 24 * 60 * 60); // 24 hours
define('API_TIMEOUT', 30); // seconds

// Logging
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_FILE', __DIR__ . '/../logs/app.log');

// Email configuration
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password');
define('SMTP_FROM_EMAIL', 'noreply@yellowcrypto.com');
define('SMTP_FROM_NAME', 'YELLOW Crypto Exchange');

// SMS configuration (for OTP)
define('SMS_API_KEY', 'your-sms-api-key');
define('SMS_SENDER_ID', 'YELLOW');

// External APIs
define('COINMARKETCAP_API_KEY', 'your-coinmarketcap-api-key');
define('BLOCKCHAIN_API_URL', 'https://api.blockchain.info');
define('ETHERSCAN_API_KEY', 'your-etherscan-api-key');
define('BSCSCAN_API_KEY', 'your-bscscan-api-key');

// Error reporting
if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}

// Timezone
date_default_timezone_set('Africa/Kampala');
?>
