<?php
/**
 * Database Configuration
 * Cryptocurrency Exchange System - SQLite Support
 */

class Database {
    private $host;
    private $db_name;
    private $username;
    private $password;
    private $db_type;
    private $conn;

    public function __construct() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env.local');
        
        $this->db_type = EnvConfig::get('DB_TYPE', 'mysql');
        $this->host = EnvConfig::get('DB_HOST', 'localhost');
        $this->db_name = EnvConfig::get('DB_NAME', 'crypto_exchange');
        $this->username = EnvConfig::get('DB_USER', 'root');
        $this->password = EnvConfig::get('DB_PASS', '');
        
        $this->ensureDatabaseExists();
    }

    private function ensureDatabaseExists() {
        try {
            if ($this->db_type === 'sqlite') {
                // SQLite connection
                $db_path = __DIR__ . '/../' . $this->db_name;
                $this->conn = new PDO(
                    "sqlite:{$db_path}",
                    null,
                    null,
                    [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_EMULATE_PREPARES => false,
                    ]
                );
                
                // Create tables if they don't exist
                $this->createTablesIfNotExist();
            } else {
                // MySQL connection (original code)
                $conn = new PDO(
                    "mysql:host={$this->host};charset=utf8mb4",
                    $this->username,
                    $this->password,
                    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                );
                
                // Create database if it doesn't exist
                $conn->exec("CREATE DATABASE IF NOT EXISTS `{$this->db_name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                $conn = null;
                
                // Now connect to the database
                $this->conn = new PDO(
                    "mysql:host={$this->host};dbname={$this->db_name};charset=utf8mb4",
                    $this->username,
                    $this->password,
                    [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_EMULATE_PREPARES => false,
                        PDO::ATTR_TIMEOUT => 60,
                        PDO::ATTR_PERSISTENT => false,
                    ]
                );
                
                // Check if tables exist, if not create them
                $this->createTablesIfNotExist();
            }
            
        } catch(PDOException $exception) {
            error_log("Database connection error: " . $exception->getMessage());
            throw new Exception("Database connection failed: " . $exception->getMessage());
        }
    }

    public function getConnection() {
        if (!$this->conn) {
            $this->ensureDatabaseExists();
        }
        return $this->conn;
    }

    public function closeConnection() {
        if ($this->conn) {
            $this->conn = null;
        }
    }

    public function __destruct() {
        $this->closeConnection();
    }

    private function createTablesIfNotExist() {
        $conn = $this->conn;
        
        if ($this->db_type === 'sqlite') {
            // SQLite table creation
            $this->createSQLiteTables($conn);
        } else {
            // MySQL table creation (original code)
            $schema_file = __DIR__ . '/../database/schema.sql';
            if (file_exists($schema_file)) {
                $sql = file_get_contents($schema_file);
                
                // Split SQL into individual statements
                $statements = array_filter(array_map('trim', explode(';', $sql)));
                
                foreach ($statements as $statement) {
                    if (!empty($statement) && !preg_match('/^(CREATE DATABASE|USE)/i', $statement)) {
                        try {
                            $conn->exec($statement);
                        } catch (PDOException $e) {
                            // Ignore "table already exists" errors
                            if (strpos($e->getMessage(), 'already exists') === false) {
                                error_log("Schema execution error: " . $e->getMessage());
                            }
                        }
                    }
                }
            }
        }
    }
    
    private function createSQLiteTables($conn) {
        // Create essential tables for SQLite
        $tables = [
            // Users table
            "CREATE TABLE IF NOT EXISTS users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                email TEXT UNIQUE NOT NULL,
                phone TEXT UNIQUE,
                password_hash TEXT NOT NULL,
                first_name TEXT NOT NULL,
                last_name TEXT NOT NULL,
                country_code TEXT NOT NULL DEFAULT 'UG',
                currency TEXT NOT NULL DEFAULT 'UGX',
                is_verified INTEGER DEFAULT 0,
                is_active INTEGER DEFAULT 1,
                kyc_status TEXT DEFAULT 'pending',
                referral_code TEXT UNIQUE,
                referred_by INTEGER,
                two_factor_enabled INTEGER DEFAULT 0,
                two_factor_secret TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (referred_by) REFERENCES users(id)
            )",
            
            // User profiles table
            "CREATE TABLE IF NOT EXISTS user_profiles (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                date_of_birth DATE,
                address TEXT,
                city TEXT,
                state TEXT,
                postal_code TEXT,
                id_type TEXT DEFAULT 'national_id',
                id_number TEXT,
                id_front_image TEXT,
                id_back_image TEXT,
                selfie_image TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )",
            
            // Cryptocurrencies table
            "CREATE TABLE IF NOT EXISTS cryptocurrencies (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                symbol TEXT UNIQUE NOT NULL,
                name TEXT NOT NULL,
                logo_url TEXT,
                is_active INTEGER DEFAULT 1,
                min_withdrawal REAL DEFAULT 0.00000001,
                max_withdrawal REAL DEFAULT 1000000.00000000,
                withdrawal_fee REAL DEFAULT 0.00000000,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )",
            
            // Networks table
            "CREATE TABLE IF NOT EXISTS networks (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                symbol TEXT NOT NULL,
                icon TEXT,
                fee_usd REAL DEFAULT 0.00,
                is_active INTEGER DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )",
            
            // User wallets table
            "CREATE TABLE IF NOT EXISTS user_wallets (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                crypto_id INTEGER NOT NULL,
                network_id INTEGER NOT NULL,
                address TEXT NOT NULL,
                private_key_encrypted TEXT,
                balance REAL DEFAULT 0.00000000,
                locked_balance REAL DEFAULT 0.00000000,
                is_active INTEGER DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (crypto_id) REFERENCES cryptocurrencies(id),
                FOREIGN KEY (network_id) REFERENCES networks(id),
                UNIQUE(user_id, crypto_id, network_id)
            )",
            
            // Transactions table
            "CREATE TABLE IF NOT EXISTS transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                type TEXT NOT NULL,
                crypto_id INTEGER,
                network_id INTEGER,
                amount REAL,
                fee REAL DEFAULT 0.00000000,
                from_address TEXT,
                to_address TEXT,
                tx_hash TEXT,
                status TEXT DEFAULT 'pending',
                confirmation_count INTEGER DEFAULT 0,
                required_confirmations INTEGER DEFAULT 3,
                notes TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id),
                FOREIGN KEY (crypto_id) REFERENCES cryptocurrencies(id),
                FOREIGN KEY (network_id) REFERENCES networks(id)
            )",
            
            // System settings table
            "CREATE TABLE IF NOT EXISTS system_settings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                setting_key TEXT UNIQUE NOT NULL,
                setting_value TEXT,
                description TEXT,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )"
        ];
        
        foreach ($tables as $table) {
            try {
                $conn->exec($table);
            } catch (PDOException $e) {
                error_log("Table creation error: " . $e->getMessage());
            }
        }
    }
}

// Database configuration constants
define('DB_HOST', 'localhost');
define('DB_NAME', 'crypto_exchange');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');

// Application constants
define('APP_NAME', 'YELLOW Crypto Exchange');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'http://localhost:8000');
define('API_URL', 'http://localhost:8000/api');

// Security constants
define('JWT_SECRET', 'your-secret-key-change-this-in-production');
define('ENCRYPTION_KEY', 'your-encryption-key-change-this-in-production');
define('BCRYPT_ROUNDS', 12);

// Rate limiting
define('RATE_LIMIT_REQUESTS', 100);
define('RATE_LIMIT_WINDOW', 60); // seconds

// File upload limits
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_FILE_TYPES', ['jpg', 'jpeg', 'png', 'pdf']);

// Transaction limits
define('MIN_TRADE_AMOUNT', 0.00000001);
define('MAX_TRADE_AMOUNT', 1000000.00000000);
define('MIN_WITHDRAWAL', 0.00000001);
define('MAX_WITHDRAWAL', 1000000.00000000);

// Commission rates
define('DEFAULT_COMMISSION_RATE', 20.00); // Percentage
define('TRADING_FEE_RATE', 0.25); // Percentage

// Timeouts
define('SESSION_TIMEOUT', 24 * 60 * 60); // 24 hours
define('API_TIMEOUT', 30); // seconds

// Logging
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_FILE', __DIR__ . '/../logs/app.log');

// Email configuration
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password');
define('SMTP_FROM_EMAIL', 'noreply@yellowcrypto.com');
define('SMTP_FROM_NAME', 'YELLOW Crypto Exchange');

// SMS configuration (for OTP)
define('SMS_API_KEY', 'your-sms-api-key');
define('SMS_SENDER_ID', 'YELLOW');

// External APIs
define('COINMARKETCAP_API_KEY', 'your-coinmarketcap-api-key');
define('BLOCKCHAIN_API_URL', 'https://api.blockchain.info');
define('ETHERSCAN_API_KEY', 'your-etherscan-api-key');
define('BSCSCAN_API_KEY', 'your-bscscan-api-key');

// Error reporting
if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}

// Timezone
date_default_timezone_set('Africa/Kampala');
?>
