<?php
/**
 * Environment Configuration Loader
 * Loads configuration from config.env file
 */

class EnvConfig {
    private static $config = [];
    private static $loaded = false;

    public static function load($envFile = null) {
        if (self::$loaded) {
            return self::$config;
        }

        if ($envFile === null) {
            $envFile = __DIR__ . '/../config.env';
        }

        if (file_exists($envFile)) {
            $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            
            foreach ($lines as $line) {
                // Skip comments
                if (strpos(trim($line), '#') === 0) {
                    continue;
                }
                
                // Parse key=value pairs
                if (strpos($line, '=') !== false) {
                    list($key, $value) = explode('=', $line, 2);
                    $key = trim($key);
                    $value = trim($value);
                    
                    // Remove quotes if present
                    if ((substr($value, 0, 1) === '"' && substr($value, -1) === '"') ||
                        (substr($value, 0, 1) === "'" && substr($value, -1) === "'")) {
                        $value = substr($value, 1, -1);
                    }
                    
                    self::$config[$key] = $value;
                }
            }
        }

        self::$loaded = true;
        return self::$config;
    }

    public static function get($key, $default = null) {
        if (!self::$loaded) {
            self::load();
        }
        
        return isset(self::$config[$key]) ? self::$config[$key] : $default;
    }

    public static function getBool($key, $default = false) {
        $value = self::get($key, $default);
        
        if (is_bool($value)) {
            return $value;
        }
        
        return in_array(strtolower($value), ['true', '1', 'yes', 'on']);
    }

    public static function getInt($key, $default = 0) {
        return (int) self::get($key, $default);
    }

    public static function getFloat($key, $default = 0.0) {
        return (float) self::get($key, $default);
    }

    public static function getAll() {
        if (!self::$loaded) {
            self::load();
        }
        
        return self::$config;
    }
}

