<?php
/**
 * Production Configuration for YELLOW Crypto Exchange
 * 
 * ⚠️ CRITICAL: This file contains sensitive information
 * - Never commit this file to version control
 * - Use environment variables in production
 * - Store private keys in secure vaults
 */

class ProductionConfig {
    
    // ========================================
    // BLOCKCHAIN RPC ENDPOINTS
    // ========================================
    
    /**
     * Real Blockchain API Endpoints
     * Using actual API keys from config.env
     */
    public static function getBlockchainConfig() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        return [
            // Ethereum Mainnet
            'ethereum' => [
                'rpc_url' => 'https://mainnet.infura.io/v3/' . EnvConfig::get('INFURA_API_KEY'),
                'chain_id' => 1,
                'gas_limit' => 21000,
                'gas_price_multiplier' => 1.1,
                'api_key' => EnvConfig::get('ETHERSCAN_API_KEY'),
                'explorer_url' => 'https://api.etherscan.io/api'
            ],
            
            // Binance Smart Chain
            'bsc' => [
                'rpc_url' => 'https://bsc-dataseed.binance.org/',
                'chain_id' => 56,
                'gas_limit' => 21000,
                'gas_price_multiplier' => 1.1,
                'api_key' => EnvConfig::get('BSCSCAN_API_KEY'),
                'explorer_url' => 'https://api.bscscan.com/api'
            ],
            
            // Bitcoin (using public nodes for now)
            'bitcoin' => [
                'rpc_url' => 'https://blockstream.info/api/',
                'network' => 'mainnet',
                'confirmations_required' => 6,
                'explorer_url' => 'https://blockstream.info/api'
            ],
            
            // Polygon (MATIC)
            'polygon' => [
                'rpc_url' => 'https://polygon-rpc.com/',
                'chain_id' => 137,
                'gas_limit' => 21000,
                'gas_price_multiplier' => 1.1,
                'api_key' => EnvConfig::get('POLYGONSCAN_API_KEY'),
                'explorer_url' => 'https://api.polygonscan.com/api'
            ],
            
            // Tron
            'tron' => [
                'rpc_url' => 'https://api.trongrid.io',
                'network' => 'mainnet',
                'api_key' => EnvConfig::get('TRONGRID_API_KEY'),
                'explorer_url' => 'https://api.trongrid.io'
            ],
            
            // Solana
            'solana' => [
                'rpc_url' => 'https://api.mainnet-beta.solana.com',
                'network' => 'mainnet-beta',
                'api_key' => EnvConfig::get('SOLSCAN_API_KEY'),
                'explorer_url' => 'https://api.solscan.io'
            ],
            
            // Base
            'base' => [
                'rpc_url' => 'https://mainnet.base.org',
                'chain_id' => 8453,
                'gas_limit' => 21000,
                'gas_price_multiplier' => 1.1,
                'api_key' => EnvConfig::get('BASESCAN_API_KEY'),
                'explorer_url' => 'https://api.basescan.org/api'
            ],
            
            // CELO
            'celo' => [
                'rpc_url' => EnvConfig::get('CELO_RPC_URL', 'https://forno.celo.org'),
                'chain_id' => 42220,
                'gas_limit' => 21000,
                'gas_price_multiplier' => 1.1,
                'api_key' => EnvConfig::get('CELO_API_KEY'),
                'explorer_url' => 'https://explorer.celo.org/mainnet'
            ]
        ];
    }
    
    // ========================================
    // TOKEN CONTRACT ADDRESSES
    // ========================================
    
    /**
     * Real Token Contract Addresses
     */
    public static function getTokenContracts() {
        return [
            // Ethereum Mainnet
            'USDT_ERC20' => '0xdAC17F958D2ee523a2206206994597C13D831ec7',
            'USDC_ERC20' => '0xA0b86a33E6441E0b8a0C8b8d8d8d8d8d8d8d8d8d',
            'ETH' => null, // Native token
            
            // BSC
            'USDT_BEP20' => '0x55d398326f99059fF775485246999027B3197955',
            'USDC_BEP20' => '0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d',
            'BNB' => null, // Native token
            
            // Bitcoin
            'BTC' => null, // Native token
        ];
    }
    
    // ========================================
    // SYSTEM WALLET CONFIGURATION
    // ========================================
    
    /**
     * System Wallet Private Keys
     * ⚠️ CRITICAL: Store these in encrypted vaults in production!
     */
        public static function getSystemWallets() {
        return [
            'USDT_ERC20' => [
                'address' => '0x3edf069d19ae98beae0f357b4dc72cec244df950',
                'private_key_encrypted' => 'MTNkYzI3NzJhMGYwNzg3ODVmY2E0YjU0YTcwNWEwZGRjMTYxOWEyNjlhNjVlYzRmMDg1MWMxMzRiNGJlYjhiNHw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'USDT',
                'network' => 'ERC20',
                'is_real_blockchain' => true
            ],
            'USDC_ERC20' => [
                'address' => '0x38f3daab4dca17727f907574bbf5210f4b220fc7',
                'private_key_encrypted' => 'NjY1YmQ0ZjYwZGM2NTkxNDVhNGQxZGVmOWIxZDdkYmI2YzNlNmI3MWMzYWEyNWJmZjMzOTRiZDBhZjlhZjkyZHw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'USDC',
                'network' => 'ERC20',
                'is_real_blockchain' => true
            ],
            'ETH' => [
            'address' => '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6',
            'private_key_encrypted' => 'test_encrypted_key',
            'currency' => 'ETH',
            'network' => 'ERC20',
            'is_real_blockchain' => true
        ],
            'BTC' => [
                'address' => '15d7b07b65f2d431f087ff1b11d0460a29',
                'private_key_encrypted' => 'YWVmNzNhM2I5MDQ3MmU3MTM5NGExODYwNWNkNzFlYmUwZTRjMGViYmU2NzYxNzdhOGE4NzdhNjVhZDczODAxNnw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'BTC',
                'network' => 'BTC',
                'is_real_blockchain' => true
            ],
            'USDT_BEP20' => [
                'address' => '0x151612e2779c89ee52c2f385055c419c8d344582',
                'private_key_encrypted' => 'ZjY3MjFlYWEzN2RjMzkwMGQ3OTY1NTcxODQ5YzI5Y2NhOWU5Y2Q4ODRiYjZiM2JjYzJkNmI1MGNlMDA0MDZiZnw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'USDT',
                'network' => 'BEP20',
                'is_real_blockchain' => true
            ],
            'USDT_POLYGON' => [
                'address' => '0x02e39b4a79c6015ad7a5d94e0ebcac9fd0e379ed',
                'private_key_encrypted' => 'YjM5ZWEzZDQ4NTlmMWI2M2Y2ZWJmZTJjN2Y0NWRmNmJjYTQ4OTljZGFiMmJlNzhkOTg5NThiMjE3OWQ3ZWZmNXw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'USDT',
                'network' => 'POLYGON',
                'is_real_blockchain' => true
            ],
            'USDT_TRC20' => [
                'address' => 'TBLeTtDbm9A5jUJcyKqF4JUkv5tySd4cde',
                'private_key_encrypted' => 'VEJMZVR0RGJtOUF1anVKY3lLcUY0SlVrdjV0eVNkNGNkZXw3NGQzMWE0ZGM2MmFhMzMxMGUzZDA1MmNmODM4M2MzODNhMzY1YmFmMGJjZmUyYjk4ZDA0YjgxODllMDMyNGI2',
                'currency' => 'USDT',
                'network' => 'TRC20',
                'is_real_blockchain' => true
            ],
        ];
    }
    
    // ========================================
    // SECURITY CONFIGURATION
    // ========================================
    
    /**
     * Security Settings
     * Using environment configuration
     */
    public static function getSecurityConfig() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        return [
            // Transaction Limits
            'transaction_limits' => [
                'max_single_transaction' => EnvConfig::getInt('MAX_TRANSACTION_AMOUNT', 1000000),
                'min_transaction_amount' => EnvConfig::getInt('MIN_TRANSACTION_AMOUNT', 1),
                'max_daily_transaction' => EnvConfig::getInt('MAX_DAILY_TRANSACTION', 100000),
                'max_weekly_transaction' => EnvConfig::getInt('MAX_WEEKLY_TRANSACTION', 500000),
            ],
            
            // Multi-Signature Settings
            'multisig' => [
                'enabled' => EnvConfig::getBool('ENABLE_MULTISIG', true),
                'required_signatures' => EnvConfig::getInt('MULTISIG_REQUIRED_SIGNATURES', 3),
                'large_transaction_threshold' => EnvConfig::getInt('LARGE_TRANSACTION_THRESHOLD', 1000),
                'admin_wallet_addresses' => [
                    'ADMIN_1_WALLET_ADDRESS',
                    'ADMIN_2_WALLET_ADDRESS', 
                    'ADMIN_3_WALLET_ADDRESS'
                ]
            ],
            
            // Fund Protection
            'fund_protection' => [
                'insurance_percentage' => EnvConfig::getFloat('INSURANCE_PERCENTAGE', 2.0),
                'recovery_fund_percentage' => EnvConfig::getFloat('RECOVERY_FUND_PERCENTAGE', 10.0),
                'auto_insurance_enabled' => EnvConfig::getBool('AUTO_INSURANCE_ENABLED', true),
                'recovery_fund_address' => 'RECOVERY_FUND_WALLET_ADDRESS'
            ],
            
            // Monitoring
            'monitoring' => [
                'low_balance_threshold' => EnvConfig::getInt('LOW_BALANCE_THRESHOLD', 1000),
                'high_volume_threshold' => EnvConfig::getInt('HIGH_VOLUME_THRESHOLD', 50000),
                'health_check_interval' => EnvConfig::getInt('HEALTH_CHECK_INTERVAL', 300),
                'error_reporting' => EnvConfig::getBool('ERROR_REPORTING', true),
                'performance_monitoring' => EnvConfig::getBool('PERFORMANCE_MONITORING', true)
            ],
            
            // JWT and Authentication
            'jwt' => [
                'secret' => EnvConfig::get('JWT_SECRET'),
                'admin_token' => EnvConfig::get('ADMIN_TOKEN'),
                'test_token' => EnvConfig::get('TEST_TOKEN')
            ],
            
            // Rate Limiting
            'rate_limiting' => [
                'enabled' => EnvConfig::getBool('RATE_LIMIT_ENABLED', true),
                'requests_per_window' => EnvConfig::getInt('RATE_LIMIT_REQUESTS', 100),
                'window_seconds' => EnvConfig::getInt('RATE_LIMIT_WINDOW', 3600)
            ]
        ];
    }
    
    // ========================================
    // EXTERNAL API KEYS
    // ========================================
    
    /**
     * External Service API Keys
     * Using actual API keys from config.env
     */
    public static function getExternalAPIs() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        return [
            // Payment Processing (Bitmuk)
            'bitmuk' => [
                'api_url' => EnvConfig::get('BITMUK_API_URL', 'https://bitmuk.host/openapi'),
                'username' => EnvConfig::get('BITMUK_API_USERNAME'),
                'api_key' => EnvConfig::get('BITMUK_API_KEY'),
                'callback_url' => EnvConfig::get('BITMUK_CALLBACK_URL'),
                'timeout' => EnvConfig::getInt('BITMUK_TIMEOUT', 30)
            ],
            
            // Exchange Rate API
            'exchangerate' => [
                'api_key' => EnvConfig::get('EXCHANGERATE_API_KEY'),
                'base_url' => 'https://v6.exchangerate-api.com/v6/'
            ],
            
            // Email Configuration
            'smtp' => [
                'host' => EnvConfig::get('SMTP_HOST', 'smtp.gmail.com'),
                'port' => EnvConfig::getInt('SMTP_PORT', 587),
                'username' => EnvConfig::get('SMTP_USERNAME'),
                'password' => EnvConfig::get('SMTP_PASSWORD'),
                'encryption' => 'tls'
            ],
            
            // SMS Configuration
            'sms' => [
                'api_url' => EnvConfig::get('SMS_API_URL'),
                'api_key' => EnvConfig::get('SMS_API_KEY')
            ],
            
            // Blockchain Explorer APIs
            'blockchain_explorers' => [
                'etherscan' => [
                    'api_key' => EnvConfig::get('ETHERSCAN_API_KEY'),
                    'base_url' => 'https://api.etherscan.io/api'
                ],
                'bscscan' => [
                    'api_key' => EnvConfig::get('BSCSCAN_API_KEY'),
                    'base_url' => 'https://api.bscscan.com/api'
                ],
                'polygonscan' => [
                    'api_key' => EnvConfig::get('POLYGONSCAN_API_KEY'),
                    'base_url' => 'https://api.polygonscan.com/api'
                ],
                'basescan' => [
                    'api_key' => EnvConfig::get('BASESCAN_API_KEY'),
                    'base_url' => 'https://api.basescan.org/api'
                ],
                'solscan' => [
                    'api_key' => EnvConfig::get('SOLSCAN_API_KEY'),
                    'base_url' => 'https://api.solscan.io'
                ]
            ]
        ];
    }
    
    // ========================================
    // DATABASE CONFIGURATION
    // ========================================
    
    /**
     * Production Database Settings
     * Using environment configuration
     */
    public static function getDatabaseConfig() {
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        return [
            'type' => EnvConfig::get('DB_TYPE', 'sqlite'),
            'path' => EnvConfig::get('DB_PATH', 'database/crypto_exchange.db'),
            'host' => EnvConfig::get('DB_HOST', 'localhost'),
            'port' => EnvConfig::getInt('DB_PORT', 3306),
            'database' => EnvConfig::get('DB_DATABASE', 'yellow_crypto_production'),
            'username' => EnvConfig::get('DB_USERNAME', 'yellow_crypto_user'),
            'password' => EnvConfig::get('DB_PASSWORD', ''),
            'charset' => EnvConfig::get('DB_CHARSET', 'utf8mb4'),
            'timeout' => EnvConfig::getInt('DB_TIMEOUT', 30),
            'options' => [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::ATTR_TIMEOUT => EnvConfig::getInt('DB_TIMEOUT', 30)
            ]
        ];
    }
    
    // ========================================
    // ENCRYPTION SETTINGS
    // ========================================
    
    /**
     * Encryption Configuration
     */
    public static function getEncryptionConfig() {
        return [
            'algorithm' => 'AES-256-GCM',
            'key' => 'YOUR_32_BYTE_ENCRYPTION_KEY', // Generate with: openssl_random_pseudo_bytes(32)
            'iv_length' => 12,
            'tag_length' => 16
        ];
    }
    
    // ========================================
    // BACKUP CONFIGURATION
    // ========================================
    
    /**
     * Backup Settings
     */
    public static function getBackupConfig() {
        return [
            'database_backup' => [
                'enabled' => true,
                'frequency' => 'daily',
                'retention_days' => 30,
                'backup_location' => '/secure/backups/database/',
                'encrypt_backups' => true
            ],
            
            'wallet_backup' => [
                'enabled' => true,
                'frequency' => 'daily',
                'backup_location' => '/secure/backups/wallets/',
                'encrypt_backups' => true
            ]
        ];
    }
    
    // ========================================
    // UTILITY METHODS
    // ========================================
    
    /**
     * Get configuration by key
     */
    public static function get($key, $default = null) {
        $config = [
            'blockchain' => self::getBlockchainConfig(),
            'tokens' => self::getTokenContracts(),
            'wallets' => self::getSystemWallets(),
            'security' => self::getSecurityConfig(),
            'apis' => self::getExternalAPIs(),
            'database' => self::getDatabaseConfig(),
            'encryption' => self::getEncryptionConfig(),
            'backup' => self::getBackupConfig()
        ];
        
        return self::arrayGet($config, $key, $default);
    }
    
    /**
     * Array get helper
     */
    private static function arrayGet($array, $key, $default = null) {
        if (is_null($key)) {
            return $array;
        }
        
        if (isset($array[$key])) {
            return $array[$key];
        }
        
        foreach (explode('.', $key) as $segment) {
            if (!is_array($array) || !array_key_exists($segment, $array)) {
                return $default;
            }
            
            $array = $array[$segment];
        }
        
        return $array;
    }
    
    /**
     * Validate production configuration
     * Using environment configuration
     */
    public static function validateConfig() {
        $errors = [];
        
        // Load environment configuration
        require_once __DIR__ . '/env.php';
        EnvConfig::load(__DIR__ . '/../config.env');
        
        // Check blockchain endpoints
        $blockchain = self::getBlockchainConfig();
        foreach ($blockchain as $network => $config) {
            if (empty($config['rpc_url']) || strpos($config['rpc_url'], 'YOUR_') !== false) {
                $errors[] = "Blockchain RPC URL not configured for $network";
            }
        }
        
        // Check critical API keys
        $required_keys = [
            'INFURA_API_KEY' => 'Infura API Key',
            'ETHERSCAN_API_KEY' => 'Etherscan API Key',
            'BSCSCAN_API_KEY' => 'BSCScan API Key',
            'POLYGONSCAN_API_KEY' => 'PolygonScan API Key',
            'TRONGRID_API_KEY' => 'TronGrid API Key',
            'SOLSCAN_API_KEY' => 'Solscan API Key',
            'BITMUK_API_KEY' => 'Bitmuk API Key',
            'JWT_SECRET' => 'JWT Secret Key'
        ];
        
        foreach ($required_keys as $key => $description) {
            $value = EnvConfig::get($key);
            if (empty($value) || strpos($value, 'your_') !== false || strpos($value, 'YOUR_') !== false) {
                $errors[] = "$description not configured ($key)";
            }
        }
        
        // Check system wallets (still need manual configuration)
        $wallets = self::getSystemWallets();
        foreach ($wallets as $currency => $wallet) {
            if (strpos($wallet['address'], 'YOUR_') !== false) {
                $errors[] = "System wallet address not configured for $currency";
            }
            if (strpos($wallet['private_key'], 'YOUR_') !== false) {
                $errors[] = "System wallet private key not configured for $currency";
            }
        }
        
        // Check optional but recommended configurations
        $optional_keys = [
            'EXCHANGERATE_API_KEY' => 'Exchange Rate API Key',
            'SMTP_USERNAME' => 'SMTP Username',
            'SMS_API_KEY' => 'SMS API Key'
        ];
        
        foreach ($optional_keys as $key => $description) {
            $value = EnvConfig::get($key);
            if (empty($value) || strpos($value, 'your_') !== false || strpos($value, 'YOUR_') !== false) {
                $errors[] = "Warning: $description not configured ($key) - optional but recommended";
            }
        }
        
        return $errors;
    }
}

// ========================================
// PRODUCTION SETUP INSTRUCTIONS
// ========================================

/*
SETUP INSTRUCTIONS:

1. REPLACE ALL PLACEHOLDER VALUES:
   - Replace 'YOUR_INFURA_PROJECT_ID' with your actual Infura project ID
   - Replace 'YOUR_SYSTEM_*_ADDRESS' with your actual system wallet addresses
   - Replace 'YOUR_SYSTEM_*_PRIVATE_KEY' with your actual private keys
   - Replace 'YOUR_*_API_KEY' with your actual API keys

2. SECURE YOUR PRIVATE KEYS:
   - Use hardware wallets for large amounts
   - Store private keys in encrypted vaults
   - Use multi-signature wallets for extra security
   - Never store private keys in plain text

3. FUND YOUR SYSTEM WALLETS:
   - Transfer real crypto to your system wallet addresses
   - Ensure adequate liquidity for trading
   - Monitor balances regularly

4. TEST ON TESTNET FIRST:
   - Test all functionality on testnets
   - Use small amounts for initial testing
   - Verify all transactions work correctly

5. MONITOR AND MAINTAIN:
   - Set up monitoring and alerts
   - Regular security audits
   - Keep software updated
   - Backup regularly

SECURITY CHECKLIST:
✅ Private keys encrypted and secured
✅ Multi-signature protection enabled
✅ Fund insurance activated
✅ Real-time monitoring setup
✅ Backup systems configured
✅ Security audits completed
✅ Testnet testing completed
✅ Production deployment verified
*/
?>
