#!/usr/bin/env php
<?php
/**
 * Cron Job: Automated Deposit Detection
 * 
 * This script should be run every 1-5 minutes to check for new deposits
 * 
 * Crontab example (run every 2 minutes):
 * star-slash-2 star star star star /usr/bin/php /path/to/YELLOW/cron_check_deposits.php
 * 
 * Or for more frequent checks (every minute):
 * star star star star star /usr/bin/php /path/to/YELLOW/cron_check_deposits.php
 */

// Ensure script is run from command line only
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from the command line.\n");
}

// Set timezone
date_default_timezone_set('UTC');

// Start time
$startTime = microtime(true);
$timestamp = date('Y-m-d H:i:s');

echo "\n";
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║        YELLOW CRYPTO EXCHANGE - DEPOSIT MONITOR           ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";
echo "🕐 Started: {$timestamp}\n";
echo "────────────────────────────────────────────────────────────\n\n";

try {
    // Load deposit monitor
    require_once __DIR__ . '/api/deposit_monitor.php';
    
    echo "✓ DepositMonitor class loaded\n";
    
    $monitor = new DepositMonitor();
    
    echo "✓ DepositMonitor instantiated\n";
    echo "🔍 Scanning blockchain for deposits...\n\n";
    
    // Check all deposits
    $results = $monitor->checkAllDeposits();
    
    echo "✓ Scan completed\n";
    
    // Display results
    echo "────────────────────────────────────────────────────────────\n";
    echo "📊 SCAN RESULTS:\n";
    echo "────────────────────────────────────────────────────────────\n";
    echo "✓ Wallets Checked:     {$results['checked']}\n";
    echo "💰 Deposits Found:      {$results['deposits_found']}\n";
    echo "✅ Deposits Credited:   {$results['deposits_credited']}\n";
    
    if (count($results['deposits']) > 0) {
        echo "\n💸 DEPOSITS PROCESSED:\n";
        echo "────────────────────────────────────────────────────────────\n";
        foreach ($results['deposits'] as $deposit) {
            echo "  User #{$deposit['user_id']}:\n";
            echo "    Amount:   {$deposit['amount']} {$deposit['currency']}\n";
            echo "    Network:  {$deposit['network']}\n";
            echo "    Address:  {$deposit['address']}\n";
            echo "    Balance:  {$deposit['new_balance']} {$deposit['currency']}\n";
            echo "\n";
        }
    }
    
    if (count($results['errors']) > 0) {
        echo "\n❌ ERRORS ENCOUNTERED:\n";
        echo "────────────────────────────────────────────────────────────\n";
        foreach ($results['errors'] as $error) {
            echo "  • {$error}\n";
        }
    }
    
    // Calculate duration
    $duration = round(microtime(true) - $startTime, 2);
    $endTime = date('Y-m-d H:i:s');
    
    echo "\n────────────────────────────────────────────────────────────\n";
    echo "⏱️  Duration: {$duration}s\n";
    echo "🕐 Completed: {$endTime}\n";
    echo "────────────────────────────────────────────────────────────\n";
    
    // Exit with appropriate code
    if ($results['success']) {
        echo "✅ Deposit scan completed successfully!\n\n";
        exit(0);
    } else {
        echo "⚠️  Deposit scan completed with errors.\n\n";
        exit(1);
    }
    
} catch (Exception $e) {
    echo "\n";
    echo "╔════════════════════════════════════════════════════════════╗\n";
    echo "║                      FATAL ERROR                           ║\n";
    echo "╚════════════════════════════════════════════════════════════╝\n";
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "📍 File: " . $e->getFile() . ":" . $e->getLine() . "\n";
    echo "\n";
    
    exit(2);
}
?>
