#!/usr/bin/env php
<?php
/**
 * Cron Job Script for YELLOW Crypto Exchange
 * Handles automated tasks like price updates, backups, and monitoring
 */

// Set timezone
date_default_timezone_set('Africa/Kampala');

// Include required files
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/api/price_feed_service.php';
require_once __DIR__ . '/api/backup_monitoring_service.php';
require_once __DIR__ . '/api/email_service.php';
require_once __DIR__ . '/api/sms_service.php';

class CronJob {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function run() {
        echo "🕐 Starting cron job at " . date('Y-m-d H:i:s') . "\n";
        
        try {
            // Run different tasks based on time
            $hour = (int)date('H');
            $minute = (int)date('i');
            
            // Every minute: Update prices
            $this->updatePrices();
            
            // Every 5 minutes: Check blockchain service
            if ($minute % 5 === 0) {
                $this->checkBlockchainService();
            }
            
            // Every hour: Create backup
            if ($minute === 0) {
                $this->createBackup();
            }
            
            // Every 6 hours: Clean old data
            if ($hour % 6 === 0 && $minute === 0) {
                $this->cleanOldData();
            }
            
            // Daily at midnight: Full maintenance
            if ($hour === 0 && $minute === 0) {
                $this->dailyMaintenance();
            }
            
            echo "✅ Cron job completed successfully\n";
            
        } catch (Exception $e) {
            echo "❌ Cron job failed: " . $e->getMessage() . "\n";
            MonitoringService::sendAlert('cron_job_failed', $e->getMessage(), 'critical');
        }
    }
    
    private function updatePrices() {
        try {
            echo "📈 Updating cryptocurrency prices...\n";
            
            $symbols = ['BTC', 'ETH', 'USDT', 'USDC', 'BNB', 'ADA', 'SOL', 'MATIC'];
            $prices = PriceFeedService::getPrices($symbols);
            
            if ($prices) {
                PriceFeedService::updateDatabasePrices($prices);
                MonitoringService::logMetric('price_update', count($prices), ['symbols' => implode(',', $symbols)]);
                echo "✅ Updated prices for " . count($prices) . " cryptocurrencies\n";
            } else {
                echo "⚠️ Failed to fetch prices from external APIs\n";
            }
            
        } catch (Exception $e) {
            echo "❌ Price update failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function checkBlockchainService() {
        try {
            echo "🔗 Checking blockchain service...\n";
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'http://localhost:3001/health');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                MonitoringService::logMetric('blockchain_service_status', 'healthy');
                echo "✅ Blockchain service is healthy\n";
            } else {
                MonitoringService::logMetric('blockchain_service_status', 'unhealthy');
                MonitoringService::sendAlert('blockchain_service_down', 'Blockchain service returned HTTP ' . $httpCode, 'critical');
                echo "❌ Blockchain service is unhealthy (HTTP {$httpCode})\n";
            }
            
        } catch (Exception $e) {
            MonitoringService::logMetric('blockchain_service_status', 'error');
            MonitoringService::sendAlert('blockchain_service_error', $e->getMessage(), 'critical');
            echo "❌ Blockchain service check failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function createBackup() {
        try {
            echo "💾 Creating database backup...\n";
            
            $backup_file = BackupService::createDatabaseBackup();
            if ($backup_file) {
                MonitoringService::logMetric('backup_created', 1, ['file' => basename($backup_file)]);
                echo "✅ Backup created: " . basename($backup_file) . "\n";
            } else {
                MonitoringService::sendAlert('backup_failed', 'Database backup creation failed', 'critical');
                echo "❌ Backup creation failed\n";
            }
            
        } catch (Exception $e) {
            MonitoringService::sendAlert('backup_error', $e->getMessage(), 'critical');
            echo "❌ Backup failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function cleanOldData() {
        try {
            echo "🧹 Cleaning old data...\n";
            
            $conn = $this->db->getConnection();
            
            // Clean old audit logs (older than 30 days)
            $stmt = $conn->prepare("DELETE FROM audit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
            $stmt->execute();
            $audit_deleted = $stmt->rowCount();
            
            // Clean old sessions (older than 7 days)
            $stmt = $conn->prepare("DELETE FROM user_sessions WHERE expires_at < NOW()");
            $stmt->execute();
            $sessions_deleted = $stmt->rowCount();
            
            // Clean old failed transactions (older than 7 days)
            $stmt = $conn->prepare("DELETE FROM transactions WHERE status = 'failed' AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
            $stmt->execute();
            $transactions_deleted = $stmt->rowCount();
            
            MonitoringService::logMetric('data_cleanup', $audit_deleted + $sessions_deleted + $transactions_deleted, [
                'audit_logs' => $audit_deleted,
                'sessions' => $sessions_deleted,
                'transactions' => $transactions_deleted
            ]);
            
            echo "✅ Cleaned {$audit_deleted} audit logs, {$sessions_deleted} sessions, {$transactions_deleted} transactions\n";
            
        } catch (Exception $e) {
            echo "❌ Data cleanup failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function dailyMaintenance() {
        try {
            echo "🔧 Running daily maintenance...\n";
            
            $conn = $this->db->getConnection();
            
            // Optimize database tables
            $tables = ['users', 'transactions', 'user_wallets', 'market_prices', 'audit_logs'];
            foreach ($tables as $table) {
                $stmt = $conn->prepare("OPTIMIZE TABLE {$table}");
                $stmt->execute();
            }
            
            // Update system statistics
            $this->updateSystemStatistics();
            
            // Send daily report
            $this->sendDailyReport();
            
            MonitoringService::logMetric('daily_maintenance', 1);
            echo "✅ Daily maintenance completed\n";
            
        } catch (Exception $e) {
            echo "❌ Daily maintenance failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function updateSystemStatistics() {
        try {
            $conn = $this->db->getConnection();
            
            // Get user count
            $stmt = $conn->query("SELECT COUNT(*) as count FROM users WHERE is_active = 1");
            $user_count = $stmt->fetch()['count'];
            
            // Get transaction count for today
            $stmt = $conn->query("SELECT COUNT(*) as count FROM transactions WHERE DATE(created_at) = CURDATE()");
            $transaction_count = $stmt->fetch()['count'];
            
            // Get total volume for today
            $stmt = $conn->query("SELECT SUM(amount) as volume FROM transactions WHERE DATE(created_at) = CURDATE() AND status = 'confirmed'");
            $daily_volume = $stmt->fetch()['volume'] ?? 0;
            
            // Log statistics
            MonitoringService::logMetric('daily_users', $user_count);
            MonitoringService::logMetric('daily_transactions', $transaction_count);
            MonitoringService::logMetric('daily_volume', $daily_volume);
            
            echo "📊 Updated statistics: {$user_count} users, {$transaction_count} transactions, {$daily_volume} volume\n";
            
        } catch (Exception $e) {
            echo "❌ Statistics update failed: " . $e->getMessage() . "\n";
        }
    }
    
    private function sendDailyReport() {
        try {
            $conn = $this->db->getConnection();
            
            // Get daily statistics
            $stmt = $conn->query("SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = CURDATE()");
            $new_users = $stmt->fetch()['count'];
            
            $stmt = $conn->query("SELECT COUNT(*) as count FROM transactions WHERE DATE(created_at) = CURDATE()");
            $transactions = $stmt->fetch()['count'];
            
            $stmt = $conn->query("SELECT SUM(amount) as volume FROM transactions WHERE DATE(created_at) = CURDATE() AND status = 'confirmed'");
            $volume = $stmt->fetch()['volume'] ?? 0;
            
            // Send email report to admin
            $admin_email = 'admin@yellowcrypto.com';
            $subject = "Daily Report - " . date('Y-m-d');
            $body = "
            <html>
            <body>
                <h2>Daily Report - " . date('Y-m-d') . "</h2>
                <p><strong>New Users:</strong> {$new_users}</p>
                <p><strong>Transactions:</strong> {$transactions}</p>
                <p><strong>Volume:</strong> {$volume}</p>
                <p><strong>System Status:</strong> Healthy</p>
                <p>Best regards,<br>YELLOW Crypto Exchange System</p>
            </body>
            </html>
            ";
            
            EmailService::sendEmail($admin_email, $subject, $body);
            echo "📧 Daily report sent\n";
            
        } catch (Exception $e) {
            echo "❌ Daily report failed: " . $e->getMessage() . "\n";
        }
    }
}

// Run the cron job
if (php_sapi_name() === 'cli') {
    $cron = new CronJob();
    $cron->run();
} else {
    echo "This script must be run from the command line\n";
    exit(1);
}
