#!/usr/bin/env php
<?php
/**
 * Cron Job: System Wallet Balance Synchronization
 * 
 * Automatically syncs system wallet balances from blockchain to database
 * Run every 5-10 minutes to keep system wallet balances accurate
 * 
 * Crontab example (run every 5 minutes):
 * star-slash-5 star star star star /usr/bin/php /path/to/YELLOW/cron_sync_system_wallets.php
 */

// Ensure script is run from command line only
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from the command line.\n");
}

// Set timezone
date_default_timezone_set('UTC');

// Start time
$startTime = microtime(true);
$timestamp = date('Y-m-d H:i:s');

echo "\n";
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║      YELLOW CRYPTO EXCHANGE - SYSTEM WALLET SYNC          ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";
echo "🕐 Started: {$timestamp}\n";
echo "────────────────────────────────────────────────────────────\n\n";

try {
    // Load system wallet sync
    require_once __DIR__ . '/api/system_wallet_sync.php';
    
    $sync = new SystemWalletSync();
    
    echo "🔄 Syncing system wallet balances from blockchain...\n\n";
    
    // Sync all system wallets
    $results = $sync->syncAllSystemWallets();
    
    // Display results
    echo "────────────────────────────────────────────────────────────\n";
    echo "📊 SYNC RESULTS:\n";
    echo "────────────────────────────────────────────────────────────\n";
    echo "✓ Wallets Checked:     {$results['synced']}\n";
    echo "🔄 Wallets Updated:     {$results['updated']}\n";
    
    if (count($results['wallets']) > 0) {
        echo "\n💰 UPDATED WALLETS:\n";
        echo "────────────────────────────────────────────────────────────\n";
        foreach ($results['wallets'] as $wallet) {
            if ($wallet['updated']) {
                $diff = $wallet['new_balance'] - $wallet['old_balance'];
                $symbol = $diff > 0 ? '+' : '';
                echo "  {$wallet['currency']} ({$wallet['network']}):\n";
                echo "    Address:  {$wallet['address']}\n";
                echo "    Old:      {$wallet['old_balance']}\n";
                echo "    New:      {$wallet['new_balance']}\n";
                echo "    Change:   {$symbol}{$diff}\n";
                echo "\n";
            }
        }
    }
    
    if (count($results['errors']) > 0) {
        echo "\n❌ ERRORS ENCOUNTERED:\n";
        echo "────────────────────────────────────────────────────────────\n";
        foreach ($results['errors'] as $error) {
            echo "  • {$error}\n";
        }
    }
    
    // Calculate duration
    $duration = round(microtime(true) - $startTime, 2);
    $endTime = date('Y-m-d H:i:s');
    
    echo "\n────────────────────────────────────────────────────────────\n";
    echo "⏱️  Duration: {$duration}s\n";
    echo "🕐 Completed: {$endTime}\n";
    echo "────────────────────────────────────────────────────────────\n";
    
    // Exit with appropriate code
    if ($results['success']) {
        echo "✅ System wallet sync completed successfully!\n\n";
        exit(0);
    } else {
        echo "⚠️  System wallet sync completed with errors.\n\n";
        exit(1);
    }
    
} catch (Exception $e) {
    echo "\n";
    echo "╔════════════════════════════════════════════════════════════╗\n";
    echo "║                      FATAL ERROR                           ║\n";
    echo "╚════════════════════════════════════════════════════════════╝\n";
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "📍 File: " . $e->getFile() . ":" . $e->getLine() . "\n";
    echo "\n";
    
    exit(2);
}
?>







