<?php
/**
 * Database Migration Script
 * Migrates SQLite database to production-ready schema
 */

require_once __DIR__ . '/../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "🔄 Starting database migration...\n";
    
    // Read the migration SQL file
    $migrationSQL = file_get_contents(__DIR__ . '/migrate_to_production.sql');
    
    // Split into individual statements
    $statements = array_filter(array_map('trim', explode(';', $migrationSQL)));
    
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($statements as $statement) {
        if (empty($statement) || strpos($statement, '--') === 0) {
            continue; // Skip empty statements and comments
        }
        
        try {
            $conn->exec($statement);
            $successCount++;
            echo "✅ Executed: " . substr($statement, 0, 50) . "...\n";
        } catch (PDOException $e) {
            // Handle specific errors gracefully
            if (strpos($e->getMessage(), 'duplicate column name') !== false || 
                strpos($e->getMessage(), 'no such column') !== false) {
                echo "⚠️ Skipped (column already exists or missing): " . substr($statement, 0, 50) . "...\n";
            } else {
                $errorCount++;
                echo "❌ Error executing statement: " . $e->getMessage() . "\n";
                echo "   Statement: " . substr($statement, 0, 100) . "...\n";
            }
        }
    }
    
    echo "\n📊 Migration Summary:\n";
    echo "✅ Successful statements: $successCount\n";
    echo "❌ Failed statements: $errorCount\n";
    
    if ($errorCount === 0) {
        echo "\n🎉 Database migration completed successfully!\n";
        
        // Verify critical tables exist
        $tables = ['users', 'transactions', 'payments', 'crypto_wallets', 'crypto_transactions', 'system_settings'];
        echo "\n🔍 Verifying tables:\n";
        
        foreach ($tables as $table) {
            try {
                $stmt = $conn->query("SELECT COUNT(*) FROM $table");
                $count = $stmt->fetchColumn();
                echo "✅ Table '$table': $count records\n";
            } catch (PDOException $e) {
                echo "❌ Table '$table': " . $e->getMessage() . "\n";
            }
        }
        
    } else {
        echo "\n⚠️ Migration completed with errors. Please review the errors above.\n";
    }
    
} catch (Exception $e) {
    echo "❌ Migration failed: " . $e->getMessage() . "\n";
    exit(1);
}
?>
