<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/api/security_validator.php';

echo "🔍 Testing API Crypto Buy Logic\n";
echo "===============================\n\n";

try {
    // Simulate the API class
    class TestAPI {
        private $db;
        private $user_id = 1;
        
        public function __construct() {
            $this->db = new Database();
        }
        
        public function testCryptoBuy() {
            $currency = 'USDT';
            $amount = 100;
            $payment_method = 'fiat';
            
            echo "Testing crypto buy with API logic...\n";
            echo "Currency: $currency, Amount: $amount, Payment: $payment_method\n";
            
            // Security validation
            require_once __DIR__ . '/api/security_validator.php';
            $validator = new SecurityValidator($this->db);
            $validation_result = $validator->validateCryptoBuy($this->user_id, $currency, $amount, $payment_method);
            
            if (!$validation_result['approved']) {
                echo "❌ Security validation failed\n";
                return;
            }
            echo "✅ Security validation passed\n";
            
            // Get current crypto price
            $stmt = $this->db->getConnection()->prepare("
                SELECT buy_price as price FROM crypto_prices 
                WHERE crypto = ? AND is_active = 1
                ORDER BY created_at DESC 
                LIMIT 1
            ");
            $stmt->execute([$currency]);
            $priceData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$priceData) {
                echo "❌ Price not available\n";
                return;
            }
            
            $currentPrice = floatval($priceData['price']);
            $cryptoAmount = $amount / $currentPrice;
            $fee = $cryptoAmount * 0.01;
            $totalCryptoAmount = $cryptoAmount - $fee;
            
            echo "✅ Price calculation: $currentPrice UGX per $currency\n";
            echo "   Crypto amount: $cryptoAmount, Fee: $fee, Total: $totalCryptoAmount\n";
            
            // Check user's fiat balance
            $stmt = $this->db->getConnection()->prepare("
                SELECT balance FROM user_fiat_balances 
                WHERE user_id = ? AND currency_code = 'UGX'
            ");
            $stmt->execute([$this->user_id]);
            $balanceData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$balanceData || $balanceData['balance'] < $amount) {
                echo "❌ Insufficient UGX balance\n";
                return;
            }
            echo "✅ User balance sufficient: " . $balanceData['balance'] . " UGX\n";
            
            // Check system wallet has enough crypto
            $stmt = $this->db->getConnection()->prepare("
                SELECT id, balance, address FROM crypto_wallets 
                WHERE user_id = 0 AND currency = ? AND (network = 'erc20' OR network = 'ERC20' OR network = 'bep20' OR network = 'BEP20')
                AND balance > 0
                ORDER BY balance DESC
                LIMIT 1
            ");
            $stmt->execute([$currency]);
            $systemWallet = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$systemWallet) {
                echo "❌ System wallet not available\n";
                return;
            }
            
            if ($systemWallet['balance'] < $totalCryptoAmount) {
                echo "❌ Insufficient system balance\n";
                return;
            }
            echo "✅ System wallet sufficient: " . $systemWallet['balance'] . " $currency\n";
            
            // Test BlockchainManager loading
            echo "\n🔗 Testing BlockchainManager...\n";
            try {
                require_once __DIR__ . '/api/blockchain_manager.php';
                $blockchain = new BlockchainManager($this->db);
                echo "✅ BlockchainManager created\n";
                
                // Test wallet generation
                $wallet_data = $blockchain->generateRealWallet($currency, 'erc20');
                if ($wallet_data && isset($wallet_data['address'])) {
                    echo "✅ Wallet generated: " . substr($wallet_data['address'], 0, 20) . "...\n";
                } else {
                    echo "❌ Wallet generation failed\n";
                    return;
                }
                
            } catch (Exception $e) {
                echo "❌ BlockchainManager error: " . $e->getMessage() . "\n";
                echo "Stack trace: " . $e->getTraceAsString() . "\n";
                return;
            }
            
            echo "\n🎉 All API checks passed!\n";
        }
    }
    
    $api = new TestAPI();
    $api->testCryptoBuy();
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>

