<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/api/security_validator.php';

echo "🔍 Debugging Crypto Buy Issue\n";
echo "=============================\n\n";

try {
    // Test database connection
    $db = new Database();
    $conn = $db->getConnection();
    echo "✅ Database connection successful\n";
    
    // Test user data
    $user_id = 1;
    $currency = 'USDT';
    $amount = 100;
    
    echo "\n📊 Testing with:\n";
    echo "   User ID: $user_id\n";
    echo "   Currency: $currency\n";
    echo "   Amount: $amount UGX\n";
    
    // Check user balance
    $stmt = $conn->prepare("
        SELECT balance FROM user_fiat_balances 
        WHERE user_id = ? AND currency_code = 'UGX'
    ");
    $stmt->execute([$user_id]);
    $balanceData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($balanceData) {
        echo "✅ User UGX balance: " . $balanceData['balance'] . "\n";
        if ($balanceData['balance'] < $amount) {
            echo "❌ Insufficient balance\n";
            exit;
        }
    } else {
        echo "❌ No UGX balance found\n";
        exit;
    }
    
    // Check crypto price
    $stmt = $conn->prepare("
        SELECT buy_price as price FROM crypto_prices 
        WHERE crypto = ? AND is_active = 1
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$currency]);
    $priceData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($priceData) {
        $currentPrice = floatval($priceData['price']);
        echo "✅ Current price: $currentPrice UGX per $currency\n";
        
        $cryptoAmount = $amount / $currentPrice;
        $fee = $cryptoAmount * 0.01;
        $totalCryptoAmount = $cryptoAmount - $fee;
        
        echo "   Crypto amount: $cryptoAmount\n";
        echo "   Fee: $fee\n";
        echo "   Total crypto: $totalCryptoAmount\n";
    } else {
        echo "❌ No price found for $currency\n";
        exit;
    }
    
    // Check system wallet
    $stmt = $conn->prepare("
        SELECT id, balance, address FROM crypto_wallets 
        WHERE user_id = 0 AND currency = ? AND (network = 'erc20' OR network = 'ERC20' OR network = 'bep20' OR network = 'BEP20')
        AND balance > 0
        ORDER BY balance DESC
        LIMIT 1
    ");
    $stmt->execute([$currency]);
    $systemWallet = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($systemWallet) {
        echo "✅ System wallet found:\n";
        echo "   ID: " . $systemWallet['id'] . "\n";
        echo "   Balance: " . $systemWallet['balance'] . " $currency\n";
        echo "   Address: " . $systemWallet['address'] . "\n";
        
        if ($systemWallet['balance'] < $totalCryptoAmount) {
            echo "❌ Insufficient system balance\n";
            exit;
        }
    } else {
        echo "❌ No system wallet found with balance > 0\n";
        exit;
    }
    
    // Test security validator
    echo "\n🔒 Testing Security Validator...\n";
    $validator = new SecurityValidator($db);
    $validation_result = $validator->validateCryptoBuy($user_id, $currency, $amount, 'fiat');
    
    if ($validation_result['approved']) {
        echo "✅ Security validation passed\n";
    } else {
        echo "❌ Security validation failed:\n";
        echo "   Errors: " . json_encode($validation_result['errors'] ?? []) . "\n";
        echo "   Warnings: " . json_encode($validation_result['warnings'] ?? []) . "\n";
        exit;
    }
    
    echo "\n🎉 All checks passed! The issue might be in the transaction logic.\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>

