<?php
require_once __DIR__ . '/config/database.php';

echo "🔍 Testing Simple Crypto Buy Logic\n";
echo "==================================\n\n";

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $user_id = 1;
    $currency = 'USDT';
    $amount = 100;
    $currentPrice = 3750;
    $cryptoAmount = $amount / $currentPrice;
    $fee = $cryptoAmount * 0.01;
    $totalCryptoAmount = $cryptoAmount - $fee;
    
    echo "Testing crypto buy without BlockchainManager...\n";
    echo "Amount: $amount UGX\n";
    echo "Crypto amount: $totalCryptoAmount USDT\n";
    
    // Start transaction
    $conn->beginTransaction();
    echo "✅ Transaction started\n";
    
    try {
        // Deduct fiat balance
        $stmt = $conn->prepare("
            UPDATE user_fiat_balances 
            SET balance = balance - ? 
            WHERE user_id = ? AND currency_code = 'UGX'
        ");
        $stmt->execute([$amount, $user_id]);
        echo "✅ Fiat balance deducted\n";
        
        // Deduct from system wallet
        $stmt = $conn->prepare("
            UPDATE crypto_wallets 
            SET balance = balance - ?, updated_at = CURRENT_TIMESTAMP
            WHERE id = 3
        ");
        $stmt->execute([$totalCryptoAmount]);
        echo "✅ System wallet deducted\n";
        
        // Add to user wallet (create if doesn't exist)
        $stmt = $conn->prepare("
            SELECT id, balance, address FROM crypto_wallets 
            WHERE user_id = ? AND currency = ? AND (network = 'ERC20' OR network = 'erc20')
        ");
        $stmt->execute([$user_id, $currency]);
        $userWallet = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$userWallet) {
            // Create new wallet
            $address = '0x' . bin2hex(random_bytes(20));
            $stmt = $conn->prepare("
                INSERT INTO crypto_wallets (user_id, network, currency, address, balance, created_at, updated_at)
                VALUES (?, 'erc20', ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$user_id, $currency, $address, $totalCryptoAmount]);
            $userWalletId = $conn->lastInsertId();
            echo "✅ New user wallet created\n";
        } else {
            // Update existing wallet
            $stmt = $conn->prepare("
                UPDATE crypto_wallets 
                SET balance = balance + ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$totalCryptoAmount, $userWallet['id']]);
            $userWalletId = $userWallet['id'];
            $address = $userWallet['address'];
            echo "✅ Existing user wallet updated\n";
        }
        
        // Record transaction
        $txHash = 'buy_' . time() . '_' . $user_id . '_' . bin2hex(random_bytes(16));
        $stmt = $conn->prepare("
            INSERT INTO crypto_transactions (wallet_id, user_id, network, currency, address, tx_hash, amount, type, status, created_at)
            VALUES (?, ?, 'erc20', ?, ?, ?, ?, 'buy', 'confirmed', CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$userWalletId, $user_id, $currency, $address, $txHash, $totalCryptoAmount]);
        echo "✅ Transaction recorded\n";
        
        // Commit
        $conn->commit();
        echo "✅ Transaction committed successfully\n";
        
        echo "\n🎉 SUCCESS! Crypto buy completed\n";
        echo "   User received: $totalCryptoAmount $currency\n";
        echo "   Transaction hash: " . substr($txHash, 0, 30) . "...\n";
        
    } catch (Exception $e) {
        $conn->rollback();
        echo "❌ Transaction rolled back: " . $e->getMessage() . "\n";
        throw $e;
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>

