<?php
/**
 * Production Deployment Script
 * 
 * This script helps you deploy your crypto exchange to production
 * with real blockchain integration and proper security measures.
 */

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/production_config.php';
require_once __DIR__ . '/api/blockchain_manager.php';

class ProductionDeployment {
    private $db;
    private $blockchain;
    private $config;
    
    public function __construct() {
        $this->config = ProductionConfig::class;
        echo "🚀 YELLOW Crypto Exchange - Production Deployment\n";
        echo "==================================================\n\n";
    }
    
    /**
     * Main deployment process
     */
    public function deploy() {
        try {
            // Step 1: Validate configuration
            echo "📋 Step 1: Validating Production Configuration...\n";
            $this->validateConfiguration();
            
            // Step 2: Setup database
            echo "\n💾 Step 2: Setting up Production Database...\n";
            $this->setupDatabase();
            
            // Step 3: Initialize system wallets
            echo "\n🔐 Step 3: Initializing System Wallets...\n";
            $this->initializeSystemWallets();
            
            // Step 4: Fund system wallets
            echo "\n💰 Step 4: Funding System Wallets...\n";
            $this->fundSystemWallets();
            
            // Step 5: Setup monitoring
            echo "\n📊 Step 5: Setting up Monitoring...\n";
            $this->setupMonitoring();
            
            // Step 6: Security verification
            echo "\n🛡️ Step 6: Security Verification...\n";
            $this->verifySecurity();
            
            // Step 7: Final checks
            echo "\n✅ Step 7: Final Production Checks...\n";
            $this->finalChecks();
            
            echo "\n🎉 Production deployment completed successfully!\n";
            echo "Your crypto exchange is now ready for real trading!\n";
            
        } catch (Exception $e) {
            echo "\n❌ Deployment failed: " . $e->getMessage() . "\n";
            echo "Please fix the issues and try again.\n";
            exit(1);
        }
    }
    
    /**
     * Validate production configuration
     */
    private function validateConfiguration() {
        $errors = ProductionConfig::validateConfig();
        
        if (!empty($errors)) {
            $critical_errors = [];
            $warnings = [];
            
            // Separate critical errors from warnings
            foreach ($errors as $error) {
                if (strpos($error, 'Warning:') === 0) {
                    $warnings[] = $error;
                } else {
                    $critical_errors[] = $error;
                }
            }
            
            // Show warnings
            if (!empty($warnings)) {
                echo "⚠️ Configuration warnings found:\n";
                foreach ($warnings as $warning) {
                    echo "   - $warning\n";
                }
                echo "\n";
            }
            
            // Block deployment only on critical errors
            if (!empty($critical_errors)) {
                echo "❌ Critical configuration errors found:\n";
                foreach ($critical_errors as $error) {
                    echo "   - $error\n";
                }
                echo "\nPlease update config/production_config.php with your actual values.\n";
                throw new Exception("Configuration validation failed");
            }
            
            echo "✅ Critical configuration validated successfully\n";
            if (!empty($warnings)) {
                echo "⚠️ Proceeding with warnings - some features may be limited\n";
            }
        } else {
            echo "✅ Configuration validated successfully\n";
        }
    }
    
    /**
     * Setup production database
     */
    private function setupDatabase() {
        try {
            // Initialize database with production settings
            $this->db = new Database();
            
            // Create production tables
            $this->createProductionTables();
            
            // Setup initial data
            $this->setupInitialData();
            
            echo "✅ Database setup completed\n";
            
        } catch (Exception $e) {
            echo "❌ Database setup failed: " . $e->getMessage() . "\n";
            throw $e;
        }
    }
    
    /**
     * Create production-specific tables
     */
    private function createProductionTables() {
        $conn = $this->db->getConnection();
        
        // Create production monitoring table
        $conn->exec("
            CREATE TABLE IF NOT EXISTS production_monitoring (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                metric_name TEXT NOT NULL,
                metric_value REAL NOT NULL,
                threshold REAL,
                status TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // Create production alerts table
        $conn->exec("
            CREATE TABLE IF NOT EXISTS production_alerts (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                alert_type TEXT NOT NULL,
                severity TEXT NOT NULL,
                message TEXT NOT NULL,
                resolved INTEGER DEFAULT 0,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                resolved_at DATETIME
            )
        ");
        
        // Create production backup log
        $conn->exec("
            CREATE TABLE IF NOT EXISTS backup_logs (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                backup_type TEXT NOT NULL,
                file_path TEXT NOT NULL,
                file_size INTEGER,
                status TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        echo "   - Production tables created\n";
    }
    
    /**
     * Setup initial production data
     */
    private function setupInitialData() {
        $conn = $this->db->getConnection();
        
        // Insert initial monitoring metrics
        $metrics = [
            ['system_health', 100, 90, 'healthy'],
            ['blockchain_sync', 100, 95, 'healthy'],
            ['wallet_balances', 100, 80, 'healthy'],
            ['transaction_volume', 0, 1000000, 'healthy'],
            ['error_rate', 0, 5, 'healthy']
        ];
        
        $stmt = $conn->prepare("
            INSERT INTO production_monitoring (metric_name, metric_value, threshold, status)
            VALUES (?, ?, ?, ?)
        ");
        
        foreach ($metrics as $metric) {
            $stmt->execute($metric);
        }
        
        echo "   - Initial monitoring data inserted\n";
    }
    
    /**
     * Initialize system wallets with real blockchain integration
     */
    private function initializeSystemWallets() {
        try {
            $this->blockchain = new BlockchainManager($this->db);
            
            // Initialize real blockchain system wallets
            $result = $this->blockchain->initializeSystemWallets();
            
            if ($result['success']) {
                echo "✅ System wallets initialized successfully\n";
                
                // Display wallet information
                $this->displaySystemWallets();
            } else {
                echo "❌ System wallet initialization failed: " . $result['error'] . "\n";
                throw new Exception("System wallet initialization failed");
            }
            
        } catch (Exception $e) {
            echo "❌ System wallet initialization failed: " . $e->getMessage() . "\n";
            throw $e;
        }
    }
    
    /**
     * Display system wallets information
     */
    private function displaySystemWallets() {
        $conn = $this->db->getConnection();
        
        $stmt = $conn->prepare("
            SELECT currency, network, address, balance, is_real_blockchain
            FROM crypto_wallets 
            WHERE user_id = 0
            ORDER BY currency, network
        ");
        $stmt->execute();
        $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "\n📋 System Wallets:\n";
        echo "==================\n";
        
        foreach ($wallets as $wallet) {
            $status = $wallet['is_real_blockchain'] ? '✅ Real Blockchain' : '⚠️ Mock/Simulated';
            echo "   {$wallet['currency']} ({$wallet['network']}):\n";
            echo "     Address: {$wallet['address']}\n";
            echo "     Balance: {$wallet['balance']} {$wallet['currency']}\n";
            echo "     Status: $status\n\n";
        }
    }
    
    /**
     * Fund system wallets with real crypto
     */
    private function fundSystemWallets() {
        echo "⚠️  IMPORTANT: You need to manually fund your system wallets!\n";
        echo "\n📋 System Wallet Funding Instructions:\n";
        echo "=====================================\n";
        
        $wallets = ProductionConfig::getSystemWallets();
        
        foreach ($wallets as $currency => $wallet) {
            echo "\n{$currency} Wallet:\n";
            echo "   Address: {$wallet['address']}\n";
            echo "   Required Balance: {$wallet['initial_balance']} {$currency}\n";
            echo "   Minimum Balance: {$wallet['min_balance']} {$currency}\n";
            echo "   Network: {$wallet['network']}\n";
        }
        
        echo "\n💡 Funding Steps:\n";
        echo "1. Transfer the required crypto amounts to each wallet address\n";
        echo "2. Wait for blockchain confirmations\n";
        echo "3. Run: php sync_blockchain_balances.php\n";
        echo "4. Verify balances in admin dashboard\n";
        
        echo "\n⏳ Press Enter when you've funded the wallets...";
        readline();
    }
    
    /**
     * Setup production monitoring
     */
    private function setupMonitoring() {
        echo "📊 Setting up production monitoring...\n";
        
        // Create monitoring scripts
        $this->createMonitoringScripts();
        
        // Setup alert system
        $this->setupAlertSystem();
        
        echo "✅ Monitoring setup completed\n";
    }
    
    /**
     * Create monitoring scripts
     */
    private function createMonitoringScripts() {
        // Create blockchain monitoring script
        $monitoring_script = '<?php
require_once __DIR__ . "/config/database.php";
require_once __DIR__ . "/api/blockchain_manager.php";

$db = new Database();
$blockchain = new BlockchainManager($db);

// Sync blockchain balances
$result = $blockchain->syncBlockchainBalances();
echo "Blockchain sync: " . ($result["success"] ? "Success" : "Failed") . "\n";

// Check system health
$conn = $db->getConnection();
$stmt = $conn->prepare("SELECT COUNT(*) as count FROM crypto_wallets WHERE user_id = 0");
$stmt->execute();
$system_wallets = $stmt->fetch()["count"];

echo "System wallets: $system_wallets\n";
echo "Monitoring completed at " . date("Y-m-d H:i:s") . "\n";
?>';
        
        file_put_contents('monitor_blockchain.php', $monitoring_script);
        echo "   - Blockchain monitoring script created\n";
        
        // Create backup script
        $backup_script = '<?php
require_once __DIR__ . "/config/database.php";

$db = new Database();
$backup_dir = __DIR__ . "/backups/" . date("Y-m-d");
if (!is_dir($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

$backup_file = $backup_dir . "/database_" . date("H-i-s") . ".sqlite";
copy(__DIR__ . "/database/crypto_exchange.db", $backup_file);

echo "Database backup created: $backup_file\n";
?>';
        
        file_put_contents('backup_database.php', $backup_script);
        echo "   - Database backup script created\n";
    }
    
    /**
     * Setup alert system
     */
    private function setupAlertSystem() {
        $alert_script = '<?php
require_once __DIR__ . "/config/database.php";

function sendAlert($type, $severity, $message) {
    $conn = (new Database())->getConnection();
    $stmt = $conn->prepare("
        INSERT INTO production_alerts (alert_type, severity, message)
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$type, $severity, $message]);
    
    // In production, also send email/SMS alerts
    echo "ALERT [$severity]: $message\n";
}

// Example alert
sendAlert("system_health", "warning", "System wallet balance below threshold");
?>';
        
        file_put_contents('send_alert.php', $alert_script);
        echo "   - Alert system created\n";
    }
    
    /**
     * Verify security measures
     */
    private function verifySecurity() {
        echo "🛡️ Verifying security measures...\n";
        
        // Check if security validator exists
        if (file_exists(__DIR__ . '/api/security_validator.php')) {
            echo "✅ Security validator found\n";
        } else {
            echo "❌ Security validator missing\n";
        }
        
        // Check if multisig system exists
        if (file_exists(__DIR__ . '/api/multisig_wallet.php')) {
            echo "✅ Multi-signature system found\n";
        } else {
            echo "❌ Multi-signature system missing\n";
        }
        
        // Check if fund protection exists
        if (file_exists(__DIR__ . '/api/fund_protection.php')) {
            echo "✅ Fund protection system found\n";
        } else {
            echo "❌ Fund protection system missing\n";
        }
        
        // Check encryption configuration
        $encryption = ProductionConfig::get('encryption');
        if ($encryption['algorithm'] === 'AES-256-GCM') {
            echo "✅ Strong encryption configured\n";
        } else {
            echo "❌ Weak encryption configuration\n";
        }
        
        echo "✅ Security verification completed\n";
    }
    
    /**
     * Final production checks
     */
    private function finalChecks() {
        echo "🔍 Running final production checks...\n";
        
        // Test database connection
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("SELECT 1");
            $stmt->execute();
            echo "✅ Database connection working\n";
        } catch (Exception $e) {
            echo "❌ Database connection failed\n";
        }
        
        // Test blockchain manager
        try {
            if ($this->blockchain) {
                echo "✅ Blockchain manager initialized\n";
            }
        } catch (Exception $e) {
            echo "❌ Blockchain manager failed\n";
        }
        
        // Check file permissions
        $required_files = [
            'api/index.php',
            'config/database.php',
            'config/production_config.php',
            'api/blockchain_manager.php'
        ];
        
        foreach ($required_files as $file) {
            if (file_exists($file)) {
                echo "✅ $file exists\n";
            } else {
                echo "❌ $file missing\n";
            }
        }
        
        echo "✅ Final checks completed\n";
    }
}

// ========================================
// DEPLOYMENT EXECUTION
// ========================================

if (php_sapi_name() === 'cli') {
    echo "🚀 Starting Production Deployment...\n\n";
    
    $deployment = new ProductionDeployment();
    $deployment->deploy();
    
    echo "\n🎯 NEXT STEPS:\n";
    echo "==============\n";
    echo "1. Fund your system wallets with real crypto\n";
    echo "2. Configure your server (nginx/apache, SSL)\n";
    echo "3. Set up cron jobs for monitoring\n";
    echo "4. Test all functionality thoroughly\n";
    echo "5. Launch to public!\n";
    
} else {
    echo "This script must be run from the command line.\n";
    echo "Usage: php deploy_production.php\n";
}
?>
