/**
 * Balance Manager
 * Centralized balance management for the crypto exchange
 */

class BalanceManager {
    static instance = null;
    static listeners = [];
    static portfolio = null;
    static userData = null;
    static lastUpdate = null;
    static updateInterval = null;

    static initialize() {
        if (this.instance) {
            return Promise.resolve(true);
        }

        this.instance = new BalanceManager();
        return this.loadBalanceData();
    }

    static async loadBalanceData() {
        try {
            console.log('🔄 Loading balance data...');
            
            // Load user profile
            const profileResponse = await APIHelper.get(API_CONFIG.ENDPOINTS.USER_PROFILE);
            if (profileResponse.success) {
                this.userData = profileResponse.user;
                console.log('✅ User profile loaded:', this.userData);
            }

            // Load crypto wallets
            const walletsResponse = await APIHelper.get(API_CONFIG.ENDPOINTS.CRYPTO_WALLETS);
            if (walletsResponse.success) {
                this.portfolio = walletsResponse.wallets;
                console.log('✅ Crypto wallets loaded:', this.portfolio);
            }

            // Load fiat balances
            const fiatResponse = await APIHelper.get('/user/fiat-balances');
            if (fiatResponse.success) {
                this.userData.fiat_balances = fiatResponse.balances;
                console.log('✅ Fiat balances loaded:', this.userData.fiat_balances);
            }

            this.lastUpdate = new Date();
            this.notifyListeners();
            
            // Start auto-refresh
            this.startAutoRefresh();
            
            return true;
        } catch (error) {
            console.error('❌ Failed to load balance data:', error);
            return false;
        }
    }

    static startAutoRefresh() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }

        // Refresh every 30 seconds
        this.updateInterval = setInterval(() => {
            this.refreshBalances();
        }, 30000);
    }

    static async refreshBalances() {
        try {
            console.log('🔄 Refreshing balances...');
            
            // Only refresh crypto wallets (faster)
            const walletsResponse = await APIHelper.get(API_CONFIG.ENDPOINTS.CRYPTO_WALLETS);
            if (walletsResponse.success) {
                this.portfolio = walletsResponse.wallets;
                this.lastUpdate = new Date();
                this.notifyListeners();
                console.log('✅ Balances refreshed');
            }
        } catch (error) {
            console.error('❌ Failed to refresh balances:', error);
        }
    }

    static addListener(callback) {
        this.listeners.push(callback);
    }

    static removeListener(callback) {
        const index = this.listeners.indexOf(callback);
        if (index > -1) {
            this.listeners.splice(index, 1);
        }
    }

    static notifyListeners() {
        this.listeners.forEach(callback => {
            try {
                callback(this.portfolio, this.userData);
            } catch (error) {
                console.error('Listener error:', error);
            }
        });
    }

    static getPortfolio() {
        return this.portfolio || [];
    }

    static getUserData() {
        return this.userData || {};
    }

    static getTotalPortfolioValue() {
        if (!this.portfolio) return 0;
        
        let total = 0;
        this.portfolio.forEach(wallet => {
            if (wallet.balance && wallet.usd_value) {
                total += parseFloat(wallet.usd_value);
            }
        });
        
        return total;
    }

    static getBalanceByCurrency(currency) {
        if (!this.portfolio) return 0;
        
        const wallet = this.portfolio.find(w => w.currency === currency);
        return wallet ? parseFloat(wallet.balance || 0) : 0;
    }

    static getFiatBalance(currency) {
        if (!this.userData || !this.userData.fiat_balances) return 0;
        
        const balance = this.userData.fiat_balances.find(b => b.currency === currency);
        return balance ? parseFloat(balance.balance || 0) : 0;
    }

    static updatePageBalance() {
        // Update portfolio value
        const totalValue = this.getTotalPortfolioValue();
        const totalElement = document.getElementById('total-portfolio-value');
        if (totalElement) {
            totalElement.textContent = CurrencyHelper.formatAmount(totalValue);
        }

        // Update individual balances
        if (this.portfolio) {
            this.portfolio.forEach(wallet => {
                const balanceElement = document.getElementById(`balance-${wallet.currency.toLowerCase()}`);
                if (balanceElement) {
                    balanceElement.textContent = parseFloat(wallet.balance || 0).toFixed(8);
                }

                const valueElement = document.getElementById(`value-${wallet.currency.toLowerCase()}`);
                if (valueElement && wallet.usd_value) {
                    valueElement.textContent = CurrencyHelper.formatAmount(wallet.usd_value);
                }
            });
        }

        // Update fiat balances
        if (this.userData && this.userData.fiat_balances) {
            this.userData.fiat_balances.forEach(balance => {
                const fiatElement = document.getElementById(`fiat-balance-${balance.currency.toLowerCase()}`);
                if (fiatElement) {
                    fiatElement.textContent = CurrencyHelper.formatAmount(balance.balance, balance.currency);
                }
            });
        }
    }

    static destroy() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
            this.updateInterval = null;
        }
        this.listeners = [];
        this.instance = null;
    }
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = BalanceManager;
}