/**
 * Frontend Configuration
 * API endpoints and configuration for the frontend
 */

// API Configuration
const API_CONFIG = {
    BASE_URL: 'https://kako.exchange/api',
    ENDPOINTS: {
        // Authentication
        LOGIN: '/auth/login',
        REGISTER: '/auth/register',
        LOGOUT: '/auth/logout',
        SEND_OTP: '/auth/send-otp',
        VERIFY_OTP: '/auth/verify-otp',
        RESET_PASSWORD: '/auth/reset-password',
        
        // User Management
        USER_PROFILE: '/user/profile',
        UPDATE_PROFILE: '/user/update-profile',
        USER_KYC: '/user/kyc',
        
        // Wallet Management
        USER_WALLETS: '/user/wallets',
        CREATE_WALLET: '/wallet/create',
        SEND_CRYPTO: '/wallet/send',
        RECEIVE_CRYPTO: '/wallet/receive',
        
        // Trading
        TRADING_PAIRS: '/trading/pairs',
        MARKET_PRICES: '/trading/prices',
        BUY_CRYPTO: '/trading/buy',
        SELL_CRYPTO: '/trading/sell',
        USER_ORDERS: '/trading/orders',
        
        // Transactions
        USER_TRANSACTIONS: '/user/transactions',
        TRANSACTION_HISTORY: '/user/transactions',
        TRANSACTION_DETAILS: '/transaction/details',
        
        // Deposit/Withdrawal
        CREATE_DEPOSIT: '/deposit/create',
        DEPOSIT_METHODS: '/deposit/methods',
        CREATE_WITHDRAWAL: '/withdrawal/create',
        WITHDRAW: '/withdrawal/create',
        VALIDATE_ADDRESS: '/withdrawal/validate-address',
        BITMUK_WITHDRAWAL: '/withdrawal/bitmuk',
        BITMUK_WITHDRAWAL_STATUS: '/withdrawal/bitmuk/status',
        CRYPTO_SEND: '/crypto/send',
        WALLET_GENERATE: '/wallet/generate',
        WALLET_LIST: '/wallet/list',
        BLOCKCHAIN_MONITOR: '/blockchain/monitor',
        BLOCKCHAIN_SEND: '/blockchain/send',
        
        // Market Data
        MARKET_OVERVIEW: '/market/overview',
        COIN_DETAILS: '/market/coin-details',
        CRYPTO_PRICES: '/crypto/prices',
        
        // Referral
        REFERRAL_INFO: '/referral/info',
        COMMISSION_HISTORY: '/referral/commission-history',
        
        // System
        SYSTEM_SETTINGS: '/system/settings',
        CURRENCIES: '/system/currencies',
        
        // Monetization
        MONETIZATION_TRADING_FEES: '/monetization/trading-fees',
        MONETIZATION_WITHDRAWAL_FEES: '/monetization/withdrawal-fees',
        MONETIZATION_CONVERSION_FEES: '/monetization/conversion-fees',
        MONETIZATION_REFERRAL_COMMISSION: '/monetization/referral-commission',
        
        // Admin
        ADMIN_DASHBOARD: '/admin/dashboard',
        ADMIN_REVENUE_ANALYTICS: '/admin/revenue-analytics',
        ADMIN_USER_TIERS: '/admin/user-tiers',
        ADMIN_TOP_REVENUE_USERS: '/admin/top-revenue-users',
        ADMIN_UPDATE_FEES: '/admin/update-fees'
    }
};

// Local Storage Keys
const STORAGE_KEYS = {
    AUTH_TOKEN: 'crypto_exchange_token',
    USER_DATA: 'crypto_exchange_user',
    USER_COUNTRY: 'crypto_exchange_country',
    LAST_CLICKED_PAIR: 'crypto_exchange_last_pair',
    PREFERENCES: 'crypto_exchange_preferences'
};

// Currency Configuration
const CURRENCY_CONFIG = {
    UG: { code: 'UGX', symbol: 'USh', flag: '🇺🇬', rate: 3700 },
    KE: { code: 'KES', symbol: 'KSh', flag: '🇰🇪', rate: 150 },
    TZ: { code: 'TZS', symbol: 'TSh', flag: '🇹🇿', rate: 2500 },
    RW: { code: 'RWF', symbol: 'RF', flag: '🇷🇼', rate: 1200 },
    CD: { code: 'CDF', symbol: 'FC', flag: '🇨🇩', rate: 2500 },
    NG: { code: 'NGN', symbol: '₦', flag: '🇳🇬', rate: 800 },
    US: { code: 'USD', symbol: '$', flag: '🇺🇸', rate: 1 }
};

// Network Configuration
const NETWORK_CONFIG = {
    ERC20: { name: 'Ethereum', symbol: 'ERC20', icon: 'public', fee: 2.50 },
    TRC20: { name: 'Tron', symbol: 'TRC20', icon: 'account_balance', fee: 1.00 },
    BEP20: { name: 'BSC', symbol: 'BEP20', icon: 'hub', fee: 0.50 },
    MATIC: { name: 'Polygon', symbol: 'MATIC', icon: 'hexagon', fee: 0.20 },
    BASE: { name: 'Base', symbol: 'BASE', icon: 'layers', fee: 0.30 },
    SOL: { name: 'Solana', symbol: 'SOL', icon: 'bolt', fee: 0.01 }
};

// API Helper Functions
class APIHelper {
    static async request(endpoint, options = {}) {
        const url = API_CONFIG.BASE_URL + endpoint;
        const token = localStorage.getItem(STORAGE_KEYS.AUTH_TOKEN);
        
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json',
                ...(token && { 'Authorization': `Bearer ${token}` })
            }
        };
        
        const config = { ...defaultOptions, ...options };
        
        try {
            const response = await fetch(url, config);
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Request failed');
            }
            
            return data;
        } catch (error) {
            console.error('API Request Error:', error);
            throw error;
        }
    }
    
    static async get(endpoint) {
        return this.request(endpoint, { method: 'GET' });
    }
    
    static async post(endpoint, data) {
        return this.request(endpoint, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    }
    
    static async put(endpoint, data) {
        return this.request(endpoint, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    }
    
    static async delete(endpoint) {
        return this.request(endpoint, { method: 'DELETE' });
    }
}

// Authentication Helper
class AuthHelper {
    static isLoggedIn() {
        return !!localStorage.getItem(STORAGE_KEYS.AUTH_TOKEN);
    }
    
    static getToken() {
        return localStorage.getItem(STORAGE_KEYS.AUTH_TOKEN);
    }
    
    static getUser() {
        const userData = localStorage.getItem(STORAGE_KEYS.USER_DATA);
        return userData ? JSON.parse(userData) : null;
    }
    
    static setAuth(token, user) {
        localStorage.setItem(STORAGE_KEYS.AUTH_TOKEN, token);
        localStorage.setItem(STORAGE_KEYS.USER_DATA, JSON.stringify(user));
    }
    
    static clearAuth() {
        localStorage.removeItem(STORAGE_KEYS.AUTH_TOKEN);
        localStorage.removeItem(STORAGE_KEYS.USER_DATA);
    }
    
    static async login(email, password) {
        try {
            const response = await APIHelper.post(API_CONFIG.ENDPOINTS.LOGIN, {
                email,
                password
            });
            
            if (response.token) {
                this.setAuth(response.token, response.user);
                return response;
            }
            
            return response;
        } catch (error) {
            throw error;
        }
    }
    
    static async register(userData) {
        try {
            const response = await APIHelper.post(API_CONFIG.ENDPOINTS.REGISTER, userData);
            return response;
        } catch (error) {
            throw error;
        }
    }
    
    static async logout() {
        try {
            await APIHelper.post(API_CONFIG.ENDPOINTS.LOGOUT);
        } catch (error) {
            console.error('Logout error:', error);
        } finally {
            this.clearAuth();
            window.location.href = '/login.html';
        }
    }
}

// Currency Helper
class CurrencyHelper {
    static getUserCountry() {
        return localStorage.getItem(STORAGE_KEYS.USER_COUNTRY) || 'UG';
    }
    
    static setUserCountry(country) {
        localStorage.setItem(STORAGE_KEYS.USER_COUNTRY, country);
    }
    
    static getCurrencyConfig(country = null) {
        const userCountry = country || this.getUserCountry();
        return CURRENCY_CONFIG[userCountry] || CURRENCY_CONFIG.UG;
    }
    
    static formatAmount(amount, currency = null) {
        const config = this.getCurrencyConfig();
        const currencyCode = currency || config.code;
        const symbol = config.symbol;
        
        // Format with max 3 decimal places for fiat currencies
        const formattedAmount = parseFloat(amount).toLocaleString(undefined, {
            minimumFractionDigits: 0,
            maximumFractionDigits: 3
        });
        
        return `${symbol} ${formattedAmount}`;
    }
    
    static convertToLocal(usdAmount, currency = null) {
        const config = this.getCurrencyConfig();
        const rate = config.rate;
        return (usdAmount * rate).toFixed(2);
    }
    
    static convertFromLocal(localAmount, currency = null) {
        const config = this.getCurrencyConfig();
        const rate = config.rate;
        return (localAmount / rate).toFixed(8);
    }
}

// Network Helper
class NetworkHelper {
    static getNetworkConfig(symbol) {
        return NETWORK_CONFIG[symbol] || NETWORK_CONFIG.ERC20;
    }
    
    static validateAddress(address, network) {
        const config = this.getNetworkConfig(network);
        
        switch (network) {
            case 'ERC20':
            case 'BEP20':
            case 'BASE':
                return /^0x[a-fA-F0-9]{40}$/.test(address);
            case 'TRC20':
                return /^T[a-zA-Z0-9]{33}$/.test(address);
            case 'SOL':
                return /^[1-9A-HJ-NP-Za-km-z]{32,44}$/.test(address);
            default:
                return address.length > 10;
        }
    }
    
    static detectNetwork(address) {
        if (/^0x[a-fA-F0-9]{40}$/.test(address)) {
            return 'ERC20'; // Default to ERC20 for 0x addresses
        } else if (/^T[a-zA-Z0-9]{33}$/.test(address)) {
            return 'TRC20';
        } else if (/^[1-9A-HJ-NP-Za-km-z]{32,44}$/.test(address)) {
            return 'SOL';
        }
        return null;
    }
}

// Utility Functions
class Utils {
    static formatDate(date) {
        return new Date(date).toLocaleDateString();
    }
    
    static formatDateTime(date) {
        return new Date(date).toLocaleString();
    }
    
    static formatNumber(number, decimals = 2) {
        return parseFloat(number).toFixed(decimals);
    }
    
    static generateId() {
        return Math.random().toString(36).substr(2, 9);
    }
    
    static debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    static throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        API_CONFIG,
        STORAGE_KEYS,
        CURRENCY_CONFIG,
        NETWORK_CONFIG,
        APIHelper,
        AuthHelper,
        CurrencyHelper,
        NetworkHelper,
        Utils
    };
}
