/**
 * Country Detector
 * Automatic country detection and currency selection
 */

class CountryDetector {
    static instance = null;
    static detectedCountry = null;
    static detectedCurrency = null;
    static listeners = [];

    static initialize() {
        if (this.instance) {
            return Promise.resolve(true);
        }

        this.instance = new CountryDetector();
        return this.detectCountry();
    }

    static async detectCountry() {
        try {
            console.log('🌍 Detecting country...');
            
            // Try multiple detection methods
            let country = null;
            
            // Method 1: Check localStorage first
            const savedCountry = localStorage.getItem('detected_country');
            if (savedCountry && CURRENCY_CONFIG[savedCountry]) {
                country = savedCountry;
                console.log('✅ Country from localStorage:', country);
            }
            
            // Method 2: Use IP geolocation API
            if (!country) {
                country = await this.detectByIP();
            }
            
            // Method 3: Use browser timezone
            if (!country) {
                country = this.detectByTimezone();
            }
            
            // Method 4: Use browser language
            if (!country) {
                country = this.detectByLanguage();
            }
            
            // Fallback to Uganda
            if (!country || !CURRENCY_CONFIG[country]) {
                country = 'UG';
                console.log('⚠️ Using fallback country: Uganda');
            }
            
            this.detectedCountry = country;
            this.detectedCurrency = CURRENCY_CONFIG[country];
            
            // Save to localStorage
            localStorage.setItem('detected_country', country);
            localStorage.setItem('user_country', country);
            
            console.log('✅ Country detected:', country, this.detectedCurrency);
            
            this.notifyListeners();
            return true;
            
        } catch (error) {
            console.error('❌ Country detection failed:', error);
            // Use fallback
            this.detectedCountry = 'UG';
            this.detectedCurrency = CURRENCY_CONFIG.UG;
            this.notifyListeners();
            return false;
        }
    }

    static async detectByIP() {
        try {
            // Use a free IP geolocation service
            const response = await fetch('https://ipapi.co/json/');
            if (response.ok) {
                const data = await response.json();
                const countryCode = data.country_code;
                
                if (CURRENCY_CONFIG[countryCode]) {
                    console.log('✅ Country detected by IP:', countryCode);
                    return countryCode;
                }
            }
        } catch (error) {
            console.log('IP detection failed:', error.message);
        }
        
        return null;
    }

    static detectByTimezone() {
        try {
            const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
            console.log('🕐 Timezone detected:', timezone);
            
            // Map timezones to countries
            const timezoneMap = {
                'Africa/Kampala': 'UG',
                'Africa/Nairobi': 'KE',
                'Africa/Dar_es_Salaam': 'TZ',
                'Africa/Kigali': 'RW',
                'Africa/Kinshasa': 'CD',
                'Africa/Lagos': 'NG',
                'America/New_York': 'US',
                'Europe/London': 'GB',
                'Europe/Paris': 'EU'
            };
            
            const country = timezoneMap[timezone];
            if (country && CURRENCY_CONFIG[country]) {
                console.log('✅ Country detected by timezone:', country);
                return country;
            }
        } catch (error) {
            console.log('Timezone detection failed:', error.message);
        }
        
        return null;
    }

    static detectByLanguage() {
        try {
            const language = navigator.language || navigator.userLanguage;
            console.log('🗣️ Language detected:', language);
            
            // Map languages to countries
            const languageMap = {
                'en-UG': 'UG',
                'en-KE': 'KE',
                'en-TZ': 'TZ',
                'en-RW': 'RW',
                'fr-CD': 'CD',
                'en-NG': 'NG',
                'en-US': 'US',
                'en-GB': 'GB'
            };
            
            const country = languageMap[language];
            if (country && CURRENCY_CONFIG[country]) {
                console.log('✅ Country detected by language:', country);
                return country;
            }
        } catch (error) {
            console.log('Language detection failed:', error.message);
        }
        
        return null;
    }

    static addListener(callback) {
        this.listeners.push(callback);
    }

    static removeListener(callback) {
        const index = this.listeners.indexOf(callback);
        if (index > -1) {
            this.listeners.splice(index, 1);
        }
    }

    static notifyListeners() {
        this.listeners.forEach(callback => {
            try {
                callback(this.detectedCountry, this.detectedCurrency);
            } catch (error) {
                console.error('Country listener error:', error);
            }
        });
    }

    static getDetectedCountry() {
        return this.detectedCountry || 'UG';
    }

    static getDetectedCurrency() {
        return this.detectedCurrency || CURRENCY_CONFIG.UG;
    }

    static setCountry(country) {
        if (CURRENCY_CONFIG[country]) {
            this.detectedCountry = country;
            this.detectedCurrency = CURRENCY_CONFIG[country];
            
            // Save to localStorage
            localStorage.setItem('detected_country', country);
            localStorage.setItem('user_country', country);
            
            this.notifyListeners();
            console.log('✅ Country manually set:', country);
        }
    }

    static updateCurrencyDisplay() {
        const currency = this.getDetectedCurrency();
        
        // Update currency selector
        const currencySelect = document.getElementById('currency-selector');
        if (currencySelect) {
            currencySelect.value = currency.code;
        }
        
        // Update currency display elements
        const currencyElements = document.querySelectorAll('.currency-display');
        currencyElements.forEach(element => {
            element.textContent = currency.symbol;
        });
        
        // Update flag display
        const flagElements = document.querySelectorAll('.country-flag');
        flagElements.forEach(element => {
            element.textContent = currency.flag;
        });
        
        // Update currency name
        const nameElements = document.querySelectorAll('.currency-name');
        nameElements.forEach(element => {
            element.textContent = currency.code;
        });
    }

    static createCountrySelector() {
        const selector = document.createElement('select');
        selector.id = 'country-selector';
        selector.className = 'country-selector';
        
        Object.keys(CURRENCY_CONFIG).forEach(countryCode => {
            const config = CURRENCY_CONFIG[countryCode];
            const option = document.createElement('option');
            option.value = countryCode;
            option.textContent = `${config.flag} ${config.code} - ${config.symbol}`;
            if (countryCode === this.getDetectedCountry()) {
                option.selected = true;
            }
            selector.appendChild(option);
        });
        
        selector.addEventListener('change', (e) => {
            this.setCountry(e.target.value);
            this.updateCurrencyDisplay();
        });
        
        return selector;
    }

    static destroy() {
        this.listeners = [];
        this.instance = null;
    }
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CountryDetector;
}