/**
 * Crypto Prices Manager
 * Real-time cryptocurrency price management
 */

class CryptoPricesManager {
    static instance = null;
    static prices = {};
    static lastUpdate = null;
    static updateInterval = null;
    static listeners = [];

    static initialize() {
        if (this.instance) {
            return Promise.resolve(true);
        }

        this.instance = new CryptoPricesManager();
        return this.loadPrices();
    }

    static async loadPrices() {
        try {
            console.log('🔄 Loading crypto prices...');
            
            // Load market prices
            const response = await APIHelper.get(API_CONFIG.ENDPOINTS.CRYPTO_PRICES);
            if (response.success) {
                this.prices = response.prices || {};
                this.lastUpdate = new Date();
                this.notifyListeners();
                console.log('✅ Crypto prices loaded:', this.prices);
            }

            // Start auto-refresh
            this.startAutoRefresh();
            
            return true;
        } catch (error) {
            console.error('❌ Failed to load crypto prices:', error);
            // Use fallback prices
            this.loadFallbackPrices();
            return false;
        }
    }

    static loadFallbackPrices() {
        // Fallback prices for development
        this.prices = {
            'BTC': { price: 45000, change_24h: 2.5, volume_24h: 25000000000 },
            'ETH': { price: 3000, change_24h: 1.8, volume_24h: 15000000000 },
            'USDT': { price: 1.00, change_24h: 0.1, volume_24h: 50000000000 },
            'USDC': { price: 1.00, change_24h: 0.05, volume_24h: 20000000000 },
            'BNB': { price: 300, change_24h: 3.2, volume_24h: 2000000000 },
            'ADA': { price: 0.45, change_24h: -1.2, volume_24h: 800000000 },
            'SOL': { price: 100, change_24h: 5.8, volume_24h: 3000000000 },
            'MATIC': { price: 0.85, change_24h: 2.1, volume_24h: 1200000000 }
        };
        this.lastUpdate = new Date();
        this.notifyListeners();
        console.log('✅ Fallback prices loaded');
    }

    static startAutoRefresh() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }

        // Refresh every 60 seconds
        this.updateInterval = setInterval(() => {
            this.refreshPrices();
        }, 60000);
    }

    static async refreshPrices() {
        try {
            console.log('🔄 Refreshing crypto prices...');
            
            const response = await APIHelper.get(API_CONFIG.ENDPOINTS.CRYPTO_PRICES);
            if (response.success) {
                this.prices = response.prices || {};
                this.lastUpdate = new Date();
                this.notifyListeners();
                console.log('✅ Prices refreshed');
            }
        } catch (error) {
            console.error('❌ Failed to refresh prices:', error);
        }
    }

    static addListener(callback) {
        this.listeners.push(callback);
    }

    static removeListener(callback) {
        const index = this.listeners.indexOf(callback);
        if (index > -1) {
            this.listeners.splice(index, 1);
        }
    }

    static notifyListeners() {
        this.listeners.forEach(callback => {
            try {
                callback(this.prices);
            } catch (error) {
                console.error('Price listener error:', error);
            }
        });
    }

    static getPrice(symbol) {
        return this.prices[symbol] || { price: 0, change_24h: 0, volume_24h: 0 };
    }

    static getAllPrices() {
        return this.prices;
    }

    static getPriceChange(symbol) {
        const price = this.getPrice(symbol);
        return price.change_24h || 0;
    }

    static getVolume(symbol) {
        const price = this.getPrice(symbol);
        return price.volume_24h || 0;
    }

    static formatPrice(price, decimals = 2) {
        if (price >= 1) {
            return parseFloat(price).toFixed(decimals);
        } else {
            return parseFloat(price).toFixed(8);
        }
    }

    static formatChange(change) {
        const sign = change >= 0 ? '+' : '';
        return `${sign}${parseFloat(change).toFixed(2)}%`;
    }

    static formatVolume(volume) {
        if (volume >= 1000000000) {
            return `$${(volume / 1000000000).toFixed(2)}B`;
        } else if (volume >= 1000000) {
            return `$${(volume / 1000000).toFixed(2)}M`;
        } else if (volume >= 1000) {
            return `$${(volume / 1000).toFixed(2)}K`;
        } else {
            return `$${volume.toFixed(2)}`;
        }
    }

    static updatePriceElements() {
        Object.keys(this.prices).forEach(symbol => {
            const price = this.getPrice(symbol);
            
            // Update price element
            const priceElement = document.getElementById(`price-${symbol.toLowerCase()}`);
            if (priceElement) {
                priceElement.textContent = this.formatPrice(price.price);
            }

            // Update change element
            const changeElement = document.getElementById(`change-${symbol.toLowerCase()}`);
            if (changeElement) {
                changeElement.textContent = this.formatChange(price.change_24h);
                changeElement.className = `price-change ${price.change_24h >= 0 ? 'positive' : 'negative'}`;
            }

            // Update volume element
            const volumeElement = document.getElementById(`volume-${symbol.toLowerCase()}`);
            if (volumeElement) {
                volumeElement.textContent = this.formatVolume(price.volume_24h);
            }
        });
    }

    static destroy() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
            this.updateInterval = null;
        }
        this.listeners = [];
        this.instance = null;
    }
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CryptoPricesManager;
}