/**
 * Payment Auto-Checker
 * 
 * This script automatically triggers payment status checks at strategic moments
 * without requiring cron jobs or manual intervention.
 */

class PaymentAutoChecker {
    constructor() {
        this.lastCheckTime = 0;
        this.checkInterval = 5 * 60 * 1000; // 5 minutes in milliseconds
        this.init();
    }
    
    /**
     * Initialize the auto-checker
     */
    init() {
        // Check on page load
        this.checkIfNeeded();
        
        // Check when user becomes active (focus, click, scroll)
        this.setupUserActivityTriggers();
        
        // Check when user navigates between pages
        this.setupNavigationTriggers();
        
        // Check periodically (but only when page is active)
        this.setupPeriodicChecks();
        
        console.log('PaymentAutoChecker initialized');
    }
    
    /**
     * Check if payment check is needed and trigger if so
     */
    async checkIfNeeded() {
        const now = Date.now();
        
        // Only check if enough time has passed
        if (now - this.lastCheckTime < this.checkInterval) {
            console.log('Payment check skipped - too soon since last check');
            return;
        }
        
        try {
            console.log('Triggering automatic payment check...');
            
            const response = await fetch('/api/payment/auto-check', {
                method: 'GET',
                headers: {
                    'Authorization': 'Bearer test-token',
                    'Content-Type': 'application/json'
                }
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.lastCheckTime = now;
                console.log('Auto payment check completed:', data.data);
                
                // If payments were completed, refresh balance if on relevant pages
                if (data.data.completed > 0) {
                    this.refreshBalanceIfNeeded();
                }
            } else {
                console.log('Auto payment check skipped:', data.data.message);
            }
            
        } catch (error) {
            console.error('Auto payment check error:', error);
        }
    }
    
    /**
     * Setup triggers based on user activity
     */
    setupUserActivityTriggers() {
        let activityTimeout;
        
        const triggerCheck = () => {
            clearTimeout(activityTimeout);
            activityTimeout = setTimeout(() => {
                this.checkIfNeeded();
            }, 10000); // Check 10 seconds after user becomes active
        };
        
        // Trigger on user activity
        ['focus', 'click', 'scroll', 'keydown', 'mousemove'].forEach(event => {
            document.addEventListener(event, triggerCheck, { passive: true });
        });
        
        // Trigger when page becomes visible
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                triggerCheck();
            }
        });
    }
    
    /**
     * Setup triggers for page navigation
     */
    setupNavigationTriggers() {
        // Check when navigating to dashboard or crypto wallet
        const relevantPages = [
            'user_dashboard.html',
            'crypto_wallet.html',
            'transaction_history.html',
            'deposit_success.html',
            'payment_status.html'
        ];
        
        const currentPage = window.location.pathname.split('/').pop();
        
        if (relevantPages.includes(currentPage)) {
            // Check immediately when landing on relevant pages
            setTimeout(() => {
                this.checkIfNeeded();
            }, 2000); // Wait 2 seconds for page to load
        }
    }
    
    /**
     * Setup periodic checks (only when page is active)
     */
    setupPeriodicChecks() {
        setInterval(() => {
            // Only check if page is visible and active
            if (!document.hidden && document.hasFocus()) {
                this.checkIfNeeded();
            }
        }, 10 * 60 * 1000); // Check every 10 minutes when page is active
    }
    
    /**
     * Refresh balance if user is on relevant pages
     */
    refreshBalanceIfNeeded() {
        const currentPage = window.location.pathname.split('/').pop();
        
        if (currentPage === 'user_dashboard.html') {
            // Refresh dashboard balance
            if (typeof BalanceManager !== 'undefined') {
                BalanceManager.initialize();
            }
        } else if (currentPage === 'crypto_wallet.html') {
            // Refresh crypto wallet balance
            if (typeof BalanceManager !== 'undefined') {
                BalanceManager.initialize();
            }
        }
        
        console.log('Balance refreshed due to completed payments');
    }
    
    /**
     * Force a payment check (can be called manually)
     */
    async forceCheck() {
        console.log('Force checking payments...');
        this.lastCheckTime = 0; // Reset last check time
        await this.checkIfNeeded();
    }
}

// Initialize the auto-checker when the page loads
document.addEventListener('DOMContentLoaded', () => {
    window.paymentAutoChecker = new PaymentAutoChecker();
});

// Also initialize immediately if DOM is already loaded
if (document.readyState === 'loading') {
    // DOM is still loading, wait for DOMContentLoaded
} else {
    // DOM is already loaded
    window.paymentAutoChecker = new PaymentAutoChecker();
}

