// Multi-Currency Withdraw Functionality
// This script adds support for withdrawing from any currency with a balance

// Wait for the page to load and BalanceManager to be available
document.addEventListener('DOMContentLoaded', function() {
    // Wait for BalanceManager to initialize
    setTimeout(initializeMultiCurrency, 1000);
});

function initializeMultiCurrency() {
    console.log('Initializing multi-currency support...');
    
    // Add currency selector event listener
    const currencySelector = document.getElementById('currency_selector');
    if (currencySelector) {
        currencySelector.addEventListener('change', handleCurrencyChange);
    }
    
    // Populate currency selector when balances are loaded
    if (typeof BalanceManager !== 'undefined' && BalanceManager.wallets) {
        populateCurrencySelector();
    }
}

function handleCurrencyChange() {
    const selectedCurrency = document.getElementById('currency_selector').value;
    if (!selectedCurrency) {
        // Hide the display if no currency selected
        const displayElement = document.getElementById('selected-currency-display');
        if (displayElement) {
            displayElement.classList.add('hidden');
        }
        return;
    }
    
    console.log(`Currency changed to: ${selectedCurrency}`);
    
    // Show the selected currency display
    const displayElement = document.getElementById('selected-currency-display');
    const textElement = document.getElementById('selected-currency-text');
    if (displayElement && textElement) {
        displayElement.classList.remove('hidden');
        textElement.textContent = selectedCurrency;
    }
    
    // Update currency display
    const currencyElement = document.querySelector('.selected-currency');
    if (currencyElement) {
        currencyElement.textContent = selectedCurrency;
    }
    
    // Update all currency text elements
    const currencyTextElements = document.querySelectorAll('.selected-currency-text');
    currencyTextElements.forEach(el => {
        el.textContent = selectedCurrency;
    });
    
    // Update available balance display
    const balance = BalanceManager.getWalletBalance(selectedCurrency);
    const balanceElement = document.querySelector('.withdraw-balance');
    if (balanceElement) {
        balanceElement.textContent = `${balance} ${selectedCurrency}`;
        balanceElement.dataset.currency = selectedCurrency;
    }
    
    // Update currency logo
    updateCurrencyLogo(selectedCurrency);
    
    // Update transaction summary with new currency
    if (typeof updateTransactionSummary === 'function') {
        updateTransactionSummary();
    }
    
    console.log(`Updated to ${selectedCurrency}, Balance: ${balance}`);
    
    // Auto-select Bitmuk for UGX withdrawals
    if (selectedCurrency === 'UGX') {
        console.log('UGX selected - auto-selecting Bitmuk provider');
        
        // Switch to mobile money tab
        const mobileTab = document.getElementById('tab-mobile');
        const bankTab = document.getElementById('tab-bank');
        const mobilePanel = document.getElementById('panel-mobile');
        const bankPanel = document.getElementById('panel-bank');
        
        if (mobileTab && bankTab && mobilePanel && bankPanel) {
            // Remove active class from bank tab
            bankTab.classList.remove('bg-accent-dark', 'text-white');
            bankTab.classList.add('bg-accent/10', 'text-accent-dark');
            bankPanel.classList.add('hidden');
            
            // Add active class to mobile tab
            mobileTab.classList.remove('bg-accent/10', 'text-accent-dark');
            mobileTab.classList.add('bg-accent-dark', 'text-white');
            mobilePanel.classList.remove('hidden');
            
            console.log('Switched to mobile money tab');
        }
        
        // Auto-select Bitmuk provider
        const providerSelect = document.getElementById('mobile_money_provider');
        if (providerSelect) {
            // Add Bitmuk option if not exists
            let bitmukOption = providerSelect.querySelector('option[value="Bitmuk"]');
            if (!bitmukOption) {
                bitmukOption = document.createElement('option');
                bitmukOption.value = 'Bitmuk';
                bitmukOption.textContent = 'Bitmuk';
                providerSelect.appendChild(bitmukOption);
            }
            
            // Select Bitmuk
            providerSelect.value = 'Bitmuk';
            console.log('Auto-selected Bitmuk provider for UGX');
        }
    }
}

function updateCurrencyLogo(currency) {
    const currencyLogos = {
        'USDC': 'https://lh3.googleusercontent.com/aida-public/AB6AXuCYO8uom00K9q7d4p5WksZgIQV-IvRgG3l3_JF00x0AbTojZCECe6F_eQ3xNGk_6XSizAZj-qInn7fkPhu7VsPSoFPJ3eMBFiY8K5VvGy-Q3Ldu8UX-pwVhKHTuZeknhaWd_EAy5LYnX8Mr13VbNst1p2eLhesOESnuD7wWsWo20_7YDxj4uAxXnbBm2B4ZxEBGJ2udoL_KfvkZHnBb2CX6rdmAaLy28orPGnBc2vkUMO8XjGjYutfFdC7dYAiaQ_mp9oSqh3lz53A',
        'USDT': 'https://lh3.googleusercontent.com/aida-public/AB6AXuCYO8uom00K9q7d4p5WksZgIQV-IvRgG3l3_JF00x0AbTojZCECe6F_eQ3xNGk_6XSizAZj-qInn7fkPhu7VsPSoFPJ3eMBFiY8K5VvGy-Q3Ldu8UX-pwVhKHTuZeknhaWd_EAy5LYnX8Mr13VbNst1p2eLhesOESnuD7wWsWo20_7YDxj4uAxXnbBm2B4ZxEBGJ2udoL_KfvkZHnBb2CX6rdmAaLy28orPGnBc2vkUMO8XjGjYutfFdC7dYAiaQ_mp9oSqh3lz53A',
        'UGX': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'KES': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'NGN': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'TZS': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'RWF': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'CDF': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360',
        'GHS': 'https://lh3.googleusercontent.com/aida-public/AB6AXuBAtmfKFILAtUwo30FisKOSqaYgZQW_ZAFlWIZtw3JZYrd7xdxUk_McmEviv2DaUQc5W2Our0rpWc7vLqH8f67oCmtsDvLd5DG4HA4RznWp38CqOgrOWsEisTfewDasUcPU9iHL1LlFuF1zywfK1I0kzVzRIQJV2xTaYc6FVX4U16-VKYCFHKTIqzakXoxzzJLg4ygFRT8vaF-FiDfmNDXR7ZW8l5iTGsVZexmu6_iku5kQ3v9e6hm9faoXiOL_vKvXbphH6YmR360'
    };
    
    const logoElement = document.querySelector('.currency-logo');
    if (logoElement && currencyLogos[currency]) {
        logoElement.src = currencyLogos[currency];
    }
}

function populateCurrencySelector() {
    const currencySelector = document.getElementById('currency_selector');
    if (!currencySelector) {
        console.log('Currency selector not found');
        return;
    }
    
    // Clear existing options
    currencySelector.innerHTML = '<option value="">Select currency to withdraw</option>';
    
    // Get user's wallets with balances
    const wallets = BalanceManager.wallets || [];
    const currenciesWithBalance = wallets.filter(wallet => wallet.balance > 0);
    
    console.log('Wallets with balance:', currenciesWithBalance);
    
    if (currenciesWithBalance.length === 0) {
        currencySelector.innerHTML = '<option value="">No currencies with balance</option>';
        return;
    }
    
    // Add options for currencies with balance
    currenciesWithBalance.forEach(wallet => {
        const option = document.createElement('option');
        option.value = wallet.currency;
        option.textContent = `${wallet.currency} (${wallet.balance} available)`;
        currencySelector.appendChild(option);
    });
    
    // Select the first currency by default, but make it clear to the user
    // This ensures the form is functional while still requiring user awareness
    if (currenciesWithBalance.length > 0) {
        const firstCurrency = currenciesWithBalance[0].currency;
        currencySelector.value = firstCurrency;
        
        // Add a visual indicator that this is auto-selected
        const selectedOption = currencySelector.querySelector(`option[value="${firstCurrency}"]`);
        if (selectedOption) {
            selectedOption.textContent = `${firstCurrency} (${currenciesWithBalance.find(w => w.currency === firstCurrency).balance} available) - SELECTED`;
        }
        
        handleCurrencyChange();
        console.log(`Auto-selected first currency: ${firstCurrency} - User can change this`);
    }
    
    console.log(`Populated currency selector with ${currenciesWithBalance.length} currencies`);
}

// Override the existing updateWithdrawBalances function to include currency population
if (typeof updateWithdrawBalances === 'function') {
    const originalUpdateWithdrawBalances = updateWithdrawBalances;
    window.updateWithdrawBalances = function() {
        originalUpdateWithdrawBalances();
        populateCurrencySelector();
    };
}