<?php

require_once 'config/production_config.php';
require_once 'api/blockchain_manager.php';

class SystemWalletManager {
    private $blockchainManager;
    
    public function __construct() {
        require_once 'config/database.php';
        $db = new Database();
        $this->blockchainManager = new BlockchainManager($db);
    }
    
    /**
     * Get all system wallet addresses and balances
     */
    public function getSystemWallets() {
        $systemWallets = ProductionConfig::getSystemWallets();
        $wallets = [];
        
        foreach ($systemWallets as $key => $wallet) {
            $wallets[] = [
                'key' => $key,
                'currency' => $wallet['currency'],
                'network' => $wallet['network'],
                'address' => $wallet['address'],
                'is_real_blockchain' => $wallet['is_real_blockchain']
            ];
        }
        
        return $wallets;
    }
    
    /**
     * Get funding instructions for each wallet
     */
    public function getFundingInstructions() {
        $wallets = $this->getSystemWallets();
        $instructions = [];
        
        $recommendedAmounts = [
            'USDT' => 2000,
            'USDC' => 1000,
            'ETH' => 5,
            'BTC' => 0.1,
            'SOL' => 100
        ];
        
        foreach ($wallets as $wallet) {
            $currency = $wallet['currency'];
            $amount = $recommendedAmounts[$currency] ?? 1000;
            
            $instructions[] = [
                'currency' => $currency,
                'network' => $wallet['network'],
                'address' => $wallet['address'],
                'recommended_amount' => $amount,
                'funding_method' => $this->getFundingMethod($wallet['network']),
                'notes' => $this->getFundingNotes($wallet['network'])
            ];
        }
        
        return $instructions;
    }
    
    private function getFundingMethod($network) {
        switch ($network) {
            case 'ERC20':
                return 'Send via MetaMask, Coinbase, or any Ethereum wallet';
            case 'BEP20':
                return 'Send via MetaMask (BSC network), Trust Wallet, or Binance';
            case 'BTC':
                return 'Send via Bitcoin wallet (Electrum, Bitcoin Core, etc.)';
            case 'POLYGON':
                return 'Send via MetaMask (Polygon network) or Polygon Bridge';
            case 'TRC20':
                return 'Send via TronLink or any TRON wallet';
            case 'solana':
                return 'Send via Phantom, Solflare, or any Solana wallet';
            default:
                return 'Send via compatible wallet for ' . $network;
        }
    }
    
    private function getFundingNotes($network) {
        switch ($network) {
            case 'ERC20':
                return '⚠️ High gas fees on Ethereum mainnet. Consider using Polygon or BSC for lower fees.';
            case 'BEP20':
                return '✅ Lower fees than Ethereum. Fast transactions.';
            case 'BTC':
                return '⚠️ Bitcoin transactions are slower and have higher fees.';
            case 'POLYGON':
                return '✅ Very low fees. Fast transactions.';
            case 'TRC20':
                return '✅ Low fees. Fast transactions.';
            case 'solana':
                return '✅ Very low fees. Very fast transactions.';
            default:
                return 'Check network-specific requirements.';
        }
    }
    
    /**
     * Display system wallet information
     */
    public function displaySystemWallets() {
        echo "🔐 SYSTEM WALLET ADDRESSES FOR FUNDING\n";
        echo "=====================================\n\n";
        
        $instructions = $this->getFundingInstructions();
        
        foreach ($instructions as $instruction) {
            echo "💰 {$instruction['currency']} ({$instruction['network']})\n";
            echo "   Address: {$instruction['address']}\n";
            echo "   Recommended: {$instruction['recommended_amount']} {$instruction['currency']}\n";
            echo "   Method: {$instruction['funding_method']}\n";
            echo "   Notes: {$instruction['notes']}\n\n";
        }
        
        echo "📋 FUNDING CHECKLIST:\n";
        echo "=====================\n";
        echo "1. Copy the wallet addresses above\n";
        echo "2. Send crypto from your personal wallet to these addresses\n";
        echo "3. Wait for confirmations (varies by network)\n";
        echo "4. Check balances in admin dashboard\n";
        echo "5. Start accepting user trades\n\n";
        
        echo "⚠️  SECURITY REMINDERS:\n";
        echo "=======================\n";
        echo "• These are HOT WALLETS - only fund what you need for trading\n";
        echo "• Keep most of your crypto in COLD STORAGE (hardware wallets)\n";
        echo "• Monitor balances regularly\n";
        echo "• Set up alerts for low balances\n";
        echo "• Consider multi-signature wallets for large amounts\n\n";
    }
    
    /**
     * Check current balances via API
     */
    public function checkCurrentBalances() {
        echo "📊 CURRENT SYSTEM WALLET BALANCES\n";
        echo "=================================\n\n";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'http://localhost:8000/api/admin/system-wallets');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer admin-token'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if ($data['success'] && isset($data['data'])) {
                foreach ($data['data'] as $wallet) {
                    $balance = floatval($wallet['balance']);
                    $usdValue = floatval($wallet['usd_value']);
                    $status = $balance > 0 ? '✅ FUNDED' : '❌ EMPTY';
                    
                    echo "{$status} {$wallet['currency']} ({$wallet['network']})\n";
                    echo "   Balance: {$balance} {$wallet['currency']}\n";
                    echo "   USD Value: $" . number_format($usdValue, 2) . "\n";
                    echo "   Address: {$wallet['address']}\n\n";
                }
            }
        } else {
            echo "❌ Failed to fetch wallet balances. Make sure the server is running.\n";
        }
    }
}

// Run the wallet manager
$manager = new SystemWalletManager();

echo "🚀 YELLOW CARD STYLE SYSTEM WALLET MANAGEMENT\n";
echo "=============================================\n\n";

$manager->displaySystemWallets();
$manager->checkCurrentBalances();

echo "🎯 NEXT STEPS:\n";
echo "==============\n";
echo "1. Fund the wallets above with your crypto\n";
echo "2. Monitor balances in: http://localhost:8000/admin#system-wallets\n";
echo "3. Set up automated alerts for low balances\n";
echo "4. Start accepting user trades once funded\n\n";

echo "💡 TIP: Start with smaller amounts for testing, then scale up!\n";
